/* Copyright (C) 2019 Moxa Inc. All rights reserved. */
/* SPDX-License-Identifier: Apache-2.0 */
/*!
    \file mx_node_node_def.h
    \copyright Moxa Inc.
    \brief <b> \a MX_NODE_NODE \a 's correlative types.</b>
    \date 2019-06-11
    \author Rich Liao
    \version V1.0
*/

#ifndef __MX_NODE_NODE_DEFINE__
#define __MX_NODE_NODE_DEFINE__

#include <stdint.h>
#include <stdbool.h>

/*!
    \addtogroup Library
    @{
*/

/*!
    \addtogroup MX_Node_SDK
    @{
*/

/*!
    \addtogroup Node_Type
    @{
*/

#define MAX_NODE_NAME_LEN 30
#define MAX_NODE_DESC_LEN 256

/*!
    \brief Type of OPC UA node.
*/
typedef enum _MX_NODE_NODE_TYPE
{
    MX_NODE_NODE_TYPE_UNKNOWN = -1,
    MX_NODE_NODE_TYPE_FOLDER = 0,
    MX_NODE_NODE_TYPE_OBJECT = 1,
    MX_NODE_NODE_TYPE_VARIABLE = 2,
    MX_NODE_NODE_TYPE_PROPERTY = 3
} MX_NODE_NODE_TYPE;

/*!
    \brief Folder node's attributes, it is an empty sturcture.
*/
typedef struct _MX_NODE_FOLDER
{
} MX_NODE_FOLDER;

/*!
    \brief Object node's attributes, it is an empty sturcture.
*/
typedef struct _MX_NODE_OBJECT
{
} MX_NODE_OBJECT;

/*!
    \brief Variable node's data type.
*/
typedef enum _MX_NODE_VALUE_TYPE
{
    MX_NODE_VALUE_TYPE_UNKNOWN = -1,
    MX_NODE_VALUE_TYPE_BOOLEAN = 0,
    MX_NODE_VALUE_TYPE_SBYTE = 1,
    MX_NODE_VALUE_TYPE_INT16 = 2,
    MX_NODE_VALUE_TYPE_INT32 = 3,
    MX_NODE_VALUE_TYPE_INT64 = 4,
    MX_NODE_VALUE_TYPE_BYTE = 5,
    MX_NODE_VALUE_TYPE_UINT16 = 6,
    MX_NODE_VALUE_TYPE_UINT32 = 7,
    MX_NODE_VALUE_TYPE_UINT64 = 8,
    MX_NODE_VALUE_TYPE_FLOAT = 9,
    MX_NODE_VALUE_TYPE_DOUBLE = 10,
    MX_NODE_VALUE_TYPE_STRING = 11
} MX_NODE_VALUE_TYPE;

/*!
    \brief Variable node's access right.
 */
typedef enum _MX_NODE_ACCESS_RIGHT
{
    MX_NODE_ACCESS_RIGHT_NONE = 0,
    MX_NODE_ACCESS_RIGHT_READONLY = 1,
    MX_NODE_ACCESS_RIGHT_WRITEONLY = 2,
    MX_NODE_ACCESS_RIGHT_READWRITE = 3
} MX_NODE_ACCESS_RIGHT;

/*!
    \brief A variant structure is used to store different type's data.
    \note value's other is used to pointer to other type's data.
 */
typedef struct _MX_NODE_VARIANT
{
    MX_NODE_VALUE_TYPE type;
    union
    {
        bool b;
        int8_t i8;
        int16_t i16;
        int32_t i32;
        int64_t i64;
        uint8_t u8;
        uint16_t u16;
        uint32_t u32;
        uint64_t u64;
        float f;
        double d;
        char* s;
        void* other;
    } value;
} MX_NODE_VARIANT;

/*!
    \brief Variable node's attributes.
*/
typedef struct _MX_NODE_VARIABLE
{
    MX_NODE_VARIANT value;
    MX_NODE_ACCESS_RIGHT access_right;
} MX_NODE_VARIABLE;

/*!
    \brief Property node's attributes.
*/
typedef struct _MX_NODE_PROPERTY
{
    MX_NODE_VARIANT value;
    MX_NODE_ACCESS_RIGHT access_right;
} MX_NODE_PROPERTY;

/*!
    \brief A node's sturcture contain name, description, node type and attribute.
*/
typedef struct _MX_NODE_NODE
{
    char node_name[MAX_NODE_NAME_LEN];
    char description[MAX_NODE_DESC_LEN];
    MX_NODE_NODE_TYPE node_type;
    union
    {
        MX_NODE_FOLDER folder;
        MX_NODE_OBJECT object;
        MX_NODE_VARIABLE variable;
        MX_NODE_PROPERTY property;
    };
} MX_NODE_NODE;

/*!
    \brief A result enumeration represents success or not.
*/
typedef enum _MX_NODE_RESULT
{
    MX_NODE_RESULT_GOOD,
    MX_NODE_RESULT_BAD
} MX_NODE_RESULT;

//! @}

//! @}

//! @}

#endif