import ctypes
from datetime import datetime
import os
import serial

BIN_TYPE = 2
SLOT_CH_NUM = 16


class ioThinxError(Exception):
    def __init__(self, value):
        self.value = value

    def __str__(self):
        error = ["ok",
                 "init fail",
                 "wrong slot settings",
                 "wrong argument",
                 "wrong config",
                 "wrong range",
                 "wrong communication"
                 ]
        return repr(error[self.value])


class ioThinx_4530_API(object):
    def __init__(self):
        super(ioThinx_4530_API, self).__init__()
        self.shared_library_dir = "/usr/lib/iothinx/"
        self.shared_library_filename = "libiothinxio.so"

        shared_library = self.shared_library_dir + self.shared_library_filename
        self.lib = ctypes.cdll.LoadLibrary(shared_library)
        self.ioThinx_IO_Client_Init()

    def get_shared_library_version(self):
        for file in os.listdir(self.shared_library_dir):
            if self.shared_library_filename in file and len(file.split(".")) == 5:
                return(str(".".join(file.split(".")[2:5])))

    def ioThinx_IO_Client_Init(self):
        error = self.lib.ioThinx_IO_Client_Init(None)
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_IO_GetBusStatus(self):
        p_status = ctypes.c_uint8()
        error = self.lib.ioThinx_IO_GetBusStatus(ctypes.byref(p_status))
        if error is not 0:
            raise ioThinxError(error)
        return p_status.value

    def ioThinx_IO_Config_Reload(self):
        error = self.lib.ioThinx_IO_Config_Reload(None)
        if error is not 0:
            raise ioThinxError(error)

#
# DI
#

    def ioThinx_DI_GetValues(self, slot):
        p_values = ctypes.c_uint32()
        error = self.lib.ioThinx_DI_GetValues(ctypes.c_uint32(slot),
                                              ctypes.byref(p_values))
        if error is not 0:
            raise ioThinxError(error)
        values = [int(b) for b in bin(p_values.value)
                  [2:].zfill(SLOT_CH_NUM)[::-1]]
        return values

    def ioThinx_DI_GetCntValues(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_DI_GetCntValues(ctypes.c_uint32(slot),
                                                 ctypes.c_uint8(start),
                                                 ctypes.c_uint8(count),
                                                 ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DI_SetCntValues(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint32 * count)(*buff_list)
        error = self.lib.ioThinx_DI_SetCntValues(ctypes.c_uint32(slot),
                                                 ctypes.c_uint8(start),
                                                 ctypes.c_uint8(count),
                                                 ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_GetCntStarts(self, slot):
        p_starts = ctypes.c_uint32()
        error = self.lib.ioThinx_DI_GetCntStarts(ctypes.c_uint32(slot),
                                                 ctypes.byref(p_starts))
        if error is not 0:
            raise ioThinxError(error)
        starts = [int(b) for b in bin(p_starts.value)
                  [2:].zfill(SLOT_CH_NUM)[::-1]]
        return starts

    def ioThinx_DI_SetCntStarts(self, slot, buff_list):
        starts = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_DI_SetCntStarts(ctypes.c_uint32(slot),
                                                 ctypes.c_uint32(starts))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_SetCntStops(self, slot, buff_list):
        starts = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_DI_SetCntStops(ctypes.c_uint32(slot),
                                                ctypes.c_uint32(starts))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_GetCntOverflows(self, slot):
        p_overflows = ctypes.c_uint32()
        error = self.lib.ioThinx_DI_GetCntOverflows(ctypes.c_uint32(slot),
                                                    ctypes.byref(p_overflows))
        if error is not 0:
            raise ioThinxError(error)
        overflows = [int(b) for b in bin(p_overflows.value)
                     [2:].zfill(SLOT_CH_NUM)[::-1]]
        return overflows

    def ioThinx_DI_SetCntOverflows(self, slot, buff_list):
        overflows = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_DI_SetCntOverflows(ctypes.c_uint32(slot),
                                                    ctypes.c_uint32(overflows))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_Config_GetModes(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_DI_Config_GetModes(ctypes.c_uint32(slot),
                                                    ctypes.c_uint8(start),
                                                    ctypes.c_uint8(count),
                                                    ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DI_Config_SetModes(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_DI_Config_SetModes(ctypes.c_uint32(slot),
                                                    ctypes.c_uint8(start),
                                                    ctypes.c_uint8(count),
                                                    ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_Config_GetFilters(self, slot, start, count):
        buff = (ctypes.c_uint16 * count)()
        error = self.lib.ioThinx_DI_Config_GetFilters(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(start),
                                                      ctypes.c_uint8(count),
                                                      ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DI_Config_SetFilters(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint16 * count)(*buff_list)
        error = self.lib.ioThinx_DI_Config_SetFilters(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(start),
                                                      ctypes.c_uint8(count),
                                                      ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_Config_GetCntTriggers(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_DI_Config_GetCntTriggers(ctypes.c_uint32(slot),
                                                          ctypes.c_uint8(start),
                                                          ctypes.c_uint8(count),
                                                          ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DI_Config_SetCntTriggers(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_DI_Config_SetCntTriggers(ctypes.c_uint32(slot),
                                                          ctypes.c_uint8(start),
                                                          ctypes.c_uint8(count),
                                                          ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DI_Config_GetCntValues(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_DI_Config_GetCntValues(ctypes.c_uint32(slot),
                                                        ctypes.c_uint8(start),
                                                        ctypes.c_uint8(count),
                                                        ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DI_Config_SetCntValues(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint32 * count)(*buff_list)
        error = self.lib.ioThinx_DI_Config_SetCntValues(ctypes.c_uint32(slot),
                                                        ctypes.c_uint8(start),
                                                        ctypes.c_uint8(count),
                                                        ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
#
# DO
#

    def ioThinx_DO_GetValues(self, slot):
        p_values = ctypes.c_uint32()
        error = self.lib.ioThinx_DO_GetValues(ctypes.c_uint32(slot),
                                              ctypes.byref(p_values))
        if error is not 0:
            raise ioThinxError(error)
        values = [int(b) for b in bin(p_values.value)
                  [2:].zfill(SLOT_CH_NUM)[::-1]]
        return values

    def ioThinx_DO_SetValues(self, slot, buff_list):
        values = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_DO_SetValues(ctypes.c_uint32(slot),
                                              ctypes.c_uint32(values))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_GetPwmCounts(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_DO_GetPwmCounts(ctypes.c_uint32(slot),
                                                 ctypes.c_uint8(start),
                                                 ctypes.c_uint8(count),
                                                 ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DO_SetPwmCounts(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint32 * count)(*buff_list)
        error = self.lib.ioThinx_DO_SetPwmCounts(ctypes.c_uint32(slot),
                                                 ctypes.c_uint8(start),
                                                 ctypes.c_uint8(count),
                                                 ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_GetPwmStarts(self, slot):
        p_starts = ctypes.c_uint32()
        error = self.lib.ioThinx_DO_GetPwmStarts(ctypes.c_uint32(slot),
                                                 ctypes.byref(p_starts))
        if error is not 0:
            raise ioThinxError(error)
        starts = [int(b) for b in bin(p_starts.value)
                  [2:].zfill(SLOT_CH_NUM)[::-1]]
        return starts

    def ioThinx_DO_SetPwmStarts(self, slot, buff_list):
        starts = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_DO_SetPwmStarts(ctypes.c_uint32(slot),
                                                 ctypes.c_uint32(starts))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_SetPwmStops(self, slot, buff_list):
        stops = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_DO_SetPwmStops(ctypes.c_uint32(slot),
                                                ctypes.c_uint32(stops))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_GetPwmConfigures(self, slot, start, count):
        frequencies_buff = (ctypes.c_uint16 * count)()
        duty_cycles_buff = (ctypes.c_uint16 * count)()
        error = self.lib.ioThinx_DO_GetPwmConfigures(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(
                                                         frequencies_buff),
                                                     ctypes.byref(duty_cycles_buff))
        if error is not 0:
            raise ioThinxError(error)

        frequencies_buff_list = []
        for value in frequencies_buff:
            frequencies_buff_list.append(value)
        duty_cycles_buff_list = []
        for value in duty_cycles_buff:
            duty_cycles_buff_list.append(value)
        return frequencies_buff_list, duty_cycles_buff_list

    def ioThinx_DO_SetPwmConfigures(self, slot, start, count,
                                    frequencies_buff_list,
                                    duty_cycles_buff_list):
        frequencies_buff = (ctypes.c_uint16 * count)(*frequencies_buff_list)
        duty_cycles_buff = (ctypes.c_uint16 * count)(*duty_cycles_buff_list)
        error = self.lib.ioThinx_DO_SetPwmConfigures(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(
                                                         frequencies_buff),
                                                     ctypes.byref(duty_cycles_buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_Config_GetModes(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_DO_Config_GetModes(ctypes.c_uint32(slot),
                                                    ctypes.c_uint8(start),
                                                    ctypes.c_uint8(count),
                                                    ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DO_Config_SetModes(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_DO_Config_SetModes(ctypes.c_uint32(slot),
                                                    ctypes.c_uint8(start),
                                                    ctypes.c_uint8(count),
                                                    ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_Config_GetPwmCounts(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_DO_Config_GetPwmCounts(ctypes.c_uint32(slot),
                                                        ctypes.c_uint8(start),
                                                        ctypes.c_uint8(count),
                                                        ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_DO_Config_SetPwmCounts(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint32 * count)(*buff_list)
        error = self.lib.ioThinx_DO_Config_SetPwmCounts(ctypes.c_uint32(slot),
                                                        ctypes.c_uint8(start),
                                                        ctypes.c_uint8(count),
                                                        ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_DO_Config_GetPwmConfigures(self, slot, start, count):
        frequencies_buff = (ctypes.c_uint16 * count)()
        duty_cycles_buff = (ctypes.c_uint16 * count)()
        error = self.lib.ioThinx_DO_Config_GetPwmConfigures(ctypes.c_uint32(slot),
                                                            ctypes.c_uint8(
                                                                start),
                                                            ctypes.c_uint8(
                                                                count),
                                                            ctypes.byref(
                                                                frequencies_buff),
                                                            ctypes.byref(duty_cycles_buff))
        if error is not 0:
            raise ioThinxError(error)

        frequencies_buff_list = []
        for value in frequencies_buff:
            frequencies_buff_list.append(value)
        duty_cycles_buff_list = []
        for value in duty_cycles_buff:
            duty_cycles_buff_list.append(value)

        return frequencies_buff_list, duty_cycles_buff_list

    def ioThinx_DO_Config_SetPwmConfigures(self, slot, start, count,
                                           frequencies_buff_list,
                                           duty_cycles_buff_list):
        frequencies_buff = (ctypes.c_uint16 * count)(*frequencies_buff_list)
        duty_cycles_buff = (ctypes.c_uint16 * count)(*duty_cycles_buff_list)
        error = self.lib.ioThinx_DO_Config_SetPwmConfigures(ctypes.c_uint32(slot),
                                                            ctypes.c_uint8(
                                                                start),
                                                            ctypes.c_uint8(
                                                                count),
                                                            ctypes.byref(
                                                                frequencies_buff),
                                                            ctypes.byref(duty_cycles_buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_Relay_GetValues(self, slot):
        p_values = ctypes.c_uint32()
        error = self.lib.ioThinx_Relay_GetValues(ctypes.c_uint32(slot),
                                                 ctypes.byref(p_values))
        if error is not 0:
            raise ioThinxError(error)
        values = [int(b) for b in bin(p_values.value)
                  [2:].zfill(SLOT_CH_NUM)[::-1]]
        return values

    def ioThinx_Relay_SetValues(self, slot, buff_list):
        values = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_Relay_SetValues(ctypes.c_uint32(slot),
                                                 ctypes.c_uint32(values))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_Relay_GetTotalCounts(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_Relay_GetTotalCounts(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(start),
                                                      ctypes.c_uint8(count),
                                                      ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_Relay_GetCurrentCounts(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_Relay_GetCurrentCounts(ctypes.c_uint32(slot),
                                                        ctypes.c_uint8(start),
                                                        ctypes.c_uint8(count),
                                                        ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_Relay_ResetCurrentCounts(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_Relay_ResetCurrentCounts(ctypes.c_uint32(slot),
                                                          ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AI_GetEngs(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_AI_GetEngs(ctypes.c_uint32(slot),
                                            ctypes.c_uint8(start),
                                            ctypes.c_uint8(count),
                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_GetMinEngs(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_AI_GetMinEngs(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_GetMaxEngs(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_AI_GetMaxEngs(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_GetRaws(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_AI_GetRaws(ctypes.c_uint32(slot),
                                            ctypes.c_uint8(start),
                                            ctypes.c_uint8(count),
                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_GetMinRaws(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_AI_GetMinRaws(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_GetMaxRaws(self, slot, start, count):
        buff = (ctypes.c_uint32 * count)()
        error = self.lib.ioThinx_AI_GetMaxRaws(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_ResetMins(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_AI_ResetMins(ctypes.c_uint32(slot),
                                              ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AI_ResetMaxs(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_AI_ResetMaxs(ctypes.c_uint32(slot),
                                              ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AI_GetStatuss(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_AI_GetStatuss(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_Config_GetRanges(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_AI_Config_GetRanges(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_Config_SetRanges(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_AI_Config_SetRanges(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AI_Config_GetBurnoutValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_AI_Config_GetBurnoutValues(ctypes.c_uint32(slot),
                                                            ctypes.c_uint8(
                                                                start),
                                                            ctypes.c_uint8(
                                                                count),
                                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AI_Config_SetBurnoutValues(self, slot, start, count, buff_list):
        buff = (ctypes.c_float * count)(*buff_list)
        error = self.lib.ioThinx_AI_Config_SetBurnoutValues(ctypes.c_uint32(slot),
                                                            ctypes.c_uint8(
                                                                start),
                                                            ctypes.c_uint8(
                                                                count),
                                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AO_GetEngs(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_AO_GetEngs(ctypes.c_uint32(slot),
                                            ctypes.c_uint8(start),
                                            ctypes.c_uint8(count),
                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AO_SetEngs(self, slot, start, count, buff_list):
        buff = (ctypes.c_float * count)(*buff_list)
        error = self.lib.ioThinx_AO_SetEngs(ctypes.c_uint32(slot),
                                            ctypes.c_uint8(start),
                                            ctypes.c_uint8(count),
                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AO_GetRaws(self, slot, start, count):
        buff = (ctypes.c_uint16 * count)()
        error = self.lib.ioThinx_AO_GetRaws(ctypes.c_uint32(slot),
                                            ctypes.c_uint8(start),
                                            ctypes.c_uint8(count),
                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AO_SetRaws(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint16 * count)(*buff_list)
        error = self.lib.ioThinx_AO_SetRaws(ctypes.c_uint32(slot),
                                            ctypes.c_uint8(start),
                                            ctypes.c_uint8(count),
                                            ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_AO_GetStatuss(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_AO_GetStatuss(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AO_Config_GetRanges(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_AO_Config_GetRanges(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_AO_Config_SetRanges(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_AO_Config_SetRanges(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_TC_GetValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_TC_GetValues(ctypes.c_uint32(slot),
                                              ctypes.c_uint8(start),
                                              ctypes.c_uint8(count),
                                              ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_TC_GetMinValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_TC_GetMinValues(ctypes.c_uint32(slot),
                                                 ctypes.c_uint8(start),
                                                 ctypes.c_uint8(count),
                                                 ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_TC_GetMaxValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_TC_GetMaxValues(ctypes.c_uint32(slot),
                                                 ctypes.c_uint8(start),
                                                 ctypes.c_uint8(count),
                                                 ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_TC_ResetMins(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_TC_ResetMins(ctypes.c_uint32(slot),
                                              ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_TC_ResetMaxs(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_TC_ResetMaxs(ctypes.c_uint32(slot),
                                              ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_TC_GetStatuss(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_TC_GetStatuss(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_TC_SetCalibrations(self, slot, start, count, buff_list):
        buff = (ctypes.c_float * count)(*buff_list)
        error = self.lib.ioThinx_TC_SetCalibrations(ctypes.c_uint32(slot),
                                                    ctypes.c_uint8(start),
                                                    ctypes.c_uint8(count),
                                                    ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_TC_ResetCalibrations(self, slot, start, count):
        error = self.lib.ioThinx_TC_ResetCalibrations(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(start),
                                                      ctypes.c_uint8(count))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_TC_Config_GetSensorTypes(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_TC_Config_GetSensorTypes(ctypes.c_uint32(slot),
                                                          ctypes.c_uint8(start),
                                                          ctypes.c_uint8(count),
                                                          ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_TC_Config_SetSensorTypes(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_TC_Config_SetSensorTypes(ctypes.c_uint32(slot),
                                                          ctypes.c_uint8(start),
                                                          ctypes.c_uint8(count),
                                                          ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_RTD_GetValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_RTD_GetValues(ctypes.c_uint32(slot),
                                               ctypes.c_uint8(start),
                                               ctypes.c_uint8(count),
                                               ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_RTD_GetMinValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_RTD_GetMinValues(ctypes.c_uint32(slot),
                                                  ctypes.c_uint8(start),
                                                  ctypes.c_uint8(count),
                                                  ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_RTD_GetMaxValues(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_RTD_GetMaxValues(ctypes.c_uint32(slot),
                                                  ctypes.c_uint8(start),
                                                  ctypes.c_uint8(count),
                                                  ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_RTD_ResetMins(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_RTD_ResetMins(ctypes.c_uint32(slot),
                                               ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_RTD_ResetMaxs(self, slot, buff_list):
        resets = int("".join(map(str, buff_list[:: -1])), BIN_TYPE)
        error = self.lib.ioThinx_RTD_ResetMaxs(ctypes.c_uint32(slot),
                                               ctypes.c_uint32(resets))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_RTD_GetStatuss(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_RTD_GetStatuss(ctypes.c_uint32(slot),
                                                ctypes.c_uint8(start),
                                                ctypes.c_uint8(count),
                                                ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_RTD_SetCalibrations(self, slot, start, count, buff_list):
        buff = (ctypes.c_float * count)(*buff_list)
        error = self.lib.ioThinx_RTD_SetCalibrations(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(start),
                                                     ctypes.c_uint8(count),
                                                     ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_RTD_ResetCalibrations(self, slot, start, count):
        error = self.lib.ioThinx_RTD_ResetCalibrations(ctypes.c_uint32(slot),
                                                       ctypes.c_uint8(start),
                                                       ctypes.c_uint8(count))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_RTD_Config_GetSensorTypes(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_RTD_Config_GetSensorTypes(ctypes.c_uint32(slot),
                                                           ctypes.c_uint8(
                                                               start),
                                                           ctypes.c_uint8(
                                                               count),
                                                           ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_RTD_Config_SetSensorTypes(self, slot, start, count, buff_list):
        buff = (ctypes.c_uint8 * count)(*buff_list)
        error = self.lib.ioThinx_RTD_Config_SetSensorTypes(ctypes.c_uint32(slot),
                                                           ctypes.c_uint8(
                                                               start),
                                                           ctypes.c_uint8(
                                                               count),
                                                           ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

#
# PWR
#

    def ioThinx_PWR_GetSysStatus(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_PWR_GetSysStatus(ctypes.c_uint32(slot),
                                                  ctypes.c_uint8(start),
                                                  ctypes.c_uint8(count),
                                                  ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_PWR_GetFieldStatus(self, slot, start, count):
        buff = (ctypes.c_uint8 * count)()
        error = self.lib.ioThinx_PWR_GetFieldStatus(ctypes.c_uint32(slot),
                                                    ctypes.c_uint8(start),
                                                    ctypes.c_uint8(count),
                                                    ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_PWR_Config_GetAlarms(self, slot, start, count):
        buff = (ctypes.c_float * count)()
        error = self.lib.ioThinx_PWR_Config_GetAlarms(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(start),
                                                      ctypes.c_uint8(count),
                                                      ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)
        buff_list = []
        for value in buff:
            buff_list.append(value)
        return buff_list

    def ioThinx_PWR_Config_SetAlarms(self, slot, start, count, buff_list):
        buff = (ctypes.c_float * count)(*buff_list)
        error = self.lib.ioThinx_PWR_Config_SetAlarms(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(start),
                                                      ctypes.c_uint8(count),
                                                      ctypes.byref(buff))
        if error is not 0:
            raise ioThinxError(error)

#
# Misc
#

    def ioThinx_Misc_GetModuleInfo(self, slot):
        class MODULE_INFO(ctypes.Structure):
            _fields_ = [('model_name', (ctypes.c_uint8 * 20)),
                        ('product_id', ctypes.c_uint32),
                        ('fwr_version', ctypes.c_uint16),
                        ('fwr_build_date', ctypes.c_uint32),
                        ('serial_number', (ctypes.c_uint8 * 13))]

        p_module_info = MODULE_INFO()
        error = self.lib.ioThinx_Misc_GetModuleInfo(ctypes.c_uint8(slot),
                                                    ctypes.byref(p_module_info))
        if error is not 0:
            raise ioThinxError(error)

        def convert_build_date_from_hex_to_datetime():
            build_date = hex(p_module_info.fwr_build_date).strip(
                '0x').strip('L')
            build_date = ['{:02d}'.format(
                int(build_date[i:i + 2], 16)) for i in range(0, len(build_date), BIN_TYPE)]
            build_date = datetime.strptime("".join(build_date), '%y%m%d%H')
            return build_date
        fwr_build_date = convert_build_date_from_hex_to_datetime()

        module_info = {
            'model_name': str(bytearray(p_module_info.model_name).decode().rstrip('\x00')),
            'product_id': hex(p_module_info.product_id).rstrip("L"),
            'fwr_version': hex(p_module_info.fwr_version),
            'fwr_build_date': fwr_build_date,
            'serial_number': str(bytearray(p_module_info.serial_number).decode().rstrip('\x00')),
        }
        return module_info

    def ioThinx_Misc_GetModuleInfoML(self, slot):
        class MODULE_INFO(ctypes.Structure):
            _fields_ = [('model_name', (ctypes.c_uint8 * 20)),
                        ('product_id', ctypes.c_uint32),
                        ('fwr_version', ctypes.c_uint16),
                        ('fwr_build_date', ctypes.c_uint32),
                        ('serial_number', (ctypes.c_uint8 * 13))]

        p_module_info = MODULE_INFO()
        error = self.lib.ioThinx_Misc_GetModuleInfoML(ctypes.c_uint8(slot),
                                                      ctypes.byref(p_module_info))
        if error is not 0:
            raise ioThinxError(error)

        def convert_build_date_from_hex_to_datetime():
            build_date = hex(p_module_info.fwr_build_date).strip(
                '0x').strip('L')
            build_date = ['{:02d}'.format(
                int(build_date[i:i + 2], 16)) for i in range(0, len(build_date), BIN_TYPE)]
            build_date = datetime.strptime("".join(build_date), '%y%m%d%H')
            return build_date
        fwr_build_date = convert_build_date_from_hex_to_datetime()

        module_info = {
            'model_name': str(bytearray(p_module_info.model_name).decode().rstrip('\x00')),
            'product_id': hex(p_module_info.product_id).rstrip("L"),
            'fwr_version': hex(p_module_info.fwr_version),
            'fwr_build_date': fwr_build_date,
            'serial_number': str(bytearray(p_module_info.serial_number).decode().rstrip('\x00')),
        }
        return module_info

    def ioThinx_Misc_GetModuleCount(self):
        p_module_count = ctypes.c_uint32()
        error = self.lib.ioThinx_Misc_GetModuleCount(
            ctypes.byref(p_module_count))
        if error is not 0:
            raise ioThinxError(error)
        return p_module_count.value

    def ioThinx_Misc_GetModuleCountML(self):
        p_module_count = ctypes.c_uint32()
        error = self.lib.ioThinx_Misc_GetModuleCountML(
            ctypes.byref(p_module_count))
        if error is not 0:
            raise ioThinxError(error)
        return p_module_count.value

    def ioThinx_Misc_GetRotarySwitchState(self, slot):
        p_state = ctypes.c_uint8()
        error = self.lib.ioThinx_Misc_GetRotarySwitchState(ctypes.c_uint32(slot),
                                                           ctypes.byref(p_state))
        if error is not 0:
            raise ioThinxError(error)
        return p_state.value

    def ioThinx_Misc_SetUserLedState(self, slot, channel, state):
        error = self.lib.ioThinx_Misc_SetUserLedState(ctypes.c_uint32(slot),
                                                      ctypes.c_uint8(channel),
                                                      ctypes.c_uint8(state))
        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_Misc_GetPushButtonState(self, slot):
        p_state = ctypes.c_uint8()
        error = self.lib.ioThinx_Misc_GetPushButtonState(ctypes.c_uint32(slot),
                                                         ctypes.byref(p_state))
        if error is not 0:
            raise ioThinxError(error)
        return p_state.value

    def ioThinx_Misc_GetLocateState(self, slot):
        p_state = ctypes.c_uint8()
        error = self.lib.ioThinx_Misc_GetLocateState(ctypes.c_uint32(slot),
                                                     ctypes.byref(p_state))
        if error is not 0:
            raise ioThinxError(error)
        return p_state.value

    def ioThinx_Misc_SetLocateState(self, slot, state):
        error = self.lib.ioThinx_Misc_SetLocateState(ctypes.c_uint32(slot),
                                                     ctypes.c_uint8(state))
        if error is not 0:
            raise ioThinxError(error)

#
# Communication
#

    def ioThinx_Serial_GetDevName(self, slot, port):
        buff = (ctypes.c_uint8 * 20)()
        p_name = (ctypes.c_uint8 * 20)()
        error = self.lib.ioThinx_Serial_GetDevName(
            ctypes.c_uint32(slot),
            ctypes.c_uint32(port),
            ctypes.byref(p_name)
        )
        if error is not 0:
            raise ioThinxError(error)

        buff_list = []
        for value in p_name:
            buff_list.append(value)

        dev_name = bytes(buff_list).decode('ascii')
        return dev_name

    def ioThinx_Serial_SetInterface(self, slot, port, mode):
        error = self.lib.ioThinx_Serial_SetInterface(
            ctypes.c_uint32(slot),
            ctypes.c_uint32(port),
            ctypes.c_uint32(mode)
        )

        if error is not 0:
            raise ioThinxError(error)

    def ioThinx_Serial_GetInterface(self, slot, port):
        mode = ctypes.c_uint32()
        error = self.lib.ioThinx_Serial_GetInterface(
            ctypes.c_uint32(slot),
            ctypes.c_uint32(port),
            ctypes.byref(mode)
        )
        if error is not 0:
            raise ioThinxError(error)
        return mode.value
