/* Copyright (C) 2019 Moxa Inc. All rights reserved. */
/* SPDX-License-Identifier: Apache-2.0 */
/*!
    \file iothinxio.h
    \copyright Moxa Inc.
    \brief <b>\a Moxa \a ioThinx Library</b>
    \date 2018-10-07
    \author Wanhan Hsieh
    \version V1.0
*/

#ifndef __IOTHINXIO_H__
#define __IOTHINXIO_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

/*!
    \addtogroup Library
    @{
*/

/*!
    \addtogroup ioThinx_IO
    @{
*/
/*!
    \addtogroup IOTHINX_ERR
    @{
*/
typedef int32_t IOTHINX_ERR;
#define IOTHINX_ERR_OK              0
#define IOTHINX_ERR_INIT            1
#define IOTHINX_ERR_SLOT            2
#define IOTHINX_ERR_ARGUMENT        3
#define IOTHINX_ERR_CONFIG          4
#define IOTHINX_ERR_RANGE           5
#define IOTHINX_ERR_COMMUNICATION   6
//! @}

/*!
    \addtogroup IOTHINX_BUS_STATUS
    @{
*/
#define IOTHINX_BUS_OK              0
#define IOTHINX_BUS_ERROR           1
//! @}

/*!
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note       This function \b MUST be called before using ioThinx library.
*/
IOTHINX_ERR ioThinx_IO_Client_Init(void);

/*!
    \param[out] p_status A pointer that stores \link IOTHINX_BUS_STATUS I/O bus status\endlink.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_IO_GetBusStatus(uint8_t *p_status);

/*!
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note       This function \b MUST be called after using ioThinx config APIs.
                Config will not be loaded to the I/O modules until this function called.
*/
IOTHINX_ERR ioThinx_IO_Config_Reload(void);
//! @}

/*!
    \addtogroup DI
    @{
    \details
    \b Sample \b Code: \n
    \link di.c di.c\endlink \n
    \link counter.c counter.c\endlink
*/
/*!
    \addtogroup DI_MODE
    @{
*/
//DI MODE
#define DI_MODE_DI                  0
#define DI_MODE_COUNTER             1
//! @}

/*!
    \addtogroup CNT_TRIGGER
    @{
*/
//DI COUNTER TRIGGER TYPE
#define CNT_TRIGGER_RISING          0
#define CNT_TRIGGER_FALLING         1
#define CNT_TRIGGER_BOTH            2
//! @}

/*!
    \param[in]  slot Slot number of the I/O module.
    \param[out] p_values A pointer that stores DI values of the contiguous channels;
                    each bit holds one channel's value.
                    The bit 0 represents digital input status of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_GetValues(uint32_t slot, uint32_t *p_values);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf An array that stores the counter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_GetCntValues(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the counter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_SetCntValues(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[out] p_starts A pointer that stores counter start flags of the contiguous channels (0: stop, 1: start);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_GetCntStarts(uint32_t slot, uint32_t *p_starts);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] starts Stores counter start flags of the contiguous channels (0: ignore, 1: start);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_SetCntStarts(uint32_t slot, uint32_t starts);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] stops Stores counter stop flags of the contiguous channels (0: ignore, 1: stop);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_SetCntStops(uint32_t slot, uint32_t stops);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[out] p_overflows A pointer that stores counter overflow flags of the contiguous channels (0: normal, 1: overflow);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_GetCntOverflows(uint32_t slot, uint32_t *p_overflows);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] overflows Stores counter overflow flags of the contiguous channels (0: normal, 1: overflow);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_SetCntOverflows(uint32_t slot, uint32_t overflows);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf An array that stores \link DI_MODE DI modes\endlink of the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_GetModes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores \link DI_MODE DI modes\endlink of the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
 */
IOTHINX_ERR ioThinx_DI_Config_SetModes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf An array that stores the filter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \li 45MR-16xx: one unit = 500us.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_GetFilters(uint32_t slot, uint8_t start, uint8_t count, uint16_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the filter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \li 45MR-16xx: one unit = 500us.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_SetFilters(uint32_t slot, uint8_t start, uint8_t count, uint16_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf An array that stores \link CNT_TRIGGER counter triggers\endlink of the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_GetCntTriggers(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf An array that stores \link CNT_TRIGGER counter triggers\endlink of the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_SetCntTriggers(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf An array that stores the counter values in config for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_GetCntValues(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the counter values in config for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DI_Config_SetCntValues(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup DO
    @{
    \details
    \b Sample \b Code: \n
    \link do.c do.c\endlink \n
    \link pwm.c pwm.c\endlink
*/

/*!
    \addtogroup DO_MODE
    @{
*/
//DO MODE
#define DO_MODE_DO                  0
#define DO_MODE_PWM                 1
//! @}

/*!
    \param[in]  slot Slot number of the I/O module.
    \param[out] p_values A pointer that stores DO values of the contiguous channels;
                    each bit holds one channel's value.
                    The bit 0 represents digital output status of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_GetValues(uint32_t slot, uint32_t *p_values);

/*!
    \param[in]  slot Slot number of the I/O module.
    \param[in]  values DO values of the contiguous channels;
                    each bit holds one channel's value.
                    The bit 0 represents digital output status of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_SetValues(uint32_t slot, uint32_t values);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the PWM counter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_GetPwmCounts(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be set.
    \param[in] buf  An array that stores the PWM counter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_SetPwmCounts(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[out] p_starts A pointer that stores PWM start flags of the contiguous channels (0: stop, 1: start);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_GetPwmStarts(uint32_t slot, uint32_t *p_starts);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] starts Stores PWM start flags of the contiguous channels (0: ignore, 1: start);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_SetPwmStarts(uint32_t slot, uint32_t starts);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] stops Stores PWM stop flags of the contiguous channels (0: ignore, 1: start);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_SetPwmStops(uint32_t slot, uint32_t stops);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be set.
    \param[out] frequencies An array that stores the PWM frequency (Hz) for the corresponding channels. Range = 1Hz ~ MAX supported freqency.
                    frequencies[0] represents the value of the start channel.
    \param[out] duty_cycles An array that stores the PWM ON duty cycle (%) for the corresponding channels. Range = 0, 1, ..., 100 (%).
                    duty_cycles[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_GetPwmConfigures(uint32_t slot, uint8_t start, uint8_t count, uint16_t frequencies[], uint16_t duty_cycles[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be set.
    \param[in] frequencies An array that stores the PWM frequency (Hz) for the corresponding channels. Range = 1Hz ~ MAX supported freqency.
                    frequencies[0] represents the value of the start channel.
    \param[in] duty_cycles An array that stores the PWM ON duty cycle (%) for the corresponding channels. Range = 0, 1, ..., 100 (%).
                    duty_cycles[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_SetPwmConfigures(uint32_t slot, uint8_t start, uint8_t count, uint16_t frequencies[], uint16_t duty_cycles[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf An array that stores \link DO_MODE DO modes\endlink of the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_Config_GetModes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores \link DO_MODE DO modes\endlink of the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_Config_SetModes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the PWM counter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_Config_GetPwmCounts(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be set.
    \param[in] buf  An array that stores the counter values for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_Config_SetPwmCounts(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be set.
    \param[out] frequencies An array that stores the PWM frequency (Hz) for the corresponding channels. Range = 1Hz ~ MAX supported freqency.
                    frequencies[0] represents the value of the start channel.
    \param[out] duty_cycles An array that stores the PWM ON duty cycle (%) for the corresponding channels. Range = 0, 1, ..., 100 (%).
                    duty_cycles[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_Config_GetPwmConfigures(uint32_t slot, uint8_t start, uint8_t count, uint16_t frequencies[], uint16_t duty_cycles[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be set.
    \param[in] frequencies An array that stores the PWM frequency (Hz) for the corresponding channels. Range = 1Hz ~ MAX supported freqency.
                    frequencies[0] represents the value of the start channel.
    \param[in] duty_cycles An array that stores the PWM ON duty cycle (%) for the corresponding channels. Range = 0, 1, ..., 100 (%).
                    duty_cycles[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_DO_Config_SetPwmConfigures(uint32_t slot, uint8_t start, uint8_t count, uint16_t frequencies[], uint16_t duty_cycles[]);

//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup Relay
    @{
    \details
    \b Sample \b Code: \n
    \link relay.c relay.c\endlink
*/

/*!
    \param[in]  slot Slot number of the I/O module.
    \param[out] p_values A pointer that stores Relay values of the contiguous channels;
                    each bit holds one channel's value.
                    The bit 0 represents relay status of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Relay_GetValues(uint32_t slot, uint32_t *p_values);

/*!
    \param[in]  slot Slot number of the I/O module.
    \param[in]  values Relay values of the contiguous channels;
                    each bit holds one channel's value.
                    The bit 0 represents relay status of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Relay_SetValues(uint32_t slot, uint32_t values);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the relay total count for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Relay_GetTotalCounts(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the relay current count for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Relay_GetCurrentCounts(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in]  slot Slot number of the I/O module.
    \param[in]  resets Stores the current count reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Relay_ResetCurrentCounts(uint32_t slot, uint32_t resets);
//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup AI
    @{
    \details
    \b Sample \b Code: \n
    \link ai.c ai.c\endlink
*/
/*!
    \addtogroup AI_RANGE
    @{
*/
//AI RANGE
#define AI_RANGE_DISABLE                0
#define AI_RANGE_0_10V                  1
#define AI_RANGE_0_20mA                 2
#define AI_RANGE_4_20mA_BURNOUT         3
#define AI_RANGE_4_20mA                 4
#define AI_RANGE_10V                    5
//! @}

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the AI engineering data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.

*/
IOTHINX_ERR ioThinx_AI_GetEngs(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the minimum AI engineering data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_GetMinEngs(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the maximal AI engineering data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_GetMaxEngs(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the AI raw data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_GetRaws(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the minimal AI raw data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_GetMinRaws(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the maximal AI raw data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_GetMaxRaws(uint32_t slot, uint8_t start, uint8_t count, uint32_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] resets Stores the minimal value reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_ResetMins(uint32_t slot, uint32_t resets);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] resets Stores the maximal value reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_ResetMaxs(uint32_t slot, uint32_t resets);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the AI status for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_GetStatuss(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the \link AI_RANGE AI range\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_Config_GetRanges(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the \link AI_RANGE AI range\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AI_Config_SetRanges(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the burnout value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note These values are valid only when the \link AI_RANGE AI range\endlink is 4-20mA
*/
IOTHINX_ERR ioThinx_AI_Config_GetBurnoutValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the burnout value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note These values are valid only when the \link AI_RANGE AI range\endlink is 4-20mA
*/
IOTHINX_ERR ioThinx_AI_Config_SetBurnoutValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

//! @}
// ----------------------------------------------------------------------------
/*!
    \addtogroup AO
    @{
    \details
    \b Sample \b Code: \n
    \link ao.c ao.c\endlink \n
*/
/*!
    \addtogroup AO_RANGE
    @{
*/
//AO RANGE
#define AO_RANGE_DISABLE                0
#define AO_RANGE_0_10_V                 1
#define AO_RANGE_0_20mA                 2
#define AO_RANGE_4_20mA                 3
//! @}

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the AO eng data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_GetEngs(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf An array that stores the AO eng data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_SetEngs(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the AO raw data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_GetRaws(uint32_t slot, uint8_t start, uint8_t count, uint16_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf An array that stores the AO raw data for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_SetRaws(uint32_t slot, uint8_t start, uint8_t count, uint16_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the AO status for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_GetStatuss(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the \link AO_RANGE AO range\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_Config_GetRanges(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the \link AO_RANGE AO range\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_AO_Config_SetRanges(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup TC
    @{
    \details
    \b Sample \b Code: \n
    \link tc.c tc.c\endlink \n
    \link tc_calibration.c tc_calibration.c\endlink
*/
/*!
    \addtogroup TC_SENSOR_TYPE
    @{
*/
#define TC_SENSOR_TYPE_J                0 // (0 to 750 Celsius degree)
#define TC_SENSOR_TYPE_K                1 // (-200 to 1250 Celsius degree)
#define TC_SENSOR_TYPE_T                2 // (-200 to 350  Celsius degree)
#define TC_SENSOR_TYPE_E                3 // (-200 to 900  Celsius degree)
#define TC_SENSOR_TYPE_R                4 // (-50 to 1600  Celsius degree)
#define TC_SENSOR_TYPE_S                5 // (-50 to 1760  Celsius degree)
#define TC_SENSOR_TYPE_B                6 // (600 to 1700  Celsius degree)
#define TC_SENSOR_TYPE_N                7 // (-200 to 1300 Celsius degree)
#define TC_SENSOR_TYPE_75_126_MV        14 // (+/- 78.126 mV)
#define TC_SENSOR_TYPE_39_062_MV        15 // (+/- 39.062 mV)
#define TC_SENSOR_TYPE_19_532_MV        16 // (+/- 19.532 mV)
#define TC_SENSOR_TYPE_DISABLE          63
//! @}

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the TC value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_GetValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the minimal TC value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_GetMinValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the maximal TC value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_GetMaxValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] resets Stores the minimal value reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note This API only reset the minimal value, reset is setting the value to be current value.
*/
IOTHINX_ERR ioThinx_TC_ResetMins(uint32_t slot, uint32_t resets);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] resets Stores the maximal value reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note This API only reset the maximal value, reset is setting the value to be current value.
*/
IOTHINX_ERR ioThinx_TC_ResetMaxs(uint32_t slot, uint32_t resets);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the TC status for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_GetStatuss(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the current temperatures for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_SetCalibrations(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be reset.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_ResetCalibrations(uint32_t slot, uint8_t start, uint8_t count);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the \link TC_SENSOR_TYPE TC type\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_Config_GetSensorTypes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the \link TC_SENSOR_TYPE TC type\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_TC_Config_SetSensorTypes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup RTD
    @{
    \details
    \b Sample \b Code: \n
    \link rtd.c rtd.c\endlink \n
    \link rtd_calibration.c rtd_calibration.c\endlink
*/
/*!
    \addtogroup RTD_SENSOR_TYPE
    @{
*/
#define RTD_SENSOR_TYPE_PT50            0 // (-200 to 850 Celsius degree)
#define RTD_SENSOR_TYPE_PT100           1 // (-200 to 850 Celsius degree)
#define RTD_SENSOR_TYPE_PT200           2 // (-200 to 850 Celsius degree)
#define RTD_SENSOR_TYPE_PT500           3 // (-200 to 850 Celsius degree)
#define RTD_SENSOR_TYPE_PT1000          4 // (-200 to 350 Celsius degree)
#define RTD_SENSOR_TYPE_8X              14 // (310 ohm)
#define RTD_SENSOR_TYPE_4X              15 // (620 ohm)
#define RTD_SENSOR_TYPE_2X              16 // (1250 ohm)
#define RTD_SENSOR_TYPE_1X              17 // (2200 ohm)
#define RTD_SENSOR_TYPE_JPT100          20 // (-200 to 640 Celsius degree)
#define RTD_SENSOR_TYPE_JPT200          21 // (-200 to 640 Celsius degree)
#define RTD_SENSOR_TYPE_JPT500          22 // (-200 to 640 Celsius degree)
#define RTD_SENSOR_TYPE_JPT1000         23 // (-200 to 350 Celsius degree)
#define RTD_SENSOR_TYPE_NI100           30 // (-60 to 250  Celsius degree)
#define RTD_SENSOR_TYPE_NI200           31 // (-60 to 250  Celsius degree)
#define RTD_SENSOR_TYPE_NI500           32 // (-60 to 250  Celsius degree)
#define RTD_SENSOR_TYPE_NI1000          33 // (-60 to 150  Celsius degree)
#define RTD_SENSOR_TYPE_NI120           34 // (-80 to 260  Celsius degree)
#define RTD_SENSOR_TYPE_DISABLE         63
//! @}

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the RTD value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_GetValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the minimal RTD value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_GetMinValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the maximal RTD value for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_GetMaxValues(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] resets Stores the minimal value reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note This API only reset the minimal value, reset is setting the value to be current value.
*/
IOTHINX_ERR ioThinx_RTD_ResetMins(uint32_t slot, uint32_t resets);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] resets Stores the maximal value reset flags of the contiguous channels (0: ignore, 1: reset);
                    each bit holds one channel's flag.
                    The bit 0 represents value of the channel 0 and so on.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note This API only reset the maximal value, reset is setting the value to be current value.
*/
IOTHINX_ERR ioThinx_RTD_ResetMaxs(uint32_t slot, uint32_t resets);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the RTD status for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_GetStatuss(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the current temperatures for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_SetCalibrations(uint32_t slot, uint8_t start, uint8_t count, float buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be reset.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_ResetCalibrations(uint32_t slot, uint8_t start, uint8_t count);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the \link RTD_SENSOR_TYPE RTD type\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_Config_GetSensorTypes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the I/O module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the \link RTD_SENSOR_TYPE RTD type\endlink for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_RTD_Config_SetSensorTypes(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

//! @}


// ----------------------------------------------------------------------------
/*!
    \addtogroup Power
    @{
    \details
    \b Sample \b Code: \n
    \link pwr.c pwr.c\endlink
*/
/*!
    \addtogroup PWR_STATUS
    @{
*/
#define SYS_PWR_STATUS_CONNECTED        0
#define SYS_PWR_STATUS_WARNING          1 // system power under setting
#define FIELD_PWR_STATUS_CONNECTED      0
#define FIELD_PWR_STATUS_DISCONNECTED   1
#define FIELD_PWR_STATUS_WARNING        2 // field power over Spec.
//! @}

/*!
    \param[in] slot Slot number of the module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the system power status for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_PWR_GetSysStatus(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the field power status for the contiguous channels.
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_PWR_GetFieldStatus(uint32_t slot, uint8_t start, uint8_t count, uint8_t buf[]);

/*!
    \param[in] slot Slot number of the module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[out] buf  An array that stores the alarm (V) configs for the contiguous channels. Range = 9 ~ 45 (V).
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_PWR_Config_GetAlarms(uint32_t slot, uint8_t start, uint8_t count, float buf[]);


/*!
    \param[in] slot Slot number of the module.
    \param[in] start Specifies the start channel.
    \param[in] count The number of channels to be get.
    \param[in] buf  An array that stores the alarm (V) configs for the contiguous channels. Range = 9 ~ 45 (V).
                    buf[0] represents the value of the start channel.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_PWR_Config_SetAlarms(uint32_t slot, uint8_t start, uint8_t count, float buf[]);
//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup Misc
    @{
    \details
    \b Sample \b Code: \n
    \link misc.c misc.c\endlink
*/

/*!
    \addtogroup Module_Info
    @{
*/
struct MODULE_INFO
{
    uint8_t model_name[20];
    uint32_t product_id;
    uint16_t fwr_version;
    uint32_t fwr_build_date;
    uint8_t serial_number[13];
};
//! @}

/*!
    \addtogroup LED_state
    @{
*/
#define LED_STATE_DARK      0
#define LED_STATE_GREEN     1
#define LED_STATE_RED       2
//! @}
/*!
    \addtogroup User_LED_Channel
    @{
*/
#define LED_CHANNEL_U1      1
#define LED_CHANNEL_U2      2
//! @}

/*!
    \param[in]  slot Slot number of the module.
    \param[out] p_module_info A pointer that stores the module information of the slot.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_GetModuleInfo(uint8_t slot, struct MODULE_INFO *p_module_info);

/*!
    \param[in]  slot Slot number of the module.
    \param[out] p_module_info A pointer that stores the module information of the slot.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
    \note Using this API to get ML module's information.
*/
IOTHINX_ERR ioThinx_Misc_GetModuleInfoML(uint8_t slot, struct MODULE_INFO *p_module_info);

/*!
    \param[out] p_module_count A pointer that stores the connected MR module count of the device.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_GetModuleCount(uint32_t *p_module_count);

/*!
    \param[out] p_module_count A pointer that stores the connected ML module count of the device.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_GetModuleCountML(uint32_t *p_module_count);

/*!
    \param[in] slot Slot number of the module.
    \param[out] p_state A pointer that stores the rotary switch state of the slot.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_GetRotarySwitchState(uint32_t slot, uint8_t *p_state);

/*!
    \param[in] slot Slot number of the module.
    \param[in] channel Specifies the \link User_LED_Channel user led channel\endlink.
    \param[in] state Stores the \link LED_state led state\endlink.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_SetUserLedState(uint32_t slot, uint8_t channel, uint8_t state);

/*!
    \param[in] slot Slot number of the module.
    \param[out] p_state A pointer that stores the push button state of the slot.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_GetPushButtonState(uint32_t slot, uint8_t *p_state);

/*!
    \param[in] slot Slot number of the module.
    \param[out] p_state A pointer that stores the locate state of the slot.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_GetLocateState(uint32_t slot, uint8_t *p_state);

/*!
    \param[in] slot Slot number of the module.
    \param[in] state Stores the locate state of the slot.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Misc_SetLocateState(uint32_t slot, uint8_t state);

//! @}

// ----------------------------------------------------------------------------
/*!
    \addtogroup SERIAL
    @{
    \details
    \b Sample \b Code: \n
    \link serial.c seria.c\endlink
*/
/*!
    \addtogroup SERIAL_PORT
    @{
*/
#define SERIAL_PORT_1           0
#define SERIAL_PORT_2           1
#define SERIAL_PORT_3           2
#define SERIAL_PORT_4           3
//! @}
/*!
    \addtogroup SERIAL_INTERFACE
    @{
*/
#define SERIAL_INTERFACE_RS232          0
#define SERIAL_INTERFACE_RS485_2WIRE    1
#define SERIAL_INTERFACE_RS422          2
//! @}
/*!
    \param[in] slot Slot number of the module.
    \param[in] port Specifies the \link SERIAL_PORT SERIAL port\endlink.
    \param[out] name A pointer that stores the device name.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Serial_GetDevName(uint32_t slot, uint32_t port, uint8_t *name);

/*!
    \param[in] slot Slot number of the module.
    \param[in] port Specifies the \link SERIAL_PORT SERIAL port\endlink.
    \param[in] interface Stores the \link SERIAL_INTERFACE SERIAL interface\endlink.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Serial_SetInterface(uint32_t slot, uint32_t port, uint32_t interface);

/*!
    \param[in] slot Slot number of the module.
    \param[in] port Specifies the \link SERIAL_PORT SERIAL port\endlink.
    \param[out] interface Stores the \link SERIAL_INTERFACE SERIAL interface\endlink.
    \return     \link IOTHINX_ERR_OK IOTHINX_ERR_OK\endlink if everything is ok.
                Otherwise, refer to the \link IOTHINX_ERR IOTHINX_ERR\endlink.
*/
IOTHINX_ERR ioThinx_Serial_GetInterface(uint32_t slot, uint32_t port, uint32_t *interface);

//! @}

//! @}

#ifdef __cplusplus
} // extern "C"
#endif

#endif // __IOTHINXIO_H__
