/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * IO Sample for Embedded OPC UA Server
 *
 * Date          Author            Comment
 * 2019-06-05    TJ Tai            Created it.
 ******************************************************************************/

/**
    \file io_node_operator.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>IO Sample for Embedded OPC UA Server</b>
    \date 2019-06-05
    \author TJ Tai
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is IO sample code for Embedded OPC UA Server.
    </pre>
**/


#include <unistd.h>
#include <string.h>

#include <iothinx/iothinxio.h>

/// include interfaces
#include <mx_node_sdk/mx_node_interface_basic.h>
#include <mx_node_sdk/mx_node_interface_data_access.h>

#include "DataSet.h"
#include "cJSON.h"

typedef enum
{
    OPERATOR_STATE_STOP,
    OPERATOR_STATE_STOPPING,
    OPERATOR_STATE_RUNNING,
} OPERATOR_STATE;

struct timeval now();
MX_NODE_RESULT add_io_nodes();

MX_NODE_NODE_OPERATOR_HANDLE g_operator_handle;
volatile OPERATOR_STATE g_state;

NODE **io_example;
int g_totalSlot;
int *g_slotNodeCountRecorder;

//! \cond
/// interface basic
const char* mx_node_operator_get_version()
{
    return "1.0";
}

long long mx_node_operator_get_supported_interfaces()
{
    return INTERFACE_MX_NODE_BASIC | INTERFACE_MX_NODE_DATA_ACCESS;
}

MX_NODE_RESULT mx_node_operator_initialize(MX_NODE_NODE_OPERATOR_HANDLE operator_handle)
{
    int rc, i, j;
    char buffer[64];

    g_operator_handle = operator_handle;
    g_state = OPERATOR_STATE_STOP;

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return MX_NODE_RESULT_BAD;
    }

    // decide load config place in the device
    rc = parse_json_file("/usr/local/bin/embeddedopcuaserver/Config/io.conf");
    if (rc == MX_PARSE_CONFIG_ERR){
        printf("Load configure error, can not open config file.");
        return MX_NODE_RESULT_BAD;
    }

    return add_io_nodes();
}

void mx_node_operator_uninitialize()
{
    size_t i,j;
    for (i = 0; i < g_totalSlot; ++i)
        free(io_example[i]);

    free(io_example);
    free(g_slotNodeCountRecorder);
    printf("Uninit finished!...\n");
}

void mx_node_operator_start()
{
    g_state = OPERATOR_STATE_RUNNING;
    while (g_state == OPERATOR_STATE_RUNNING)
    {
        for(int i=0;i<g_totalSlot;i++)
        {
            for(int j=0;j<g_slotNodeCountRecorder[i];j++)
            {
                if (io_example[i][j].access == MX_NODE_ACCESS_RIGHT_READONLY || io_example[i][j].access == MX_NODE_ACCESS_RIGHT_READWRITE) {
                    io_control_read(i, j, &io_example[i][j].data);
                }

                io_example[i][j].timestamp = Now();
                mx_node_update_node(
                        io_example[i][j].node_handle,
                        &io_example[i][j].data,
                        &io_example[i][j].timestamp
                        );
            }
        }
        sleep(1);
    }
    g_state = OPERATOR_STATE_STOP;
}

void mx_node_operator_stop()
{
    if (g_state == OPERATOR_STATE_RUNNING)
    {
        g_state = OPERATOR_STATE_STOPPING;
        while (g_state != OPERATOR_STATE_STOP);
    }
}
/// interface basic end

/// interface data_access
MX_NODE_RESULT mx_node_operator_read_node(MX_NODE_NODE_HANDLE node_handle, MX_NODE_VARIANT* node_value, struct timeval* node_timestamp)
{
    int rc, i,j;
    FUNC_TYPE fn;

    for(i=0;i<g_totalSlot;i++)
    {
        for(j=0;j<g_slotNodeCountRecorder[i];j++)
        {
            if (io_example[i][j].node_handle == node_handle) {
                printf("io[%d,%d] ch: %d, access: %d, var_type: %d, func_type: %d, name: %s, description: %s\n",
                        i,
                        j,
                        io_example[i][j].ch,
                        io_example[i][j].access,
                        io_example[i][j].var_type,
                        io_example[i][j].func_type,
                        io_example[i][j].name,
                        io_example[i][j].description);

                if (io_example[i][j].access == MX_NODE_ACCESS_RIGHT_READONLY || io_example[i][j].access == MX_NODE_ACCESS_RIGHT_READWRITE) {
                    rc = io_control_read(i, j, node_value);
                    if (rc != MX_IO_CONTROL_OK) {
                        return MX_NODE_RESULT_BAD;
                    }
                    *node_timestamp = io_example[i][j].timestamp;
                }
            }
        }
    }

    return MX_NODE_RESULT_GOOD;
}

MX_NODE_RESULT mx_node_operator_write_node(MX_NODE_NODE_HANDLE node_handle, const MX_NODE_VARIANT* node_value)
{
    int rc, i,j;

    for(i=0;i<g_totalSlot;i++)
    {
        for(j=0;j<g_slotNodeCountRecorder[i];j++)
        {
            if (io_example[i][j].node_handle == node_handle) {
                printf("io[%d,%d] ch: %d, access: %d, var_type: %d, func_type: %d, name: %s, description: %s\n",
                        i,
                        j,
                        io_example[i][j].ch,
                        io_example[i][j].access,
                        io_example[i][j].var_type,
                        io_example[i][j].func_type,
                        io_example[i][j].name,
                        io_example[i][j].description);

                if (io_example[i][j].access == MX_NODE_ACCESS_RIGHT_WRITEONLY || io_example[i][j].access == MX_NODE_ACCESS_RIGHT_READWRITE) {
                    rc = io_control_write(i, j, node_value);
                    if (rc != MX_IO_CONTROL_OK) {
                        return MX_NODE_RESULT_BAD;
                    }
                    io_example[i][j].timestamp = Now();
                }
            }
        }
    }

    return MX_NODE_RESULT_GOOD;
}
/// interface data_access end
//! \endcond

MX_NODE_RESULT add_io_nodes()
{
    MX_NODE_RESULT res;
    int rc, i,j;

    // folder
    MX_NODE_NODE folder;
    {
        strcpy(folder.node_name, "ioThinx-4533");
        folder.node_type = MX_NODE_NODE_TYPE_FOLDER;
        strcpy(folder.description, "device name");
    }
    MX_NODE_NODE_HANDLE folder_handle;
    res = mx_node_add_node(g_operator_handle, &folder, &folder_handle);
    if (res != MX_NODE_RESULT_GOOD)
        return res;

    // object
    for(i=0;i<g_totalSlot;i++)
    {
        MX_NODE_NODE object;
        {
            sprintf(object.node_name, "Slot_%d", i);
            object.node_type = MX_NODE_NODE_TYPE_OBJECT;
            strcpy(object.description, "number of slot");
        }
        MX_NODE_NODE_HANDLE object_handle;
        res = mx_node_add_node(g_operator_handle, &object, &object_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
        res = mx_node_set_parent_node(object_handle, folder_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;

    // variable
        for(j=0;j<g_slotNodeCountRecorder[i];j++){
            MX_NODE_NODE variable;
            {
                strcpy(variable.node_name, io_example[i][j].name);
                variable.node_type = MX_NODE_NODE_TYPE_VARIABLE;
                strcpy(variable.description, io_example[i][j].description);
                variable.variable.access_right = io_example[i][j].access;
                io_example[i][j].data.type = io_example[i][j].var_type;
                memset(&io_example[i][j].data.value, 0, sizeof(io_example[i][j].data.value));
                variable.variable.value = io_example[i][j].data;
            }
            res = mx_node_add_node(g_operator_handle, &variable, &io_example[i][j].node_handle);
            if (res != MX_NODE_RESULT_GOOD)
                return res;
            res = mx_node_set_parent_node(io_example[i][j].node_handle, object_handle);
            if (res != MX_NODE_RESULT_GOOD)
                return res;
        }
    }

    // Config IO mode
    uint8_t channel_count = 1;
    uint8_t di_mode;
    uint8_t do_mode;
    uint8_t counter_triggers;
    uint16_t pwm_freq = 5;
    uint16_t pwm_duty = 50;
    for (i = 0; i < g_totalSlot; ++i){
        for(j = 0; j< g_slotNodeCountRecorder[i]; j++){
            if (io_example[i][j].func_type == OPC_DO_PULSECOUNT || io_example[i][j].func_type == OPC_DO_PULSESTATUS) {
                // set to DO_MODE PULSE
                do_mode = DO_MODE_PWM;
                rc = ioThinx_DO_Config_SetModes(i, io_example[i][j].ch, channel_count, &do_mode);
                if (rc != IOTHINX_ERR_OK)
                {
                    printf("ioThinx_DO_Config_SetModes() = %d\n", rc);
                }
                //set frq and duty cycle
                rc = ioThinx_DO_Config_SetPwmConfigures(i, io_example[i][j].ch, channel_count, &pwm_freq, &pwm_duty);
                if (rc != IOTHINX_ERR_OK)
                {
                    printf("ioThinx_DO_Config_SetPwmConfigures() = %d\n", rc);
                }
            }
            if (io_example[i][j].func_type == OPC_DI_COUNTERSTATUS || io_example[i][j].func_type == OPC_DI_COUNTERVALUE) {
                // set to DI_MODE COUNTER
                di_mode = DI_MODE_COUNTER;
                counter_triggers = CNT_TRIGGER_BOTH;
                rc = ioThinx_DI_Config_SetModes(i, io_example[i][j].ch, channel_count, &di_mode);
                if (rc != IOTHINX_ERR_OK)
                {
                    printf("ioThinx_DI_Config_SetModes() = %d\n", rc);
                }
                rc = ioThinx_DI_Config_SetCntTriggers(i, io_example[i][j].ch, channel_count, &counter_triggers);
                if (rc != IOTHINX_ERR_OK)
                {
                    printf("ioThinx_DI_Config_SetCntTriggers() = %d\n", rc);
                }
            }
        }
    }
    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }
    return MX_NODE_RESULT_GOOD;
}

int parse_json_file(const char *filename)
{
    FILE *f;
    int len;
    char *data;

    f=fopen(filename,"rb");
    if (f == NULL)
        return MX_PARSE_CONFIG_ERR;
    fseek(f,0,SEEK_END);
    len=ftell(f);
    fseek(f,0,SEEK_SET);
    data=(char*)malloc(len+1);
    fread(data,1,len,f);
    fclose(f);

    printf("parse file %s complete, len=%d.\n",filename,len);

    cJSON_to_struct_array(data);

    free(data);
    return 0;
}

//parse a struct array
int cJSON_to_struct_array(char *text)
{
    cJSON *json,*item,*object, *slotItem, *chObj;
    int i,j;

    json=cJSON_Parse(text);
    if (!json)
    {
        printf("Error before: [%s]\n",cJSON_GetErrorPtr());
    }
    else     {
        slotItem=cJSON_GetObjectItem(json,"slot");
        if(slotItem!=NULL)
        {
            g_totalSlot =cJSON_GetArraySize(slotItem);
            //printf("cJSON_GetSlotSize: g_totalSlot=%d\n",g_totalSlot);
            io_example = (NODE **)malloc(sizeof(NODE *)*g_totalSlot);
            g_slotNodeCountRecorder = (int*)malloc(sizeof(int)*g_totalSlot);

            for(i=0;i<g_totalSlot;i++)
            {
                //printf("i=%d\n",i);
                chObj=cJSON_GetArrayItem(slotItem,i);
                int chsize = cJSON_GetArraySize(chObj);
                io_example[i] = (NODE *)calloc(chsize, sizeof(NODE));
                g_slotNodeCountRecorder[i] = chsize;
                //printf("chsize=%d\n",chsize);

                for(j=0;j<chsize;j++){
                    object=cJSON_GetArrayItem(chObj,j);

                    item=cJSON_GetObjectItem(object,"ch");
                    if(item!=NULL)
                    {
                        //printf("cJSON_GetObjectItem: type=%d, string is %s, valueint=%d\n",item->type,item->string,item->valueint);
                        io_example[i][j].ch=item->valueint;
                    }
                    else{
                        printf("cJSON_GetObjectItem: get ch failed\n");
                    }

                    item=cJSON_GetObjectItem(object,"access");
                    if(item!=NULL)
                    {
                        //printf("cJSON_GetObjectItem: type=%d, string is %s, valueint=%d\n",item->type,item->string,item->valueint);
                        io_example[i][j].access=item->valueint;
                    }
                    else{
                        printf("cJSON_GetObjectItem: get access failed\n");
                    }
                    item=cJSON_GetObjectItem(object,"variant_type");
                    if(item!=NULL)
                    {
                        //printf("cJSON_GetObjectItem: type=%d, string is %s, valueint=%d\n",item->type,item->string,item->valueint);
                        io_example[i][j].var_type=item->valueint;
                    }
                    else{
                        printf("cJSON_GetObjectItem: get variable_type failed\n");
                    }
                    item=cJSON_GetObjectItem(object,"func_type");
                    if(item!=NULL)
                    {
                        //printf("cJSON_GetObjectItem: type=%d, string is %s, valueint=%d\n",item->type,item->string,item->valueint);
                        io_example[i][j].func_type=item->valueint;
                    }
                    else{
                        printf("cJSON_GetObjectItem: get func_type failed\n");
                    }
                    item=cJSON_GetObjectItem(object,"name");
                    if(item!=NULL)
                    {
                        //printf("cJSON_GetObjectItem: type=%d, string is %s\n",item->type,item->string);
                        if (i==0 && io_example[i][j].ch == -1) // slot 0 name
                            sprintf(io_example[i][j].name, "S%d_%s", i, item->valuestring);
                        else
                            sprintf(io_example[i][j].name, "S%d_C%d_%s", i, io_example[i][j].ch, item->valuestring);
                    }
                    item=cJSON_GetObjectItem(object,"description");
                    if(item!=NULL)
                    {
                        //printf("cJSON_GetObjectItem: type=%d, string is %s\n",item->type,item->string);
                        memcpy(io_example[i][j].description,item->valuestring,strlen(item->valuestring));
                    }
                }
            }
        }

        //for(i=0;i<g_totalSlot;i++)
        //{
        //    for(j=0;j<g_slotNodeCountRecorder[i];j++)
        //    {
        //        printf("io[%d,%d] ch: %d, access: %d, var_type: %d, func_type: %d, name: %s, description: %s\n",
        //                i,
        //                j,
        //                io_example[i][j].ch,
        //                io_example[i][j].access,
        //                io_example[i][j].var_type,
        //                io_example[i][j].func_type,
        //                io_example[i][j].name,
        //                io_example[i][j].description);
        //    }
        //    printf("\r\n");
        //}

        cJSON_Delete(json);
    }
    return 0;
}

int io_control_read(int slot, int index, MX_NODE_VARIANT* node_value)
{
    FUNC_TYPE fn;
    int32_t rc = 0;
    int32_t value_int;
    uint32_t di_values = 0;
    uint32_t do_values = 0;
    uint32_t relay_values = 0;
    uint32_t pwm_starts = 0;
    uint32_t counter_starts = 0;
    uint32_t value_uint = 0;
    uint32_t ai_raws = 0;
    float value_float = 0;
    uint8_t value_uint8 = 0;

    switch(io_example[slot][index].func_type){
        case OPC_DI_MODE:
             rc = ioThinx_DI_Config_GetModes(slot, io_example[slot][index].ch, 1, &value_uint8);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_DI_Config_GetModes() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             io_example[slot][index].data.value.u32 = (uint32_t)value_uint8;
             break;
        case OPC_DI_STATUS:
             rc = ioThinx_DI_GetValues(slot, &di_values);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_DI_GetValues() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             value_int = (di_values >> io_example[slot][index].ch)& 0x1;
             //printf("\r\n=========> MX_Read() set value = %d\r\n", value_int);
             io_example[slot][index].data.value.i32 = value_int;
             break;
        case OPC_DI_COUNTERSTATUS:
             rc = ioThinx_DI_GetCntStarts(slot, &counter_starts);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("\r\n ioThinx_DI_GetCntStarts() = %d\r\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             value_int = (counter_starts >> io_example[slot][index].ch)& 0x1;
             io_example[slot][index].data.value.i32 = value_int;
             break;
        case OPC_DI_COUNTERVALUE:
             rc = ioThinx_DI_GetCntValues(slot, io_example[slot][index].ch, 1, &value_uint);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_DI_GetCntValues() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             io_example[slot][index].data.value.u32 = value_uint;
             break;
        case OPC_DO_MODE:
             rc = ioThinx_DO_Config_GetModes(slot, io_example[slot][index].ch, 1, &value_uint8);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_DO_Config_GetModes() = %d\n", rc);
             }
             io_example[slot][index].data.value.u32 = (uint32_t)value_uint8;
             break;
        case OPC_DO_STATUS:
             rc = ioThinx_DO_GetValues(slot, &do_values);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("\r\n*** [ERR] ioThinx_DO_GetValues() = %d\r\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             value_int = (do_values >> io_example[slot][index].ch)& 0x1;
             //printf("\r\n=========> MX_Read() set value = %d\r\n", value_int);
             io_example[slot][index].data.value.i32 = value_int;
             break;
        case OPC_DO_PULSESTATUS:
             rc = ioThinx_DO_GetPwmStarts(slot, &pwm_starts);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("\r\n ioThinx_DO_GetPwmStarts() = %d\r\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             value_int = (pwm_starts >> io_example[slot][index].ch)& 0x1;
             io_example[slot][index].data.value.i32 = value_int;
             break;
        case OPC_DO_PULSECOUNT:
             rc = ioThinx_DO_GetPwmCounts(slot, io_example[slot][index].ch, 1, &value_uint);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_DO_Config_GetPwmCounts() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             io_example[slot][index].data.value.u32 = value_uint;
             break;
        case OPC_RELAY_VALUE:
             rc = ioThinx_Relay_GetValues(slot, &relay_values);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_Relay_GetValues() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             value_int = (relay_values >> io_example[slot][index].ch)& 0x1;
             //printf("\r\n=========> MX_Read() set value = %d\r\n", value_int);
             io_example[slot][index].data.value.i32 = value_int;
             break;
        case OPC_AI_MODE:
             rc = ioThinx_AI_Config_GetRanges(slot, io_example[slot][index].ch, 1, &value_uint8);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_AI_GetEngs() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             //printf("\r\n=========> MX_Read() set value = %u\r\n", value_uint8);
             io_example[slot][index].data.value.u8 = value_uint8;
             break;
        case OPC_AI_STATUS:
             rc = ioThinx_AI_GetStatuss(slot, io_example[slot][index].ch, 1, &value_uint8);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_AI_GetEngs() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             //printf("\r\n=========> MX_Read() set value = %u\r\n", value_uint8);
             io_example[slot][index].data.value.u8 = value_uint8;
             break;
        case OPC_AI_RAWVALUE:
             rc = ioThinx_AI_GetRaws(slot, io_example[slot][index].ch, 1, &ai_raws);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_AI_GetEngs() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             //printf("\r\n=========> MX_Read() set value = %lu\r\n", ai_raws);
             io_example[slot][index].data.value.u32 = ai_raws;
             break;
        case OPC_AI_MINIMUMRAWVALUE:
             rc = ioThinx_AI_GetMinRaws(slot, io_example[slot][index].ch, 1, &ai_raws);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_AI_GetEngs() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             //printf("\r\n=========> MX_Read() set value = %lu\r\n", ai_raws);
             io_example[slot][index].data.value.u32 = ai_raws;
             break;
        case OPC_AI_MAXIMUMRAWVALUE:
             rc = ioThinx_AI_GetMaxRaws(slot, io_example[slot][index].ch, 1, &ai_raws);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_AI_GetEngs() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             //printf("\r\n=========> MX_Read() set value = %lu\r\n", ai_raws);
             io_example[slot][index].data.value.u32 = ai_raws;
             break;
        case OPC_AI_SCALEDVALUE:
             rc = ioThinx_AI_GetEngs(slot, io_example[slot][index].ch, 1, &value_float);
             if (rc != IOTHINX_ERR_OK)
             {
                 printf("ioThinx_AI_GetEngs() = %d\n", rc);
                 return MX_IO_CONTROL_ERR;
             }
             //printf("\r\n=========> MX_Read() set value = %f\r\n", value_float);
             io_example[slot][index].data.value.f = value_float;
             break;
        default:
              printf("IO function not found!\r\n");
              return MX_IO_CONTROL_ERR;
    }
    *node_value = io_example[slot][index].data;

    return MX_IO_CONTROL_OK;
}

int io_control_write(int slot, int index, const MX_NODE_VARIANT* node_value)
{
    FUNC_TYPE fn;
    int32_t value_int = 0;
    uint32_t do_values = 0;
    uint32_t pwm_starts = 0;
    uint32_t counter_starts = 0;
    uint32_t value_uint = 0;
    uint32_t relay_values = 0;
    int rc = 0;

    switch(io_example[slot][index].func_type){
        case OPC_DI_COUNTERSTATUS:
              if (node_value->type == MX_NODE_VALUE_TYPE_INT32) {
                  value_int = node_value->value.i32;
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              if (value_int != 0 && value_int != 1) {
                  return MX_IO_CONTROL_ERR;
              }
              rc = ioThinx_DI_GetCntStarts(slot, &counter_starts);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("\r\n ioThinx_DI_GetCntStarts() = %d\r\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              if (value_int == 1) {
                  counter_starts |= 0x1 << (io_example[slot][index].ch);
                  rc = ioThinx_DI_SetCntStarts(slot, counter_starts);
                  if (rc != IOTHINX_ERR_OK)
                  {
                      printf("ioThinx_DI_SetCntStarts() = %d\n", rc);
                      return MX_IO_CONTROL_ERR;
                  }
              }
              else if (value_int == 0) {
                  counter_starts |= 0x1 << (io_example[slot][index].ch);
                  rc = ioThinx_DI_SetCntStops(slot, counter_starts);
                  if (rc != IOTHINX_ERR_OK)
                  {
                      printf("ioThinx_DI_SetCntStops() = %d\n", rc);
                      return MX_IO_CONTROL_ERR;
                  }
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              break;
        case OPC_DI_COUNTERVALUE:
              if (node_value->type == MX_NODE_VALUE_TYPE_UINT32) {
                  value_uint = node_value->value.u32;
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              rc = ioThinx_DI_SetCntValues(slot, io_example[slot][index].ch, 1, &value_uint);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("ioThinx_DI_SetCntValues() = %d\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              break;
        case OPC_DO_STATUS:
              if (node_value->type == MX_NODE_VALUE_TYPE_INT32) {
                  value_int = node_value->value.i32;
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              if (value_int != 0 && value_int != 1) {
                  return MX_IO_CONTROL_ERR;
              }
              //printf("\r\n=========> MX_Write() set value = %d\r\n", value_int);
              rc = ioThinx_DO_GetValues(slot, &do_values);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("\r\nioThinx_DO_GetValues() = %d\r\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              do_values = do_values & ~(0x1 << io_example[slot][index].ch) | (value_int << io_example[slot][index].ch);
              //printf("do value = %lu\n", (unsigned long)do_values);
              rc = ioThinx_DO_SetValues(slot, do_values);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("ioThinx_DO_SetValues() = %d\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              //printf("\r\n******************* [%lu:%u] DO value = %x\r\n", do_slot, do_channel, (do_values >> do_channel) & 0x1);
              break;
        case OPC_DO_PULSESTATUS:
              if (node_value->type == MX_NODE_VALUE_TYPE_INT32) {
                  value_int = node_value->value.i32;
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              if (value_int != 0 && value_int != 1) {
                  return MX_IO_CONTROL_ERR;
              }
              rc = ioThinx_DO_GetPwmStarts(slot, &pwm_starts);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("\r\n ioThinx_DO_GetPwmStarts() = %d\r\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              if (value_int == 1) {
                  // start pwm
                  pwm_starts |= 0x1 << (io_example[slot][index].ch);
                  rc = ioThinx_DO_SetPwmStarts(slot, pwm_starts);
                  if (rc != IOTHINX_ERR_OK)
                  {
                      printf("ioThinx_DO_SetPwmStarts() = %d\n", rc);
                      return MX_IO_CONTROL_ERR;
                  }
              }
              else if (value_int == 0) {
                  // stop pwm
                  pwm_starts |= 0x1 << (io_example[slot][index].ch);
                  rc = ioThinx_DO_SetPwmStops(slot, pwm_starts);
                  if (rc != IOTHINX_ERR_OK)
                  {
                      printf("ioThinx_DO_SetPwmStops() = %d\n", rc);
                      return MX_IO_CONTROL_ERR;
                  }
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              break;
        case OPC_DO_PULSECOUNT:
              if (node_value->type == MX_NODE_VALUE_TYPE_UINT32) {
                  value_uint = node_value->value.u32;
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              rc = ioThinx_DO_SetPwmCounts(slot, io_example[slot][index].ch, 1, &value_uint);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("ioThinx_DO_Config_SetPwmCounts() = %d\n", rc);
              }
              break;
        case OPC_RELAY_VALUE:
              if (node_value->type == MX_NODE_VALUE_TYPE_INT32) {
                  value_int = node_value->value.i32;
              }
              else {
                  return MX_IO_CONTROL_ERR;
              }
              //printf("\r\n=========> MX_Write() set value = %d\r\n", value_int);
              if (value_int != 0 && value_int != 1) {
                  return MX_IO_CONTROL_ERR;
              }
              rc = ioThinx_Relay_GetValues(slot, &relay_values);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("\r\nioThinx_DO_GetValues() = %d\r\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              relay_values = relay_values & ~(0x1 << io_example[slot][index].ch) | (value_int << io_example[slot][index].ch);
              //printf("relay value = %lu\n", (unsigned long)relay_values);
              rc = ioThinx_Relay_SetValues(slot, relay_values);
              if (rc != IOTHINX_ERR_OK)
              {
                  printf("ioThinx_DO_SetValues() = %d\n", rc);
                  return MX_IO_CONTROL_ERR;
              }
              //printf("\r\n******************* [%lu:%u] DO value = %x\r\n", slot, io_example[slot][index].ch, (relay_values >> io_example[slot][index].ch) & 0x1);
              break;
        default:
              printf("function not found!\r\n");
    }
    io_example[slot][index].data = *node_value;
    return MX_IO_CONTROL_OK;
}
