/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Sample Node Operator
 *
 * Date          Author            Comment
 * 2019-06-05    Rich Liao         Created it.
 ******************************************************************************/

/**
    \file sample_node_operator.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Sample node operator</b>
    \date 2019-06-05
    \author Rich Liao
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is sample node operator code.
    </pre>
*/

#include <unistd.h>
#include <string.h>

/// include interfaces
#include <mx_node_sdk/mx_node_interface_basic.h>
#include <mx_node_sdk/mx_node_interface_data_access.h>

/// custom types
typedef enum _OPERATOR_STATE
{
    OPERATOR_STATE_STOP,
    OPERATOR_STATE_STOPPING,
    OPERATOR_STATE_RUNNING,
} OPERATOR_STATE;

typedef struct _MY_DATA
{
    MX_NODE_NODE_HANDLE node_handle;
    int data;
    struct timeval timestamp;
} MY_DATA;

/// custom function
struct timeval now()
{
    struct timeval t;
    gettimeofday(&t, NULL);
    return t;
}

/// custom variables
volatile OPERATOR_STATE g_state;
MY_DATA g_my_node;

//! \cond
/// interface basic
const char* mx_node_operator_get_version()
{
    return "1.0";
}

long long mx_node_operator_get_supported_interfaces()
{
    return INTERFACE_MX_NODE_BASIC | INTERFACE_MX_NODE_DATA_ACCESS;
}

MX_NODE_RESULT mx_node_operator_initialize(MX_NODE_NODE_OPERATOR_HANDLE operator_handle)
{
    MX_NODE_RESULT res;

    // Initialize state and data.
    g_state = OPERATOR_STATE_STOP;
    g_my_node.data = 0;

    // Create Nodes
    // Node SampleFolder
    MX_NODE_NODE folder;
    MX_NODE_NODE_HANDLE folder_handle;
    {
        // Fill in SampleFolder's information.
        strcpy(folder.node_name, "SampleFolder");
        folder.node_type = MX_NODE_NODE_TYPE_FOLDER;
        strcpy(folder.description, "SampleFolder's Description");

        // Add SampleFolder to OPC UA server.
        res = mx_node_add_node(operator_handle, &folder, &folder_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
    }

    // Node SampleInteger
    MX_NODE_NODE var;
    {
        // Fill in SampleInteger's information.
        strcpy(var.node_name, "SampleInteger");
        var.node_type = MX_NODE_NODE_TYPE_VARIABLE;
        strcpy(var.description, "SampleInteger's Description");
        var.variable.access_right = MX_NODE_ACCESS_RIGHT_READWRITE;
        var.variable.value.type = MX_NODE_VALUE_TYPE_INT32;
        var.variable.value.value.i32 = g_my_node.data;

        // Add SampleInteger to OPC UA server.
        res = mx_node_add_node(operator_handle, &var, &g_my_node.node_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;

        // Set SampleInteger's parent to SampleFolder.
        res = mx_node_set_parent_node(g_my_node.node_handle, folder_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
    }

    return MX_NODE_RESULT_GOOD;
}

void mx_node_operator_uninitialize()
{
}

void mx_node_operator_start()
{
    g_state = OPERATOR_STATE_RUNNING;
    while (g_state == OPERATOR_STATE_RUNNING)
    {
        g_my_node.data += 1;
        g_my_node.timestamp = now();
        MX_NODE_VARIANT var;
        {
            var.type = MX_NODE_VALUE_TYPE_INT32;
            var.value.i32 = g_my_node.data;
        }
        mx_node_update_node(g_my_node.node_handle, &var, &g_my_node.timestamp);
        sleep(3);
    }
    g_state = OPERATOR_STATE_STOP;
}

void mx_node_operator_stop()
{
    if (g_state == OPERATOR_STATE_RUNNING)
    {
        g_state = OPERATOR_STATE_STOPPING;
        while (g_state != OPERATOR_STATE_STOP);
    }
}
/// interface basic end

/// interface data_access
MX_NODE_RESULT mx_node_operator_read_node(MX_NODE_NODE_HANDLE node_handle, MX_NODE_VARIANT* node_value, struct timeval* node_timestamp)
{
    if (node_handle == g_my_node.node_handle)
    {
        node_value->type = MX_NODE_VALUE_TYPE_INT32;
        node_value->value.i32 = g_my_node.data;
        *node_timestamp = g_my_node.timestamp;
        return MX_NODE_RESULT_GOOD;
    }
    return MX_NODE_RESULT_BAD;
}

MX_NODE_RESULT mx_node_operator_write_node(MX_NODE_NODE_HANDLE node_handle, const MX_NODE_VARIANT* node_value)
{
    if (node_handle == g_my_node.node_handle && MX_NODE_VALUE_TYPE_INT32 == node_value->type)
    {
        g_my_node.data = node_value->value.i32;
        g_my_node.timestamp = now();
        return MX_NODE_RESULT_GOOD;
    }
    return MX_NODE_RESULT_BAD;
}
/// interface data_access end
//! \endcond
