/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Demo Node Operator
 *
 * Date          Author            Comment
 * 2019-06-05    Rich Liao         Created it.
 ******************************************************************************/

/**
    \file demo_node_operator.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Demo node operator</b>
    \date 2019-06-05
    \author Rich Liao
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is demo node operator code.
	</pre>
*/

#include <unistd.h>
#include <string.h>

/// include interfaces
#include <mx_node_sdk/mx_node_interface_basic.h>
#include <mx_node_sdk/mx_node_interface_data_access.h>

#define NUM_DATA 12

/// custom define
typedef enum
{
    OPERATOR_STATE_STOP,
    OPERATOR_STATE_STOPPING,
    OPERATOR_STATE_RUNNING,
} OPERATOR_STATE;

typedef struct _MY_DATA
{
    MX_NODE_NODE_HANDLE node_handle;
    MX_NODE_VARIANT data;
    struct timeval timestamp;
} MY_DATA;

struct timeval now();
MX_NODE_RESULT add_my_nodes();
void rotate(char* str);

MX_NODE_NODE_OPERATOR_HANDLE g_operator_handle;
volatile OPERATOR_STATE g_state;
MY_DATA g_my_data[NUM_DATA];

//! \cond
/// interface basic
const char* mx_node_operator_get_version()
{
    return "1.0";
}

long long mx_node_operator_get_supported_interfaces()
{
    return INTERFACE_MX_NODE_BASIC | INTERFACE_MX_NODE_DATA_ACCESS;
}

MX_NODE_RESULT mx_node_operator_initialize(MX_NODE_NODE_OPERATOR_HANDLE operator_handle)
{
    g_operator_handle = operator_handle;
    g_state = OPERATOR_STATE_STOP;

    for (int i = 0; i < NUM_DATA; ++i)
    {
        g_my_data[i].data.type = i;
        if (MX_NODE_VALUE_TYPE_STRING == g_my_data[i].data.type)
        {
            g_my_data[i].data.value.s = malloc(8);
            strcpy(g_my_data[i].data.value.s, "abcdefg");
        }
        else
            memset(&g_my_data[i].data.value, 0, sizeof(g_my_data[i].data.value));
    }

    return add_my_nodes();
}

void mx_node_operator_uninitialize()
{
    for (int i = 0; i < NUM_DATA; ++i)
    {
        if (MX_NODE_VALUE_TYPE_STRING == g_my_data[i].data.type)
            free(g_my_data[i].data.value.s);
    }
}

void mx_node_operator_start()
{
    g_state = OPERATOR_STATE_RUNNING;
    while (g_state == OPERATOR_STATE_RUNNING)
    {
        for (int i = 0; i < NUM_DATA; ++i)
        {
            g_my_data[i].timestamp = now();
            switch (g_my_data[i].data.type)
            {
            case MX_NODE_VALUE_TYPE_BOOLEAN:
                g_my_data[i].data.value.b = !g_my_data[i].data.value.b;
                break;
            case MX_NODE_VALUE_TYPE_SBYTE:
                g_my_data[i].data.value.i8++;
                break;
            case MX_NODE_VALUE_TYPE_INT16:
                g_my_data[i].data.value.i16++;
                break;
            case MX_NODE_VALUE_TYPE_INT32:
                g_my_data[i].data.value.i32++;
                break;
            case MX_NODE_VALUE_TYPE_INT64:
                g_my_data[i].data.value.i64++;
                break;
            case MX_NODE_VALUE_TYPE_BYTE:
                g_my_data[i].data.value.u8++;
                break;
            case MX_NODE_VALUE_TYPE_UINT16:
                g_my_data[i].data.value.u16++;
                break;
            case MX_NODE_VALUE_TYPE_UINT32:
                g_my_data[i].data.value.u32++;
                break;
            case MX_NODE_VALUE_TYPE_UINT64:
                g_my_data[i].data.value.u64++;
                break;
            case MX_NODE_VALUE_TYPE_FLOAT:
                g_my_data[i].data.value.f++;
                break;
            case MX_NODE_VALUE_TYPE_DOUBLE:
                g_my_data[i].data.value.d++;
                break;
            case MX_NODE_VALUE_TYPE_STRING:
                rotate(g_my_data[i].data.value.s);
                break;
            default:
                continue;
            }
            mx_node_update_node(g_my_data[i].node_handle, &g_my_data[i].data, &g_my_data[i].timestamp);
        }
        sleep(3);
    }
    g_state = OPERATOR_STATE_STOP;
}

void mx_node_operator_stop()
{
    if (g_state == OPERATOR_STATE_RUNNING)
    {
        g_state = OPERATOR_STATE_STOPPING;
        while (g_state != OPERATOR_STATE_STOP);
    }
}
/// interface basic end

/// interface data_access
MX_NODE_RESULT mx_node_operator_read_node(MX_NODE_NODE_HANDLE node_handle, MX_NODE_VARIANT* node_value, struct timeval* node_timestamp)
{
    for (int i = 0; i < NUM_DATA; ++i)
    {
        if (node_handle == g_my_data[i].node_handle)
        {
            *node_value = g_my_data[i].data;
            *node_timestamp = g_my_data[i].timestamp;
            return MX_NODE_RESULT_GOOD;
        }
    }
    return MX_NODE_RESULT_BAD;
}

MX_NODE_RESULT mx_node_operator_write_node(MX_NODE_NODE_HANDLE node_handle, const MX_NODE_VARIANT* node_value)
{
    for (int i = 0; i < NUM_DATA; ++i)
    {
        if (node_handle == g_my_data[i].node_handle)
        {
            g_my_data[i].data = *node_value;
            g_my_data[i].timestamp = now();
            return MX_NODE_RESULT_GOOD;
        }
    }
    return MX_NODE_RESULT_BAD;
}
/// interface data_access end
//! \endcond

MX_NODE_RESULT add_my_nodes()
{
    // DemoFolder
    //   |--DemoObject
    //        |--DemoBoolean
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoInt8
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoInt16
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoInt32
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoInt64
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoUInt8
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoUInt16
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoUInt32
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoUInt64
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoFloat
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoDouble
    //             |--DemoTimeZone (property, value: 480)
    //        |--DemoString
    //             |--DemoTimeZone (property, value: 480)

    MX_NODE_RESULT res;

    // folder
    MX_NODE_NODE folder;
    {
        strcpy(folder.node_name, "DemoFolder");
        folder.node_type = MX_NODE_NODE_TYPE_FOLDER;
        strcpy(folder.description, "DemoFolder's Description");
    }
    MX_NODE_NODE_HANDLE folder_handle;
    res = mx_node_add_node(g_operator_handle, &folder, &folder_handle);
    if (res != MX_NODE_RESULT_GOOD)
        return res;

    // object
    MX_NODE_NODE object;
    {
        strcpy(object.node_name, "DemoObject");
        object.node_type = MX_NODE_NODE_TYPE_OBJECT;
        strcpy(object.description, "DemoObject's Description");
    }
    MX_NODE_NODE_HANDLE object_handle;
    res = mx_node_add_node(g_operator_handle, &object, &object_handle);
    if (res != MX_NODE_RESULT_GOOD)
        return res;
    res = mx_node_set_parent_node(object_handle, folder_handle);
    if (res != MX_NODE_RESULT_GOOD)
        return res;

    // variable
    char* node_name[NUM_DATA] = {
        "DemoBoolean", "DemoInt8", "DemoInt16", "DemoInt32",
        "DemoInt64", "DemoUInt8", "DemoUInt16", "DemoUInt32",
        "DemoUInt64", "DemoFloat", "DemoDouble", "DemoString"
    };

    for (int i = 0; i < NUM_DATA; ++i)
    {
        MX_NODE_NODE variable;
        {
            strcpy(variable.node_name, node_name[i]);
            variable.node_type = MX_NODE_NODE_TYPE_VARIABLE;
            strcpy(variable.description, node_name[i]);
            strcat(variable.description, "'s Description");
            variable.variable.access_right = MX_NODE_ACCESS_RIGHT_READWRITE;
            variable.variable.value = g_my_data[i].data;
        }
        res = mx_node_add_node(g_operator_handle, &variable, &g_my_data[i].node_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
        res = mx_node_set_parent_node(g_my_data[i].node_handle, object_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
    }

    // property
    MX_NODE_NODE property;
    {
        strcpy(property.node_name, "DemoTimeZone");
        property.node_type = MX_NODE_NODE_TYPE_PROPERTY;
        strcpy(property.description, "DemoTimeZone's Description");
        property.property.access_right = MX_NODE_ACCESS_RIGHT_READONLY;
        property.property.value.type = MX_NODE_VALUE_TYPE_INT32;
        property.property.value.value.i32 = 480;
    }
    for (int i = 0; i < NUM_DATA; ++i)
    {
        MX_NODE_NODE_HANDLE property_handle;
        res = mx_node_add_node(g_operator_handle, &property, &property_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
        res = mx_node_set_parent_node(property_handle, g_my_data[i].node_handle);
        if (res != MX_NODE_RESULT_GOOD)
            return res;
    }

    return MX_NODE_RESULT_GOOD;
}

struct timeval now()
{
    struct timeval t;
    gettimeofday(&t, NULL);
    return t;
}

void rotate(char* str)
{
    if (str[0] != '\0')
    {
        char tmp = str[0];
        int i = 0;
        while (str[i + 1] != '\0')
        {
            str[i] = str[i + 1];
            ++i;
        }
        str[i] = tmp;
    }
}