'''
    Copyright (C) 2019 Moxa Inc. All rights reserved.
    SPDX-License-Identifier: Apache-2.0

    TC Calibration Python Sample Application

    Date          Author            Comment
    2019-02-11    William Chang     Created it.
'''

##
# @file tc_calibration.py
# @copyright 2019 Moxa Inc. All rights reserved.
# @date 2019-02-11
# @author William Chang
# @version V1.0
# @brief <b>TC Calibration Sample</b>
# @details <b>Library:</b><br>
# @link TC TC APIs @endlink
# @include tc_calibration.py

from ioThinx_4530 import ioThinx_4530_API
import argparse
from time import sleep

TC_SENSOR_TYPE_K = 1


def main():
    parser = argparse.ArgumentParser(description="TC calibration sample program.")
    parser.add_argument("-s", "--slot", dest="tc_slot", type=int, default=7)
    parser.add_argument("-c", "--channel", dest="tc_channel", type=int, default=0)
    parser.add_argument("-e", "--calibration_temperature", dest="tc_calibration_temperature", type=int, default=0)
    parser.add_argument("-t", "--type", dest="tc_type", type=int, default=TC_SENSOR_TYPE_K)
    args = parser.parse_args()

    tc_slot = args.tc_slot
    tc_channel = args.tc_channel
    tc_calibration_temperature = args.tc_calibration_temperature
    tc_type = args.tc_type
    print("TC slot =  {}".format(tc_slot))
    print("TC channel = {}".format(tc_channel))
    print("TC type =  {}".format(tc_type))
    print("calibration temperature = {}".format(tc_calibration_temperature))

    # initialize ioThinx I/O
    device = ioThinx_4530_API.ioThinx_4530_API()

    # temporarily set config
    device.ioThinx_TC_Config_SetSensorTypes(tc_slot, tc_channel, 1, [tc_type])

    # reload config
    device.ioThinx_IO_Config_Reload()
    print("After executing this program, a non-volatile offset will be set for the selected channel.")
    print("1. Ensure the sensor is connected.")
    print("2. Ensure the channel and its sensor type is correctly selected.")
    print("3. Put the sensor into a glass that contains a mixture of ice and water.")
    print("4. Do not remove the sensor from the ice water during calibration...")
    ch = input("Continue ? (y/n): ")

    if ch == 'y':
        # get TC value
        tc_value = device.ioThinx_TC_GetValues(tc_slot, tc_channel, 1)
        print("TC value before calibration: {}".format(tc_value))
        # execute the API with zero degrees Celsius as input parameter
        device.ioThinx_TC_SetCalibrations(tc_slot, tc_channel, 1, [tc_calibration_temperature])
        # wait for calibration ok
        print("Calibrating...")
        sleep(3)
        # get TC value
        tc_value = device.ioThinx_TC_GetValues(tc_slot, tc_channel, 1)
        print("TC value after calibration: {}".format(tc_value))
        print("Finish.")
    else:
        print("Abort.")


if __name__ == '__main__':
    main()
