'''
    Copyright (C) 2019 Moxa Inc. All rights reserved.
    SPDX-License-Identifier: Apache-2.0

    RTD Calibration Python Sample Application

    Date          Author            Comment
    2019-02-11    William Chang     Created it.
'''

##
# @file rtd_calibration.py
# @copyright 2019 Moxa Inc. All rights reserved.
# @date 2019-02-11
# @author William Chang
# @version V1.0
# @brief <b>RTD Calibration Sample</b>
# @details <b>Library:</b><br>
# @link RTD RTD APIs @endlink
# @include rtd_calibration.py

from ioThinx_4530 import ioThinx_4530_API
import argparse
from time import sleep

RTD_SENSOR_TYPE_PT100 = 1


def main():
    parser = argparse.ArgumentParser(description="RTD calibration sample program.")
    parser.add_argument("-s", "--slot", dest="rtd_slot", type=int, default=8)
    parser.add_argument("-c", "--channel", dest="rtd_channel", type=int, default=0)
    parser.add_argument("-e", "--calibration_temperature", dest="rtd_calibration_temperature", type=int, default=0)
    parser.add_argument("-t", "--type", dest="rtd_type", type=int, default=RTD_SENSOR_TYPE_PT100)
    args = parser.parse_args()

    rtd_slot = args.rtd_slot
    rtd_channel = args.rtd_channel
    rtd_calibration_temperature = args.rtd_calibration_temperature
    rtd_type = args.rtd_type
    print("RTD slot =  {}".format(rtd_slot))
    print("RTD channel = {}".format(rtd_channel))
    print("RTD type =  {}".format(rtd_type))
    print("calibration temperature = {}".format(rtd_calibration_temperature))

    # initialize ioThinx I/O
    device = ioThinx_4530_API.ioThinx_4530_API()
    # temporarily set config
    device.ioThinx_RTD_Config_SetSensorTypes(rtd_slot, rtd_channel, 1, [rtd_type])
    # reload config
    device.ioThinx_IO_Config_Reload()

    print("After executing this program, a non-volatile offset will be set for the selected channel.")
    print("1. Ensure the sensor is connected.")
    print("2. Ensure the channel and its sensor type is correctly selected.")
    print("3. Put the sensor into a glass that contains a mixture of ice and water.")
    print("4. Do not remove the sensor from the ice water during calibration...")
    ch = input("Continue ? (y/n): ")
    if ch == 'y':
        # get RTD value
        rtd_value = device.ioThinx_RTD_GetValues(rtd_slot, rtd_channel, 1)
        print("RTD value before calibration: {}".format(rtd_value))
        # execute the API with zero degrees Celsius as input parameter
        device.ioThinx_RTD_SetCalibrations(rtd_slot, rtd_channel, 1, [rtd_calibration_temperature])
        # wait for calibration ok
        print("Calibrating...")
        sleep(3)
        # get RTD value
        rtd_value = device.ioThinx_RTD_GetValues(rtd_slot, rtd_channel, 1)
        print("RTD value after calibration: {}".format(rtd_value))
        print("Finish.")
    else:
        print("Abort.")


if __name__ == '__main__':
    main()
