/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Scheduling Jitter Sample Application
 *
 * Date          Author            Comment
 * 2019-04-25    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file scheduling_jitter.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Scheduling Jitter Sample</b>
    \date 2019-04-25
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is Scheduling jitter sample code.
    <b>Example:</b>
    1. Using default: ./scheduling_jitter
    2. Setting test count and cycle time: ./scheduling_jitter -n100 -t100000
    3. Setting process priority: ./scheduling_jitter -p3
    4. Setting DO slot and channel: ./scheduling_jitter -s3 -c2
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./scheduling_jitter -h
    Scheduling jitter sample program.

    Usage: ./scheduling_jitter [OPTIONS]

    Options:
         -p       Priority level[0-3]. Default level = 3
         -s       Slot of DO module. Default slot = 2
         -c       DO channel. Default channel = 0
         -n       Testing loop count. Default count = 1000
         -t       Testing loop cycle time (in us). Default time = 10000 us
    \endverbatim
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <pthread.h>
#include <errno.h>
#include <sys/time.h>
#include <time.h>
#include <iothinx/iothinxio.h>
#include <sched.h>

int set_process_priority(int pid, int priority_level)
{
    int rc = 0;
    int policy;
    struct sched_param param = { 0 };

    switch (priority_level)
    {
    case 1: // medium
        policy = SCHED_RR;
        param.sched_priority = 1;
        break;
    case 2: // high
        policy = SCHED_RR;
        param.sched_priority = 49;
        break;
    case 3: // max
        policy = SCHED_RR;
        param.sched_priority = 99;
        break;
    case 0: // low
    default:
        policy = SCHED_OTHER;
        param.sched_priority = 0;
        break;
    }
    rc = sched_setscheduler(pid, policy, &param);
    if (rc != 0)
    {
        perror("sched_setscheduler");
    }
    return rc;
}

/// \include scheduling_jitter.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t do_slot = 2;
    uint8_t do_channel = 0;
    uint8_t do_mode = DO_MODE_DO;
    uint32_t do_values = 0;
    uint32_t test_cnt = 1000;
    uint32_t cycle_time_us = 10000;
    uint32_t i;
    time_t start_time_us, end_time_us;
    time_t last_start_time_us = 0;
    int priority_level = 3;
    int32_t diff_us;
    int32_t sum_us = 0;
    int32_t max_us = 0;
    struct timespec ts;

    while (-1 != (rc = getopt(argc, argv, "c:hn:p:s:t:")))
    {
        switch (rc)
        {
        case 'c':
            do_channel = atoi(optarg);
            break;
        case 'n':
            test_cnt = atoi(optarg);
            break;
        case 'p':
            priority_level = atoi(optarg);
            break;
        case 's':
            do_slot = atoi(optarg);
            break;
        case 't':
            cycle_time_us = atoi(optarg);
            break;
        case 'h':
        default:
            printf("Scheduling jitter sample program.\n\n");
            printf("Usage: ./scheduling_jitter [OPTIONS]\n\n");
            printf("Options: \n");
            printf("\t %-8s Priority level[0-3]. Default level = %d\n", "-p", priority_level);
            printf("\t %-8s Slot of DO module. Default slot = %d\n", "-s", do_slot);
            printf("\t %-8s DO channel. Default channel = %d\n", "-c", do_channel);
            printf("\t %-8s Testing loop count. Default count = %d\n", "-n", test_cnt);
            printf("\t %-8s Testing loop cycle time (in us). Default time = %d us\n", "-t", cycle_time_us);
            printf("\n");
            return 0;
        }
    }

    printf("Priority level = %d\n", priority_level);
    printf("DO slot = %lu\n", do_slot);
    printf("DO channel = %u\n", do_channel);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    rc = ioThinx_DO_Config_SetModes(do_slot, do_channel, 1, &do_mode);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_Config_SetModes() = %d\n", rc);
        return -1;
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
        return -1;
    }

    // set process priority
    set_process_priority(0, priority_level);
    if (rc != 0)
    {
        return -1;
    }

    printf("Start testing ...\n");

    for (i = 0; i < test_cnt; i++)
    {
        // get start time
        clock_gettime(CLOCK_MONOTONIC, &ts);
        start_time_us = (ts.tv_sec * 1000000 + ts.tv_nsec / 1000);
        diff_us = (last_start_time_us == 0) ? 0 : start_time_us - last_start_time_us - cycle_time_us;
        sum_us += diff_us;
        max_us = (diff_us > max_us) ? diff_us : max_us;
        last_start_time_us = start_time_us;

        // do the stuff ...
        // invert do value
        if (do_values & (0x1 << do_channel))
        {
            do_values &= ~(0x1 << do_channel);
        }
        else
        {
            do_values |= 0x1 << do_channel;
        }
        rc = ioThinx_DO_SetValues(do_slot, do_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_DO_SetValues() = %d\n", rc);
            break;
        }

        // get end time
        clock_gettime(CLOCK_MONOTONIC, &ts);
        end_time_us = (ts.tv_sec * 1000000 + ts.tv_nsec / 1000);

        // calculate sleep time
        diff_us = cycle_time_us - (end_time_us - start_time_us);
        ts.tv_sec = 0;
        ts.tv_nsec = diff_us * 1000;

        // sleep
        clock_nanosleep(CLOCK_MONOTONIC, 0, &ts, NULL);
    }

    printf("Done.\n");
    printf("Scheduling jitter: Avg. = %.1f us, Max = %ld us\n",
           (float)sum_us / (float)(test_cnt - 1), max_us);

    return 0;
}
