/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Miscellaneous Sample Application
 *
 * Date          Author            Comment
 * 2019-01-22    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file misc.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Miscellaneous Sample</b>
    \date 2019-01-22
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is misc sample code.
    <b>Example:</b>
    1. Using default: ./misc
    2. Setting slot: ./misc -s5
    <b>Default:</b>
    Slot of module = 1
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./misc -h
    Miscellaneous sample program.

    Usage: ./misc [OPTIONS]

    Options:
            -s       Slot of the module. Default slot = 1
    \endverbatim
    \b Library: \n
    \link Misc Misc APIs\endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include misc.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t misc_slot = 0;
    uint32_t module_slot = 1;
    uint32_t module_count = 0;
    struct MODULE_INFO module_info;
    uint8_t rs_state = 0;
    uint8_t pbtn_state = 0;
    uint8_t led_state = 0;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "hs:")))
    {
        switch (rc)
        {
        case 's':
            module_slot = atoi(optarg);
            break;
        case 'h':
        default:
            printf("Miscellaneous sample program.\n\n");
            printf("Usage: ./misc [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of the module. Default slot = %d\n", "-s", module_slot);
            printf("\n");
            return 0;
        }
    }

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // module infomation
    rc = ioThinx_Misc_GetModuleCount(&module_count);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_GetModuleCount() = %d\n", rc);
    }
    printf("Module count = %lu\n", module_count);
    printf("Module slot = %lu\n", module_slot);
    rc = ioThinx_Misc_GetModuleInfo(module_slot, &module_info);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_GetModuleInfo() = %d\n", rc);
    }
    else
    {
        printf("Slot %d Module Information:\n", module_slot);
        printf("Model Name: %s, ", module_info.model_name);
        printf("Serial Number: %s\n", module_info.serial_number);
    }

    // locating
    rc = ioThinx_Misc_SetLocateState(module_slot, 1);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_SetLocateState() = %d\n", rc);
    }
    else
    {
        printf("Slot %d: Locating...\n", module_slot);
    }
    printf("Press enter to stop locate.");
    ch = getchar();
    rc = ioThinx_Misc_SetLocateState(module_slot, 0);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_SetLocateState() = %d\n", rc);
    }

    // rotary switch
    ioThinx_Misc_GetRotarySwitchState(misc_slot, &rs_state);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_GetRotarySwitchState() = %d\n", rc);
    }
    else
    {
        printf("Rotary switch state = %d\n", rs_state);
    }

    // push button
    ioThinx_Misc_GetPushButtonState(misc_slot, &pbtn_state);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_GetPushButtonState() = %d\n", rc);
    }
    else
    {
        printf("Push button state = %d\n", pbtn_state);
    }

    // user led
    ioThinx_Misc_SetUserLedState(misc_slot, LED_CHANNEL_U1, LED_STATE_GREEN);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_SetUserLedState() = %d\n", rc);
    }
    else
    {
        printf("Set LED U1 to GREEN\n");
    }
    ioThinx_Misc_SetUserLedState(misc_slot, LED_CHANNEL_U2, LED_STATE_RED);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Misc_SetUserLedState() = %d\n", rc);
    }
    else
    {
        printf("Set LED U2 to RED\n");
    }
    printf("Press enter to clear.");
    ch = getchar();
    ioThinx_Misc_SetUserLedState(misc_slot, LED_CHANNEL_U1, LED_STATE_DARK);
    ioThinx_Misc_SetUserLedState(misc_slot, LED_CHANNEL_U2, LED_STATE_DARK);
    printf("Press enter to continue.");
    ch = getchar();

    return 0;
}
