/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * TC Calibration Sample Application
 *
 * Date          Author            Comment
 * 2019-01-04    Eddy Chiu         Created.
 ******************************************************************************/

/**
    \file tc_calibration.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>TC Calibration Sample</b>
    \date 2019-01-04
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is TC calibration sample code.
    <b>Example:</b>
    1. Using default: ./tc_calibration
    2. Setting TC slot, channel, type and temperature: ./tc_calibration -s7 -c2 -t2 -e36.5
    <b>Default:</b>
    Slot of TC module = 6
    TC channel = 0
    TC type = K type
    Calibration temperature = 0 degrees Celsius
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./tc_calibration -h
    TC Calibration sample program.

    Usage: ./tc_calibration [OPTIONS]

    Options:
        -s      Slot of TC module. Default slot = 6
        -c      Channel on TC module. Default channel = 0
        -t      TC type. Default type = 1
        -e      Temperature in degrees Celsius. Default value = 0.000
    \endverbatim
    \b Library: \n
    \link TC TC APIs\endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include tc_calibration.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t tc_slot = 6;
    uint8_t tc_channel = 0;
    uint8_t tc_type = TC_SENSOR_TYPE_K;
    float tc_value;
    float tc_calibration_temperature = 0.0f;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:e:hs:t:")))
    {
        switch (rc)
        {
        case 'c':
            tc_channel = atoi(optarg);
            break;
        case 'e':
            tc_calibration_temperature = atof(optarg);
            break;
        case 's':
            tc_slot = atoi(optarg);
            break;
        case 't':
            tc_type = atoi(optarg);
            break;
        case 'h':
        default:
            printf("TC calibration sample program.\n\n");
            printf("Usage: ./tc_calibration [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of TC module. Default slot = %d\n", "-s", tc_slot);
            printf("\t%-8s TC channel. Default channel = %d\n", "-c", tc_channel);
            printf("\t%-8s TC type. Default type = %d\n", "-t", tc_type);
            printf("\t%-8s Setting temperature. Default temperature = %.3f\n", "-e", tc_calibration_temperature);
            printf("\n");
            return 0;
        }
    }

    printf("TC slot = %lu\n", tc_slot);
    printf("TC channel = %u\n", tc_channel);
    printf("TC type = %d\n", tc_type);
    printf("calibration temperature = %.3f\n", tc_calibration_temperature);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    rc = ioThinx_TC_Config_SetSensorTypes(tc_slot, tc_channel, 1, &tc_type);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_TC_Config_SetSensorTypes() = %d\n", rc);
        return -1;
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    printf("\nAfter executing this program, a non-volatile offset will be set for the selected channel.\n");
    printf("1. Ensure the sensor is connected.\n");
    printf("2. Ensure the channel and its sensor type is correctly selected.\n");
    printf("3. Put the sensor into a glass that contains a mixture of ice and water.\n");
    printf("4. Do not remove the sensor from the ice water during calibration...\n");
    printf("Continue ? (y/n): ");
    ch = getchar();
    if (ch == 'y')
    {
        // get TC value
        rc = ioThinx_TC_GetValues(tc_slot, tc_channel, 1, &tc_value);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_TC_GetValues() = %d\n", rc);
        }
        else
        {
            printf("TC value before calibration: %.3f\n", tc_value);
        }

        // execute the API with zero degrees Celsius as input parameter
        rc = ioThinx_TC_SetCalibrations(tc_slot, tc_channel, 1, &tc_calibration_temperature);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_TC_SetCalibrations() = %d\n", rc);
        }

        // wait for calibration ok
        printf("Calibrating...\n");
        sleep(3);

        // get TC value
        rc = ioThinx_TC_GetValues(tc_slot, tc_channel, 1, &tc_value);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_TC_GetValues() = %d\n", rc);
        }
        else
        {
            printf("TC value after calibration: %.3f\n", tc_value);
        }
        printf("Finish.\n");
    }
    else
    {
        printf("Abort.\n");
    }

    return 0;
}
