/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * TC Sample Application
 *
 * Date          Author            Comment
 * 2018-12-27    Eddy Chiu         Created it.
 ******************************************************************************/

/**
    \file tc.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>TC Sample</b>
    \date 2018-12-27
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is TC sample code.
    <b>Example:</b>
    1. Using default: ./tc
    2. Setting TC slot and channel: ./tc -s7 -c2 -n3
    <b>Default:</b>
    Slot of TC module = 1
    TC channel = 0
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./tc -h
    TC sample program.

    Usage: ./tc [OPTIONS]

    Options:
        -s       Slot of TC module. Default slot = 6
        -c       TC start channel. Default channel = 0
        -n       TC channel count. Default count = 2
    \endverbatim
    \b Library: \n
    \link TC TC APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include tc.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t tc_slot = 6;
    uint8_t tc_channel_start = 0;
    uint8_t tc_channel_count = 2;
    uint8_t tc_types[8];
    float tc_values[8];
    uint8_t tc_statuss[8];
    int32_t i;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:n:hs:")))
    {
        switch (rc)
        {
        case 's':
            tc_slot = atoi(optarg);
            break;
        case 'c':
            tc_channel_start = atoi(optarg);
            break;
        case 'n':
            tc_channel_count = atoi(optarg);
            break;
        case 'h':
        default:
            printf("TC sample program.\n\n");
            printf("Usage: ./tc [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of TC module. Default slot = %d\n", "-s", tc_slot);
            printf("\t%-8s TC start channel. Default channel = %d\n", "-c", tc_channel_start);
            printf("\t%-8s TC channel count. Default count = %d\n", "-n", tc_channel_count);
            printf("\n");
            return 0;
        }
    }

    printf("TC slot = %lu\n", tc_slot);
    printf("TC start channel = %u\n", tc_channel_start);
    printf("TC channel count = %u\n", tc_channel_count);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    for (i = 0; i < tc_channel_count; i++)
    {
        tc_types[i] = TC_SENSOR_TYPE_K;
    }
    rc = ioThinx_TC_Config_SetSensorTypes(tc_slot, tc_channel_start, tc_channel_count, tc_types);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_TC_Config_SetSensorTypes() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    // get value
    do
    {
        rc = ioThinx_TC_GetValues(tc_slot, tc_channel_start, tc_channel_count, tc_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_TC_GetValues() = %d\n", rc);
            break;
        }
        rc = ioThinx_TC_GetStatuss(tc_slot, tc_channel_start, tc_channel_count, tc_statuss);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_TC_GetStatuss() = %d\n", rc);
            break;
        }
        for (i = 0; i < tc_channel_count; i++)
        {
            printf("[%lu:%u] value = %.2f, status = %d\n", tc_slot, tc_channel_start+i, tc_values[i], tc_statuss[i]);
        }
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
