/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * RTD Calibration Sample Application
 *
 * Date          Author            Comment
 * 2019-01-04    Eddy Chiu         Created.
 ******************************************************************************/

/**
    \file rtd_calibration.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>RTD Calibration Sample</b>
    \date 2019-01-04
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is RTD calibration sample code.
    <b>Example:</b>
    1. Using default: ./rtd_calibration
    2. Setting RTD slot, channel, type and temperature: ./rtd_calibration -s6 -c2 -t2 -e36.5
    <b>Default:</b>
    Slot of RTD module = 5
    RTD channel = 0
    RTD type = PT100
    Calibration temperature = 0 degrees Celsius
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./rtd_calibration -h
    RTD Calibration sample program.

    Usage: ./rtd_calibration [OPTIONS]

    Options:
        -s      Slot of RTD module. Default slot = 5
        -c      Channel on RTD module. Default channel = 0
        -t      RTD type. Default type = 1
        -e      Temperature in degrees Celsius. Default value = 0.000
    \endverbatim
    \b Library: \n
    \link RTD RTD APIs\endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include rtd_calibration.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t rtd_slot = 5;
    uint8_t rtd_channel = 0;
    uint8_t rtd_type = RTD_SENSOR_TYPE_PT100;
    float rtd_value;
    float rtd_calibration_temperature = 0.0f;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:e:hs:t:")))
    {
        switch (rc)
        {
        case 'c':
            rtd_channel = atoi(optarg);
            break;
        case 'e':
            rtd_calibration_temperature = atof(optarg);
            break;
        case 's':
            rtd_slot = atoi(optarg);
            break;
        case 't':
            rtd_type = atoi(optarg);
            break;
        case 'h':
        default:
            printf("RTD calibration sample program.\n\n");
            printf("Usage: ./rtd_calibration [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of RTD module. Default slot = %d\n", "-s", rtd_slot);
            printf("\t%-8s RTD channel. Default channel = %d\n", "-c", rtd_channel);
            printf("\t%-8s RTD type. Default type = %d\n", "-t", rtd_type);
            printf("\t%-8s Setting temperature. Default temperature = %.3f\n", "-e", rtd_calibration_temperature);
            printf("\n");
            return 0;
        }
    }

    printf("RTD slot = %lu\n", rtd_slot);
    printf("RTD channel = %u\n", rtd_channel);
    printf("RTD type = %d\n", rtd_type);
    printf("calibration temperature = %.3f\n", rtd_calibration_temperature);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    rc = ioThinx_RTD_Config_SetSensorTypes(rtd_slot, rtd_channel, 1, &rtd_type);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_RTD_Config_SetSensorTypes() = %d\n", rc);
        return -1;
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    printf("\nAfter executing this program, a non-volatile offset will be set for the selected channel.\n");
    printf("1. Ensure the sensor is connected.\n");
    printf("2. Ensure the channel and its sensor type is correctly selected.\n");
    printf("3. Put the sensor into a glass that contains a mixture of ice and water.\n");
    printf("4. Do not remove the sensor from the ice water during calibration...\n");
    printf("Continue ? (y/n): ");
    ch = getchar();
    if (ch == 'y')
    {
        // get RTD value
        rc = ioThinx_RTD_GetValues(rtd_slot, rtd_channel, 1, &rtd_value);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_RTD_GetValues() = %d\n", rc);
        }
        else
        {
            printf("RTD value before calibration: %.3f\n", rtd_value);
        }

        // execute the API with zero degrees Celsius as input parameter
        rc = ioThinx_RTD_SetCalibrations(rtd_slot, rtd_channel, 1, &rtd_calibration_temperature);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_RTD_SetCalibrations() = %d\n", rc);
        }

        // wait for calibration ok
        printf("Calibrating...\n");
        sleep(3);

        // get RTD value
        rc = ioThinx_RTD_GetValues(rtd_slot, rtd_channel, 1, &rtd_value);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_RTD_GetValues() = %d\n", rc);
        }
        else
        {
            printf("RTD value after calibration: %.3f\n", rtd_value);
        }
        printf("Finish.\n");
    }
    else
    {
        printf("Abort.\n");
    }

    return 0;
}

