/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * RTD Sample Application
 *
 * Date          Author            Comment
 * 2018-12-27    Eddy Chiu         Created it.
 ******************************************************************************/

/**
    \file rtd.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>RTD Sample</b>
    \date 2018-12-27
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is RTD sample code.
    <b>Example:</b>
    1. Using default: ./rtd
    2. Setting RTD slot and channel: ./rtd -s6 -c2 -n3
    <b>Default:</b>
    Slot of RTD module = 1
    RTD channel = 0
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./rtd -h
    RTD sample program.

    Usage: ./rtd [OPTIONS]

    Options:
        -s       Slot of RTD module. Default slot = 5
        -c       RTD start channel. Default channel = 0
        -n       RTD channel count. Default count = 2
    \endverbatim
    \b Library: \n
    \link RTD RTD APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include rtd.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t rtd_slot = 5;
    uint8_t rtd_channel_start = 0;
    uint8_t rtd_channel_count = 2;
    uint8_t rtd_types[8];
    float rtd_values[8];
    uint8_t rtd_statuss[8];
    int32_t i;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:n:hs:")))
    {
        switch (rc)
        {
        case 's':
            rtd_slot = atoi(optarg);
            break;
        case 'c':
            rtd_channel_start = atoi(optarg);
            break;
        case 'n':
            rtd_channel_count = atoi(optarg);
            break;
        case 'h':
        default:
            printf("RTD sample program.\n\n");
            printf("Usage: ./rtd [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of RTD module. Default slot = %d\n", "-s", rtd_slot);
            printf("\t%-8s RTD start channel. Default channel = %d\n", "-c", rtd_channel_start);
            printf("\t%-8s RTD channel count. Default count = %d\n", "-n", rtd_channel_count);
            printf("\n");
            return 0;
        }
    }

    printf("RTD slot = %lu\n", rtd_slot);
    printf("RTD start channel = %u\n", rtd_channel_start);
    printf("RTD channel count = %u\n", rtd_channel_count);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    for (i = 0; i < rtd_channel_count; i++)
    {
        rtd_types[i] = RTD_SENSOR_TYPE_PT100;
    }
    rc = ioThinx_RTD_Config_SetSensorTypes(rtd_slot, rtd_channel_start, rtd_channel_count, rtd_types);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_RTD_Config_SetSensorTypes() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    // get value
    do
    {
        rc = ioThinx_RTD_GetValues(rtd_slot, rtd_channel_start, rtd_channel_count, rtd_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_RTD_GetValues() = %d\n", rc);
            break;
        }
        rc = ioThinx_RTD_GetStatuss(rtd_slot, rtd_channel_start, rtd_channel_count, rtd_statuss);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_RTD_GetStatuss() = %d\n", rc);
            break;
        }
        for (i = 0; i < rtd_channel_count; i++)
        {
            printf("[%lu:%u] value = %.2f, status = %d\n", rtd_slot, rtd_channel_start+i, rtd_values[i], rtd_statuss[i]);
        }
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}

