/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Relay Sample Application
 *
 * Date          Author            Comment
 * 2019-01-17    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file relay.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Relay Sample</b>
    \date 2019-01-17
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is relay sample code.
    <b>Example:</b>
    1. Using default: ./relay
    2. Setting Relay slot and channel: ./relay -s4 -c2
    <b>Default:</b>
    Slot of relay module = 3
    Relay channel = 0
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./relay -h
    Relay sample program.

    Usage: ./relay [OPTIONS]

    Options:
            -s       Slot of relay module. Default slot = 3
            -c       Relay channel. Default channel = 0
    \endverbatim
    \b Library: \n
    \link Relay Relay APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include relay.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t relay_slot = 3;
    uint8_t relay_channel = 0;
    uint32_t relay_values = 0;
    uint32_t relay_count = 0;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:hs:")))
    {
        switch (rc)
        {
        case 's':
            relay_slot = atoi(optarg);
            break;
        case 'c':
            relay_channel = atoi(optarg);
            break;
        case 'h':
        default:
            printf("Relay sample program.\n\n");
            printf("Usage: ./relay [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of relay module. Default slot = %d\n", "-s", relay_slot);
            printf("\t%-8s Relay channel. Default channel = %d\n", "-c", relay_channel);
            printf("\n");
            return 0;
        }
    }

    printf("Relay slot = %lu\n", relay_slot);
    printf("Relay channel = %u\n", relay_channel);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    rc = ioThinx_Relay_GetValues(relay_slot, &relay_values);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_Relay_GetValues() = %d\n", rc);
    }

    // set value
    do
    {
        // invert relay value
        if (relay_values & (0x1 << relay_channel))
        {
            relay_values &= ~(0x1 << relay_channel);
        }
        else
        {
            relay_values |= 0x1 << relay_channel;
        }

        rc = ioThinx_Relay_SetValues(relay_slot, relay_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_DO_GetValues() = %d\n", rc);
        }
        printf("[%lu:%u] relay value = %x\n", relay_slot, relay_channel, (relay_values >> relay_channel) & 0x1);

        rc = ioThinx_Relay_GetCurrentCounts(relay_slot, relay_channel, 1, &relay_count);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_Relay_GetCurrentCounts() = %d\n", rc);
        }
        printf("[%lu:%u] relay count = %lu\n", relay_slot, relay_channel, relay_count);
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
