/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Power Diagnostic Sample Application
 *
 * Date          Author            Comment
 * 2019-01-23    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file pwr.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Power Diagnostic Sample</b>
    \date 2019-01-23
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is power diagnostic sample code.
    <b>Example:</b>
    1. Using default: ./pwr
    2. Setting power slot and field power alarm: ./pwr -s8 -a12.3
    <b>Default:</b>
    Slot of pwr module = 1
    Diagnostic config = 0
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./pwr -h
    Power diagnostic sample program.

    Usage: ./pwr [OPTIONS]

    Options:
            -s       Slot of power module. Default slot = 7
            -a       Field power alarm setting(voltage). Default setting = 20.0
    \endverbatim
    \b Library: \n
    \link Power Power APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include pwr.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t pwr_slot = 7;
    uint8_t sys_channel = 0;
    uint8_t field_channel = 0;
    float alarm_voltage = 20.0f;
    uint8_t sys_status = 0;
    uint8_t field_status = 0;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "a:hs:")))
    {
        switch (rc)
        {
        case 'a':
            alarm_voltage = atof(optarg);
            break;
        case 's':
            pwr_slot = atoi(optarg);
            break;
        case 'h':
        default:
            printf("Power diagnostic sample program.\n\n");
            printf("Usage: ./pwr [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of power module. Default slot = %d\n", "-s", pwr_slot);
            printf("\t%-8s Field power alarm setting(voltage). Default setting = %.1f\n", "-a", alarm_voltage);
            printf("\n");
            return 0;
        }
    }

    printf("Power slot = %lu\n", pwr_slot);
    printf("Alarm setting = %.1f\n", alarm_voltage);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    rc = ioThinx_PWR_Config_SetAlarms(pwr_slot, field_channel, 1, &alarm_voltage);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_PWR_Config_SetAlarms() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    do
    {
        rc = ioThinx_PWR_GetSysStatus(pwr_slot, sys_channel, 1, &sys_status);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_PWR_GetSysStatus() = %d\n", rc);
            break;
        }
        rc = ioThinx_PWR_GetFieldStatus(pwr_slot, field_channel, 1, &field_status);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_PWR_GetFieldStatus() = %d\n", rc);
            break;
        }
        printf("[%lu] system power status = %u\n", pwr_slot, sys_status);
        printf("[%lu] filed power status = %u\n", pwr_slot, field_status);
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
