/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * PWM Sample Application
 *
 * Date          Author            Comment
 * 2018-12-25    Eddy Chiu         Created it.
 ******************************************************************************/

/**
    \file pwm.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>PWM Sample</b>
    \date 2018-12-25
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is PWM sample code.
    <b>Example:</b>
    1. Using default: ./pwm
    2. Setting DO slot and PWM channel: ./pwm -s3 -c3 -n1
    <b>Default:</b>
    Slot of DO module = 2
    PWM start channel = 0
    PWM channel count = 2
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./pwm -h
    PWM sample program.

    Usage: ./pwm [OPTIONS]

    Options:
            -s       Slot of DO module. Default slot = 2
            -c       PWM start channel. Default channel = 0
            -n       PWM channel count. Default count = 2
    \endverbatim
    \b Library: \n
    \link DO DO APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include pwm.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t do_slot = 2;
    uint8_t do_modes[16];
    uint8_t pwm_channel_start = 0;
    uint8_t pwm_channel_count = 2;
    uint32_t pwm_starts = 0;
    uint16_t pwm_freqs[4];
    uint16_t pwm_dutys[4];
    uint32_t pwm_counts[4];
    int32_t i;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:hn:s:")))
    {
        switch (rc)
        {
        case 'c':
            pwm_channel_start = atoi(optarg);
            break;
        case 'n':
            pwm_channel_count = atoi(optarg);
            break;        
        case 's':
            do_slot = atoi(optarg);
            break;
        case 'h':
        default:
            printf("PWM sample program.\n\n");
            printf("Usage: ./pwm [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of DO module. Default slot = %d\n", "-s", do_slot);
            printf("\t%-8s PWM start channel. Default channel = %d\n", "-c", pwm_channel_start);
            printf("\t%-8s PWM channel count. Default count = %d\n", "-n", pwm_channel_count);
            printf("\n");
            return 0;
        }
    }

    printf("DO slot = %lu\n", do_slot);
    printf("PWM start channel = %u\n", pwm_channel_start);
    printf("PWM channel count = %u\n", pwm_channel_count);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    for (i = 0; i < pwm_channel_count; i++)
    {
        do_modes[i] = DO_MODE_PWM;
    }
    rc = ioThinx_DO_Config_SetModes(do_slot, pwm_channel_start, pwm_channel_count, do_modes);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_Config_SetModes() = %d\n", rc);
    }

    for (i = 0; i < pwm_channel_count; i++)
    {
        pwm_counts[i] = 0; // infinite
    }
    rc = ioThinx_DO_Config_SetPwmCounts(do_slot, pwm_channel_start, pwm_channel_count, pwm_counts);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_Config_SetPwmCounts() = %d\n", rc);
    }

    for (i = 0; i < pwm_channel_count; i++)
    {
        pwm_freqs[i] = 5;
        pwm_dutys[i] = 50;
    }
    rc = ioThinx_DO_Config_SetPwmConfigures(do_slot, pwm_channel_start, pwm_channel_count, pwm_freqs, pwm_dutys);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_Config_SetPwmConfigures() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    // start pwm
    pwm_starts = 0;
    for (i = 0; i < pwm_channel_count; i++)
    {
        pwm_starts |= 0x1 << (pwm_channel_start + i);
    }
    rc = ioThinx_DO_SetPwmStarts(do_slot, pwm_starts);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_SetPwmStarts() = %d\n", rc);
    }

    // set value
    do
    {
        pwm_dutys[0] = (pwm_dutys[0] >= 100) ? 0 : pwm_dutys[0] + 10; // runtime adjust the duty cycle
        rc = ioThinx_DO_SetPwmConfigures(do_slot, pwm_channel_start, 1, pwm_freqs, pwm_dutys);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_DO_SetPwmConfigures() = %d\n", rc);
        }
        for (i = 0; i < pwm_channel_count; i++)
        {
            printf("[%lu:%u] frequency = %d, duty cycle = %d\n", do_slot, pwm_channel_start+i, pwm_freqs[i], pwm_dutys[i]);
        }
        ch = getchar();
    }
    while (ch != 'q');

    rc = ioThinx_DO_SetPwmStops(do_slot, pwm_starts);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_SetPwmStops() = %d\n", rc);
    }

    return 0;
}
