/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * DO Sample Application
 *
 * Date          Author            Comment
 * 2018-12-25    Eddy Chiu         Created it.
 ******************************************************************************/

/**
    \file do.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>DO Sample</b>
    \date 2018-12-25
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is DO sample code.
    <b>Example:</b>
    1. Using default: ./do
    2. Setting DO slot and channel: ./do -s3 -c2
    <b>Default:</b>
    Slot of DO module = 1
    DO channel = 0
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./do -h
    DO sample program.

    Usage: ./do [OPTIONS]

    Options:
        -s      Slot of DO module. Default slot = 1
        -c      DO channel. Default channel = 0
    \endverbatim
    \b Library: \n
    \link DO DO APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include do.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t do_slot = 2;
    uint8_t do_channel = 0;
    uint8_t do_mode = DO_MODE_DO;
    uint32_t do_values = 0;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:hs:")))
    {
        switch (rc)
        {
        case 's':
            do_slot = atoi(optarg);
            break;
        case 'c':
            do_channel = atoi(optarg);
            break;
        case 'h':
        default:
            printf("DO sample program.\n\n");
            printf("Usage: ./do [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of DO module. Default slot = %d\n", "-s", do_slot);
            printf("\t%-8s DO channel. Default channel = %d\n", "-c", do_channel);
            printf("\n");
            return 0;
        }
    }

    printf("DO slot = %lu\n", do_slot);
    printf("DO channel = %u\n", do_channel);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    rc = ioThinx_DO_Config_SetModes(do_slot, do_channel, 1, &do_mode);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_Config_SetModes() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    rc = ioThinx_DO_GetValues(do_slot, &do_values);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DO_GetValues() = %d\n", rc);
    }

    // set value
    do
    {
        // invert do value
        if (do_values & (0x1 << do_channel))
        {
            do_values &= ~(0x1 << do_channel);
        }
        else
        {
            do_values |= 0x1 << do_channel;
        }
        
        rc = ioThinx_DO_SetValues(do_slot, do_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_DO_SetValues() = %d\n", rc);
            break;
        }
        printf("[%lu:%u] DO value = %x\n", do_slot, do_channel, (do_values >> do_channel) & 0x1);
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
