/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * DI Sample Application
 *
 * Date          Author            Comment
 * 2018-12-04    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file di.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>DI Sample</b>
    \date 2018-12-04
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is DI sample code.
    <b>Example:</b>
    1. Using default: ./di
    2. Setting DI slot and channel: ./di -s5 -c2
    <b>Default:</b>
    Slot of DI module = 1
    DI channel = 0
    DI filter = 2
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./di -h
    DI sample program.

    Usage: ./di [OPTIONS]

    Options:
        -s      Slot of DI module. Default slot = 1
        -c      DI channel. Default channel = 0
        -f      Filter of DI module. Default filter = 2
                (One unit = 500us)
    \endverbatim
    \b Library: \n
    \link DI DI APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include di.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t di_slot = 1;
    uint8_t di_channel = 0;
    uint16_t di_filter = 2;
    uint32_t di_values = 0;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:f:hs:")))
    {
        switch (rc)
        {
        case 'c':
            di_channel = atoi(optarg);
            break;
        case 'f':
            di_filter = atoi(optarg);
            break;
        case 's':
            di_slot = atoi(optarg);
            break;
        case 'h':
        default:
            printf("DI sample program.\n\n");
            printf("Usage: ./di [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of DI module. Default slot = %d\n", "-s", di_slot);
            printf("\t%-8s DI channel. Default channel = %d\n", "-c", di_channel);
            printf("\t%-8s Filter of DI module. Default filter = %d\n", "-f", di_filter);
            printf("\t%-8s (One unit = 500us)\n", "");
            printf("\n");
            return 0;
        }
    }

    printf("DI slot = %lu\n", di_slot);
    printf("DI channel = %u\n", di_channel);
    printf("DI filter = %u\n", di_filter);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    rc = ioThinx_DI_Config_SetFilters(di_slot, di_channel, 1, &di_filter);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DI_Config_SetFilters() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    do
    {
        rc = ioThinx_DI_GetValues(di_slot, &di_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_DI_GetValues() = %d\n", rc);
            break;
        }
        printf("[%lu] di_values = 0x%x\n", di_slot, di_values);
        printf("[%lu:%u] di_val = %x\n", di_slot, di_channel, (di_values >> di_channel) & 0x1);
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
