/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Counter Sample Application
 *
 * Date          Author            Comment
 * 2018-12-04    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file counter.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Counter Sample</b>
    \date 2018-12-04
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is counter sample code.
    <b>Example:</b>
    1. Using default: ./counter
    2. Setting DI slot and counter channel: ./counter -s2 -c3 -n1
    <b>Default:</b>
    Slot of DI module = 1
    Counter start channel = 0
    Counter channel count = 2
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./counter -h
    Counter sample program.

    Usage: ./counter [OPTIONS]

    Options:
        -s       Slot of DI module. Default slot = 1
        -c       Counter start channel. Default channel = 0
        -n       Counter channel count. Default count = 2
    \endverbatim
    \b Library: \n
    \link DI DI APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include counter.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t di_slot = 1;
    uint8_t di_modes[16];
    uint8_t counter_channel_start = 0;
    uint8_t counter_channel_count = 2;
    uint32_t counter_starts = 0;
    uint8_t counter_triggers[4];
    uint32_t counter_values[4];
    int32_t i;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:hn:s:")))
    {
        switch (rc)
        {
        case 'c':
            counter_channel_start = atoi(optarg);
            break;
        case 'n':
            counter_channel_count = atoi(optarg);
            break;
        case 's':
            di_slot = atoi(optarg);
            break;
        case 'h':
        default:
            printf("Counter sample program.\n\n");
            printf("Usage: ./counter [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of DI module. Default slot = %d\n", "-s", di_slot);
            printf("\t%-8s Counter start channel. Default channel = %d\n", "-c", counter_channel_start);
            printf("\t%-8s Counter channel count. Default count = %d\n", "-n", counter_channel_count);
            printf("\n");
            return 0;
        }
    }

    printf("DI slot = %lu\n", di_slot);
    printf("Counter start channel = %u\n", counter_channel_start);
    printf("Counter channel count = %u\n", counter_channel_count);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    for (i = 0; i < counter_channel_count; i++)
    {
        di_modes[i] = DI_MODE_COUNTER;
    }
    rc = ioThinx_DI_Config_SetModes(di_slot, counter_channel_start, counter_channel_count, di_modes);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DI_Config_SetModes() = %d\n", rc);
    }
    
    for (i = 0; i < counter_channel_count; i++)
    {
        counter_triggers[i] = CNT_TRIGGER_BOTH;
    }
    rc = ioThinx_DI_Config_SetCntTriggers(di_slot, counter_channel_start, counter_channel_count, counter_triggers);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DI_Config_SetCntTriggers() = %d\n", rc);
    }
    
    for (i = 0; i < counter_channel_count; i++)
    {
        counter_values[i] = 0;
    }
    rc = ioThinx_DI_Config_SetCntValues(di_slot, counter_channel_start, counter_channel_count, counter_values);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DI_Config_SetCntValues() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    // start counter
    counter_starts = 0;
    for (i = 0; i < counter_channel_count; i++)
    {
        counter_starts |= 0x1 << (counter_channel_start + i);
    }    
    rc = ioThinx_DI_SetCntStarts(di_slot, counter_starts);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_DI_SetCntStarts() = %d\n", rc);
    }

    // get value
    do
    {
        rc = ioThinx_DI_GetCntValues(di_slot, counter_channel_start, counter_channel_count, counter_values);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_DI_GetCntValues() = %d\n", rc);
            break;
        }
        for (i = 0; i < counter_channel_count; i++)
        {
            printf("[%lu:%u] counter = %d\n", di_slot, counter_channel_start+i, counter_values[i]);
        }
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
