/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * AO Sample Application
 *
 * Date          Author            Comment
 * 2019-07-26    Koting Hsu        Created it.
 ******************************************************************************/

/**
    \file ao.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>AO Sample</b>
    \date 2019-07-26
    \author Koting Hsu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is AO sample code.
    <b>Example:</b>
    1. Using default: ./ao
    2. Setting AO slot and channel: ./ao -s8 -c2 -n1
    <b>Default:</b>
    Slot of AO module = 8
    AO start channel = 0
    AO channel count = 4
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./ao -h
    AO sample program.

    Usage: ./ao [OPTIONS]

    Options:
        -s       Slot of AO module. Default slot = 8
        -c       AO start channel. Default channel = 0
        -n       AO channel count. Default count = 4
    \endverbatim
    \b Library: \n
    \link AO AO APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include ao.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t ao_slot = 8;
    uint8_t ao_channel_start = 0;
    uint8_t ao_channel_count = 4;
    uint8_t ao_ranges[4];
    uint16_t ao_raws[4];
    float ao_engs[4];
    uint8_t ao_statuss[4];
    int32_t i;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:n:hs:")))
    {
        switch (rc)
        {
        case 's':
            ao_slot = atoi(optarg);
            break;
        case 'c':
            ao_channel_start = atoi(optarg);
            break;
        case 'n':
            ao_channel_count = atoi(optarg);
            break;
        case 'h':
        default:
            printf("AO sample program.\n\n");
            printf("Usage: ./ao [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of AO module. Default slot = %d\n", "-s", ao_slot);
            printf("\t%-8s AO start channel. Default channel = %d\n", "-c", ao_channel_start);
            printf("\t%-8s AO channel count. Default count = %d\n", "-n", ao_channel_count);
            printf("\n");
            return 0;
        }
    }

    printf("AO slot = %lu\n", ao_slot);
    printf("AO start channel = %u\n", ao_channel_start);
    printf("AO channel count = %u\n", ao_channel_count);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    for (i = 0; i < ao_channel_count; i++)
    {
        ao_ranges[i] = AO_RANGE_4_20mA;
    }
    rc = ioThinx_AO_Config_SetRanges(ao_slot, ao_channel_start, ao_channel_count, ao_ranges);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_AO_Config_SetRanges() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    // set value
    for (i = 0; i < ao_channel_count; i++)
    {
        ao_raws[i] = 0x800;
    }
    rc = ioThinx_AO_SetRaws(ao_slot, ao_channel_start, ao_channel_count, ao_raws);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_AO_Config_SetRanges() = %d\n", rc);
    }

    // wait for signal stable
    sleep(1);

    // get value
    do
    {
        rc = ioThinx_AO_GetEngs(ao_slot, ao_channel_start, ao_channel_count, ao_engs);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_AO_GetEngs() = %d\n", rc);
            break;
        }
        rc = ioThinx_AO_GetStatuss(ao_slot, ao_channel_start, ao_channel_count, ao_statuss);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_AO_GetStatuss() = %d\n", rc);
            break;
        }
        for (i = 0; i < ao_channel_count; i++)
        {
            printf("[%lu:%u] eng = %.3f, status = %d\n", ao_slot, ao_channel_start+i, ao_engs[i], ao_statuss[i]);
        }
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
