/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * AI Sample Application
 *
 * Date          Author            Comment
 * 2018-12-27    Eddy Chiu         Created it.
 ******************************************************************************/

/**
    \file ai.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>AI Sample</b>
    \date 2018-12-27
    \author Eddy Chiu
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is AI sample code.
    <b>Example:</b>
    1. Using default: ./ai
    2. Setting AI slot and channel: ./ai -s5 -c2 -n1
    <b>Default:</b>
    Slot of AI module = 4
    AI start channel = 0
    AI channel count = 8
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./ai -h
    AI sample program.

    Usage: ./ai [OPTIONS]

    Options:
        -s       Slot of AI module. Default slot = 4
        -c       AI start channel. Default channel = 0
        -n       AI channel count. Default count = 8
    \endverbatim
    \b Library: \n
    \link AI AI APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iothinx/iothinxio.h>

/// \include ai.c
int main(int argc, char **const argv)
{
    int32_t rc;
    uint32_t ai_slot = 4;
    uint8_t ai_channel_start = 0;
    uint8_t ai_channel_count = 8;
    uint8_t ai_ranges[8];
    float ai_burnouts[8];
    float ai_engs[8];
    uint8_t ai_statuss[8];
    int32_t i;
    char ch;

    while (-1 != (rc = getopt(argc, argv, "c:n:hs:")))
    {
        switch (rc)
        {
        case 's':
            ai_slot = atoi(optarg);
            break;
        case 'c':
            ai_channel_start = atoi(optarg);
            break;
        case 'n':
            ai_channel_count = atoi(optarg);
            break;
        case 'h':
        default:
            printf("AI sample program.\n\n");
            printf("Usage: ./ai [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Slot of AI module. Default slot = %d\n", "-s", ai_slot);
            printf("\t%-8s AI start channel. Default channel = %d\n", "-c", ai_channel_start);
            printf("\t%-8s AI channel count. Default count = %d\n", "-n", ai_channel_count);
            printf("\n");
            return 0;
        }
    }

    printf("AI slot = %lu\n", ai_slot);
    printf("AI start channel = %u\n", ai_channel_start);
    printf("AI channel count = %u\n", ai_channel_count);

    // initialize ioThinx I/O
    rc = ioThinx_IO_Client_Init();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Client_Init() = %d\n", rc);
        return -1;
    }

    // temporarily set config
    for (i = 0; i < ai_channel_count; i++)
    {
        ai_ranges[i] = AI_RANGE_4_20mA_BURNOUT;
    }
    rc = ioThinx_AI_Config_SetRanges(ai_slot, ai_channel_start, ai_channel_count, ai_ranges);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_AI_Config_SetRanges() = %d\n", rc);
    }
    for (i = 0; i < ai_channel_count; i++)
    {
        ai_burnouts[i] = 2.0f;
    }
    rc = ioThinx_AI_Config_SetBurnoutValues(ai_slot, ai_channel_start, ai_channel_count, ai_burnouts);
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_AI_Config_SetBurnoutValues() = %d\n", rc);
    }

    // reload config
    rc = ioThinx_IO_Config_Reload();
    if (rc != IOTHINX_ERR_OK)
    {
        printf("ioThinx_IO_Config_Reload() = %d\n", rc);
    }

    // get value
    do
    {
        rc = ioThinx_AI_GetEngs(ai_slot, ai_channel_start, ai_channel_count, ai_engs);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_AI_GetEngs() = %d\n", rc);
            break;
        }
        rc = ioThinx_AI_GetStatuss(ai_slot, ai_channel_start, ai_channel_count, ai_statuss);
        if (rc != IOTHINX_ERR_OK)
        {
            printf("ioThinx_AI_GetStatuss() = %d\n", rc);
            break;
        }
        for (i = 0; i < ai_channel_count; i++)
        {
            printf("[%lu:%u] eng = %.3f, status = %d\n", ai_slot, ai_channel_start+i, ai_engs[i], ai_statuss[i]);
        }
        ch = getchar();
    }
    while (ch != 'q');

    return 0;
}
