/*******************************************************************************
 * Copyright (C) 2019 Moxa Inc. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Serial Sample Application
 *
 * Date          Author            Comment
 * 2019-01-22    Wanhan Hsieh      Created it.
 ******************************************************************************/

/**
    \file serial.c
    \copyright 2019 Moxa Inc. All rights reserved.
    \brief <b>Serial Sample</b>
    \date 2019-01-22
    \author Wanhan Hsieh
    \version V1.0
    \details
    <pre>
    <b>Introduction:</b>
    This is Serial sample code.
    <b>Example:</b>
    1. Using default: ./serial
    2. Setting slot, port and interface: ./serial -s0 -p0 -i0
    <b>Default:</b>
    Default interface = 0
    Default port = 0
    Default slot = 0
    <b>Help:</b>
    </pre>
    \verbatim
    moxa@Moxa:~$ sudo ./serial -h
    Serial sample program.

    Usage: ./serial [OPTIONS]

    Options:
            -i       Interface. Default interface = 0
            -p       Port. Default port = 0
            -s       Slot. Default slot = 0
    \endverbatim
    \b Library: \n
    \link Serial Serial APIs \endlink
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/termios.h>
#include <iothinx/iothinxio.h>

#define SERIAL_BAUDRATE_300     B300
#define SERIAL_BAUDRATE_600     B600
#define SERIAL_BAUDRATE_1200    B1200
#define SERIAL_BAUDRATE_1800    B1800
#define SERIAL_BAUDRATE_2400    B2400
#define SERIAL_BAUDRATE_4800    B4800
#define SERIAL_BAUDRATE_9600    B9600
#define SERIAL_BAUDRATE_19200   B19200
#define SERIAL_BAUDRATE_38400   B38400
#define SERIAL_BAUDRATE_57600   B57600
#define SERIAL_BAUDRATE_115200  B115200

#define SERIAL_FLOW_CONTROL_NO
//#define SERIAL_FLOW_CONTROL_HW
//#define SERIAL_FLOW_CONTROL_SW
//#define SERIAL_FLOW_CONTROL_BOTH

#define SERIAL_PARITY_NONE
//#define SERIAL_PARITY_ODD
//#define SERIAL_PARITY_EVEN

//#define SERIAL_DATA_BITS_7
#define SERIAL_DATA_BITS_8

#define SERIAL_STOP_BIT_1
//#define SERIAL_STOP_BIT_2

/// \include serial.c
int main(int argc, char **const argv)
{
#define BUF_LEN     256
#define NAME_LEN    16
    struct termios tio;
    int32_t rc;
    char buf[BUF_LEN], devname[NAME_LEN];
    int fd;
    uint32_t interface = SERIAL_INTERFACE_RS232;
    uint32_t port = SERIAL_PORT_1;
    uint32_t slot = 0;
    uint32_t module_count;

    rc = ioThinx_Misc_GetModuleCountML(&module_count);

    if (rc != IOTHINX_ERR_OK)
    {
        printf("Error:ioThinx_Serial_SetInterface() = %d\n", rc);
        return -1;
    }

    printf("total slot = %lu\n", module_count);

    while (-1 != (rc = getopt(argc, argv, "i:p:s:h")))
    {
        switch (rc)
        {
        case 'i':
            interface = atoi(optarg);
            break;
        case 'p':
            port = atoi(optarg);
            break;
        case 's':
            slot = atoi(optarg);
            break;
        case 'h':
        default:
            printf("Serial sample program.\n\n");
            printf("Usage: ./serial [OPTIONS]\n\n");
            printf("Options:\n");
            printf("\t%-8s Interface. Default interface = %d\n", "-i", interface);
            printf("\t%-8s Port. Default port = %d\n", "-p", port);
            printf("\t%-8s Slot. Default slot = %d\n", "-s", slot);
            printf("\n");
            return 0;
        }
    }

    rc = ioThinx_Serial_SetInterface(slot, port, interface);

    if (rc != IOTHINX_ERR_OK)
    {
        printf("Error:ioThinx_Serial_SetInterface() = %d\n", rc);
        return -1;
    }

    rc = ioThinx_Serial_GetInterface(slot, port, &interface);

    if (rc != IOTHINX_ERR_OK)
    {
        printf("Error:ioThinx_Serial_GetInterface() = %d\n", rc);
        return -1;
    }

    printf("port = %lu\n", port);
    printf("slot = %lu\n", slot);
    printf("interface = %lu\n", interface);
    rc = ioThinx_Serial_GetDevName(slot, port, devname);

    if (rc != IOTHINX_ERR_OK)
    {
        printf("Error:ioThinx_Serial_GetDevName() = %d\n", rc);
        return -1;
    }

    printf("devname = %s\n", devname);
    fd = open(devname, O_RDWR);

    if(fd < 0)
    {
        printf("Error:open = %d\n", rc);
        return -1;
    }

    rc = tcgetattr(fd, &tio);

    if(rc < 0)
    {
        printf("Error:tcgetattr = %d\n", rc);
        return -1;
    }

//baudrate
    tio.c_cflag &= ~CBAUD;
    tio.c_cflag |= SERIAL_BAUDRATE_115200;

//flow control
#if defined(SERIAL_FLOW_CONTROL_NO)
    tio.c_cflag &= ~CRTSCTS;
    tio.c_iflag &= ~(IXON | IXOFF);
#elif defined(SERIAL_FLOW_CONTROL_HW)
    tio.c_cflag |= CRTSCTS;
    tio.c_iflag &= ~(IXON | IXOFF);
#elif defined(SERIAL_FLOW_CONTROL_SW)
    tio.c_cflag &= ~CRTSCTS;
    tio.c_iflag |= (IXON | IXOFF);
#elif defined(SERIAL_FLOW_CONTROL_BOTH)
    tio.c_cflag |= CRTSCTS;
    tio.c_iflag |= (IXON | IXOFF);
#else
    printf("No definition of flow control\n");
    return -1;
#endif

//parity
#if defined(SERIAL_PARITY_NONE)
    tio.c_cflag &= ~PARENB;
    tio.c_iflag &= ~INPCK;
#elif defined(SERIAL_PARITY_ODD)
    tio.c_cflag |= PARENB;
    tio.c_cflag |= PARODD;
    tio.c_iflag |= INPCK;
#elif defined(SERIAL_PARITY_EVEN)
    tio.c_cflag |= PARENB;
    tio.c_cflag &= ~PARODD;
    tio.c_iflag |= INPCK;
#else
    printf("No definition of parity\n");
    return -1;
#endif

//data bits
#if defined(SERIAL_DATA_BITS_7)
    tio.c_cflag &= ~CSIZE;
    tio.c_cflag |= CS7;
#elif defined(SERIAL_DATA_BITS_8)
    tio.c_cflag &= ~CSIZE;
    tio.c_cflag |= CS8;
#else
    printf("No definition of data bits\n");
    return -1;
#endif

//stop bit
#if defined(SERIAL_STOP_BIT_1)
    tio.c_cflag &= ~CSTOPB;
#elif defined(SERIAL_STOP_BIT_2)
    tio.c_cflag |= CSTOPB;
#else
    printf("No definition of stop bit\n");
    return -1;
#endif

    rc = tcsetattr(fd, TCSANOW, &tio);

    if(rc < 0)
    {
        printf("Error:tcsetattr = %d\n", rc);
        return -1;
    }

    printf("Start Serial echo.\n");

    while(1)
    {
        rc = read(fd, buf, BUF_LEN);

        if(rc > 0)
        {
            rc = write(fd, buf, rc);
            if (strncmp(buf, "quit", 4) == 0)
            {
                break;
            }
        }
    }

    printf("Stop Serial echo.\n");
    tcflush(fd, TCIOFLUSH);
    close(fd);
    return 0;
}
