
 /*-
 * Copyright (c) 2004 Moxa Technologies Co., Ltd.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Moxa nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*
TODO:
1. MOXA UART RXFIFO trigger enhanced.
2. GDA mode enhanced.
*/

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/types.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/errno.h>
#include <sys/kernel.h>
#include <sys/bus.h>
#include <sys/conf.h>
#include <sys/dkstat.h>
#include <sys/fcntl.h>
#include <sys/tty.h>
#include <machine/bus.h>
#include <machine/resource.h>
#include <sys/rman.h>

#if __FreeBSD_version < 500105 
#include <isa/sioreg.h>
#include <machine/limits.h>
#include <pci/pcireg.h>
#include <pci/pcivar.h>
#else
#include <dev/ic/ns16550.h>
#include <sys/limits.h>
#include <dev/pci/pcivar.h> 
#endif

#include <sys/timepps.h>
#include <sys/interrupt.h>
#include <sys/proc.h>
#include <sys/syslog.h>
#include <sys/sysctl.h>
#if __FreeBSD_version < 500000
#include <machine/ipl.h>
#endif
#include <machine/clock.h>
#include "mx.h"
#include "config.h"


#define CDEV_MAJOR	33

#define CALLOUT_MASK		0x80
#define CONTROL_MASK		0x60
#define CONTROL_INIT_STATE	0x20
#define CONTROL_LOCK_STATE	0x40

#define	LOTS_OF_EVENTS	64	/* helps separate urgent events from input */

#define	CS_BUSY		0x80	/* output in progress */
#define	CS_TTGO		0x40	/* output not stopped by XOFF */
#define	CS_ODEVREADY	0x20	/* external device h/w ready (CTS) */
#define	CS_CHECKMSR	1	/* check of MSR scheduled */
#define	CS_CTS_OFLOW	2	/* use CTS output flow control */
#define	CS_DTR_OFF	0x10	/* DTR held off */
#define	CS_ODONE	4	/* output completed */
#define	CS_RTS_IFLOW	8	/* use RTS input flow control */
#define	CSE_BUSYCHECK	1	/* siobusycheck() scheduled */

static char const * const error_desc[] = {
#define CE_OVERRUN			0
	"silo overflow",
#define CE_INTERRUPT_BUF_OVERFLOW	1
	"interrupt-level buffer overflow",
#define CE_TTY_BUF_OVERFLOW		2
	"tty-level buffer overflow",
};

#define CE_NTYPES		3
#define CE_RECORD(port, errnum) (++(port)->delta_error_counts[errnum])

#define ISA_BUS	0
#define PCI_BUS 1

#if __FreeBSD_version >= 502116
#define dev_t struct cdev*
#endif 

#if __FreeBSD_version >= 500105
static d_open_t	mxopen;
static d_close_t mxclose;
static d_read_t mxread;
static d_write_t mxwrite;
static d_ioctl_t mxioctl;
#endif

#if __FreeBSD_version < 500105 
static int mxopen(dev_t dev, int flag, int mode, struct proc *p);
static int mxclose(dev_t dev, int flag, int mode, struct proc *p);
static int mxread(dev_t dev, struct uio *uio, int flag);
static int mxwrite(dev_t dev, struct uio *uio, int flag);
static int mxioctl(dev_t dev,u_long cmd,caddr_t data,int flag,struct proc *p);
#endif


static char driver_name[] = "mx";

static struct cdevsw mx_cdevsw = {
#if __FreeBSD_version < 500105 
	mxopen,
	mxclose,
	mxread,
	mxwrite,
	mxioctl,
	ttypoll,
  	nommap,
	nostrategy,
 	driver_name,
	CDEV_MAJOR,
	nodump,
	nopsize,
	D_TTY | D_KQFILTER,
#if __FreeBSD_version < 500000 
	-1,
#endif	
	ttykqfilter,
#else
	.d_open =	mxopen,
	.d_close =	mxclose,
	.d_read =	mxread,
	.d_write =	mxwrite,
	.d_ioctl =	mxioctl,
	.d_name =	driver_name,
#if __FreeBSD_version >= 502103 
	.d_version =	D_VERSION,
	.d_flags =	D_TTY | D_NEEDGIANT, 
#else
        .d_maj =       CDEV_MAJOR,
        .d_flags =     D_TRACKCLOSE 
#endif
#endif	
};

/* queue of linear buffers */
struct lbq {
	u_char	*l_head;	/* next char to process */
	u_char	*l_tail;	/* one past the last char to process */
	struct lbq *l_next;	/* next in queue */
	u_char	l_queued;	/* nonzero if queued */
};

struct port_s {
	/* Initial state */
	struct termios		it_in;	/* should be in struct tty */
	struct termios		it_out;
	struct termios		lt_in;
	struct termios		lt_out;
	dev_t			sdev_in;
	dev_t			sdev_in_i;
	dev_t			sdev_in_l;
	dev_t			sdev_out;
	dev_t			sdev_out_i;
	dev_t			sdev_out_l;
	int			ibufsize;
	u_char			*ibuf;
	u_char			*ibufold;
	u_char			*iptr;
	u_char			*ibufend;
	u_char			*ihighwater;
	u_char			*ilowwater;
	int			ierroff;
	struct tty		*tp;
	u_int			tx_fifo_size;
	u_char			MCR;
	u_char			FIFO;
	u_char			LCR;
	u_int			iobase;
	u_int			rx_trigger;
	int			uart_type;
	u_char			hotchar;
	u_char			last_modem_status;
	u_char			prev_modem_status;
	int			dtr_wait;
	struct lbq		obufq;
	struct lbq		obufs[2];
	u_char			obuf1[256];
	u_char			obuf2[256];
	bus_space_tag_t		bst;
	bus_space_handle_t	bsh;
	u_char			state;
	u_int			delta_error_counts[CE_NTYPES];
	u_long			error_counts[CE_NTYPES];
	u_char	no_irq;		/* nonzero if irq is not attached */
	u_char  gone;		/* hardware disappeared */
	u_char	poll;		/* nonzero if polling is required */
	u_char	poll_output;	/* nonzero if polling for output is required */	
	struct pps_state	pps;
	u_long	bytes_in;	/* statistics */	
	u_long  bytes_out;
	u_char	extra_state;	/* more flag bits, separate for order trick */
	u_char			do_timestamp;
	u_char			do_dcd_timestamp;
	struct timeval		timestamp;
	struct timeval 		dcd_timestamp;
	u_int	wopeners;	/* # processes waiting for DCD in open() */	
	u_char			loses_outints;
	u_char			active_out;
		/* added by george_liu 08-21-2003 */
	int			IsMoxaMustChipFlag;

};

struct board_s {
	int			number_ports;
	int			uart_type;
	int			rid;
	u_int			vector_rid;
	u_int			vector;
	int			vector_mask;
	u_long			rclk;
	void 			*cookie;
	struct resource		*irqres;
	struct resource	 	*ioportres;
	struct resource		*vector_res;
	struct port_s		port[MXSER_PORTS_PER_BOARD];
	int			bus_type;
};

struct mxser_hwconf	*hwconf;

static 	u_int		isa_board;
static 	dev_t		mxser_dev;

int	comconsole = -1;
static 	u_int		siocniobase;
static 	u_int		ser_events; 
static 	devclass_t 	mx_devclass;

static 	u_char		mx_registered;

static	int		mx_numunits;
static 	int 		mx_pci_numunits;
static 	int 		mx_isa_numunits;

static	timeout_t 	mxbusycheck;
static	timeout_t 	mxdtrwakeup;
static	int		mx_timeout;
static	int		mx_timeouts_until_log;
static	struct	callout_handle mx_timeout_handle
    = CALLOUT_HANDLE_INITIALIZER(&mx_timeout_handle);
static	timeout_t 	ser_wakeup;    


#if __FreeBSD_version < 500000
static 	swihand_t 	mxpoll;
#else
static void mxpoll (void *);
static void *mx_fast_ih;
static void *mx_slow_ih; 
#endif

static void 	mx_isa_identify(driver_t *, device_t);
static int 	mx_isa_probe(device_t dev);
static int 	mx_isa_attach(device_t dev);
static int 	mx_isa_detach(device_t dev);

static int 	mx_pci_probe(device_t dev);
static int 	mx_pci_attach(device_t dev);
static int 	mx_pci_detach(device_t dev);

static int 	mxser_isa_modevent (module_t mod, int what, void  *arg);
static int 	mxser_pci_modevent (module_t mod, int what, void  *arg);

static int 	mxprobe(device_t dev, int port_rid, int vector_rid);
static int 	mxattach(device_t dev); 
static int 	mxdetach(device_t dev);
static int 	mxsetwater(struct port_s *port, speed_t speed);
static void 	mxinput(struct port_s *port);
static u_int 	mxdivisor(u_long rclk, speed_t speed);
static void	mxsettimeout	__P((void));

static void 	mxintr(void *arg);
static void 	mxintr1(struct port_s *port);

static void 	ser_start	__P((struct tty *tp));
static int 	ser_param	__P((struct tty *tp, struct termios *t));
static void 	ser_stop	__P((struct tty *tp, int rw));
static int 	ser_mctl	__P((struct port_s *port, int bits, int how));
static void 	ser_hardclose(struct port_s *port);

static void 	disc_optim(struct tty *tp, struct termios *t, struct port_s *port);

static int 	mxser_get_ISA_conf(int cap, struct mxser_hwconf *hwconf);
static int 	mxser_read_register(int port, unsigned short *regs);
static int 	mxser_program_mode(int port);
static void 	mxser_normal_mode(int port);
static int 	mxser_ioctl_special(u_long cmd, caddr_t data);

static struct port_s * getportinfo(int unit, int minor_no); 

static device_method_t mx_isa_methods[] = {
	DEVMETHOD(device_identify,	mx_isa_identify),
	DEVMETHOD(device_probe,		mx_isa_probe),
	DEVMETHOD(device_attach,	mx_isa_attach),
	DEVMETHOD(device_detach,	mx_isa_detach),
	{ 0, 0}
};

static driver_t mx_isa_driver = {
	driver_name, 
	mx_isa_methods,
	sizeof(struct board_s),
};

static device_method_t mx_pci_methods[] = {
	DEVMETHOD(device_probe,	mx_pci_probe),
	DEVMETHOD(device_attach, mx_pci_attach),
	DEVMETHOD(device_detach, mx_pci_detach),
	{0, 0}
};

static driver_t mx_pci_driver = {
	driver_name,
	mx_pci_methods,
	sizeof(struct board_s),
};

DRIVER_MODULE(driver_name, pci, mx_pci_driver, mx_devclass, mxser_pci_modevent, 0); 
DRIVER_MODULE(driver_name, isa, mx_isa_driver, mx_devclass, mxser_isa_modevent, 0); 

static int mxser_pci_modevent (module_t mod, int what, void  *arg)
{
	device_t *	devs;
	int		count;
	int		i;
	switch (what) {
	case MOD_LOAD:
		mxser_dev = make_dev(&mx_cdevsw, MXSER_PORTS, UID_ROOT, 
					GID_WHEEL, 0600, "mxser");
		if (!mx_registered) {
#if __FreeBSD_version < 500000
			register_swi(SWI_TTY, mxpoll);
#else
			swi_add(&tty_ithd, "mx", mxpoll, NULL, SWI_TTY, 0, &mx_fast_ih);
			swi_add(&clk_ithd, "mx", mxpoll, NULL, SWI_CLOCK, 0, &mx_slow_ih); 
#endif	
			mx_registered = TRUE;
		}

		break;
	case MOD_UNLOAD:
		destroy_dev(mxser_dev);

#if __FreeBSD_version < 500000
  		unregister_swi(SWI_TTY, mxpoll);
#else  	
		ithread_remove_handler (mx_fast_ih);
		ithread_remove_handler (mx_slow_ih); 	
#endif	
		devclass_get_devices(mx_devclass, &devs, &count);
		for (i = 0; i < count; i++) {
			device_delete_child(device_get_parent(devs[i]),
				devs[i]);
		}
		break;
	default:
		break;
	}

	return(0);
}


static int mxser_isa_modevent (module_t mod, int what, void  *arg)
{
/*
	device_t *	devs;
	int		count;
	int		i;
*/

	switch (what) {
	case MOD_LOAD:
		break;
	case MOD_UNLOAD:
/*
		devclass_get_devices(mx_devclass, &devs, &count);
		for (i = 0; i < count; i++) {
			device_delete_child(device_get_parent(devs[i]),
				devs[i]);
		}*/
		break;
	default:
		break;
	}

	return(0);
}



/********************************************************************/

/*
// follwoing is modified by George Liu. 10-09-2003
//
// follow just for Moxa Must chip define.
//
// when LCR register (offset 0x03) write following value,
// the Must chip will enter enchance mode. And write value
// on EFR (offset 0x02) bit 6,7 to change bank.
*/
#define MOXA_MUST_ENTER_ENCHANCE	0xBF

/* enhance register bank select and enhance mode setting register */
/* when LCR register equal to 0xBF*/
#define MOXA_MUST_EFR_REGISTER		0x02
/* enchance mode enable*/
#define MOXA_MUST_EFR_EFRB_ENABLE	0x10
/* enchance reister bank set 0, 1, 2*/
#define MOXA_MUST_EFR_BANK0		0x00
#define MOXA_MUST_EFR_BANK1		0x40
#define MOXA_MUST_EFR_BANK2		0x80
#define MOXA_MUST_EFR_BANK3		0xC0
#define MOXA_MUST_EFR_BANK_MASK		0xC0

/* set XON1 value register, when LCR=0xBF and change to bank0*/
#define MOXA_MUST_XON1_REGISTER		0x04

/* set XON2 value register, when LCR=0xBF and change to bank0*/
#define MOXA_MUST_XON2_REGISTER		0x05

/* set XOFF1 value register, when LCR=0xBF and change to bank0*/
#define MOXA_MUST_XOFF1_REGISTER	0x06

/* set XOFF2 value register, when LCR=0xBF and change to bank0*/
#define MOXA_MUST_XOFF2_REGISTER	0x07

#define MOXA_MUST_RBRTL_REGISTER	0x04
#define MOXA_MUST_RBRTH_REGISTER	0x05
#define MOXA_MUST_RBRTI_REGISTER	0x06
#define MOXA_MUST_THRTL_REGISTER	0x07
#define MOXA_MUST_ENUM_REGISTER		0x04
#define MOXA_MUST_HWID_REGISTER		0x05
#define MOXA_MUST_ECR_REGISTER		0x06
#define MOXA_MUST_CSR_REGISTER		0x07

/* good data mode enable*/
#define MOXA_MUST_FCR_GDA_MODE_ENABLE	0x20
/* only good data put into RxFIFO*/
#define MOXA_MUST_FCR_GDA_ONLY_ENABLE	0x10

/* enable CTS interrupt*/
#define MOXA_MUST_IER_ECTSI		0x80
/* eanble RTS interrupt*/
#define MOXA_MUST_IER_ERTSI		0x40
/* enable Xon/Xoff interrupt*/
#define MOXA_MUST_IER_XINT		0x20
/* enable GDA interrupt*/
#define MOXA_MUST_IER_EGDAI		0x10

#define MOXA_MUST_RECV_ISR		(UART_IER_RDI | MOXA_MUST_IER_EGDAI)

/* GDA interrupt pending*/
#define MOXA_MUST_IIR_GDA		0x1C
#define MOXA_MUST_IIR_RDA		0x04
#define MOXA_MUST_IIR_RTO		0x0C
#define MOXA_MUST_IIR_LSR		0x06

/* recieved Xon/Xoff or specical interrupt pending*/
#define MOXA_MUST_IIR_XSC		0x10

/* RTS/CTS change state interrupt pending*/
#define MOXA_MUST_IIR_RTSCTS		0x20
#define MOXA_MUST_IIR_MASK		0x3E

#define MOXA_MUST_MCR_XON_FLAG		0x40
#define MOXA_MUST_MCR_XON_ANY		0x80
#define MOXA_MUST_HARDWARE_ID		0x01

/* software flow control on chip mask value*/
#define MOXA_MUST_EFR_SF_MASK		0x0F
/* send Xon1/Xoff1*/
#define MOXA_MUST_EFR_SF_TX1		0x08
/* send Xon2/Xoff2*/
#define MOXA_MUST_EFR_SF_TX2		0x04
/* send Xon1,Xon2/Xoff1,Xoff2*/
#define MOXA_MUST_EFR_SF_TX12		0x0C
/* don't send Xon/Xoff*/
#define MOXA_MUST_EFR_SF_TX_NO		0x00
/* Tx software flow control mask*/
#define MOXA_MUST_EFR_SF_TX_MASK	0x0C
/* don't receive Xon/Xoff*/
#define MOXA_MUST_EFR_SF_RX_NO		0x00
/* receive Xon1/Xoff1*/
#define MOXA_MUST_EFR_SF_RX1		0x02
/* receive Xon2/Xoff2*/
#define MOXA_MUST_EFR_SF_RX2		0x01
/* receive Xon1,Xon2/Xoff1,Xoff2*/
#define MOXA_MUST_EFR_SF_RX12		0x03
/* Rx software flow control mask*/
#define MOXA_MUST_EFR_SF_RX_MASK	0x03

#define MOXA_MUST_MIN_XOFFLIMIT		66
#define MOXA_MUST_MIN_XONLIMIT		20

#ifndef UCHAR
typedef unsigned char	UCHAR;
#endif

#define ENABLE_MOXA_MUST_ENCHANCE_MODE(baseio) { \
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr |= MOXA_MUST_EFR_EFRB_ENABLE;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define DISABLE_MOXA_MUST_ENCHANCE_MODE(baseio) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_EFRB_ENABLE;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define SET_MOXA_MUST_XON1_VALUE(baseio, Value) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR, MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_BANK_MASK;	\
	__efr |= MOXA_MUST_EFR_BANK0;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+MOXA_MUST_XON1_REGISTER,(UCHAR)(Value));	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define SET_MOXA_MUST_XOFF1_VALUE(baseio, Value) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_BANK_MASK;	\
	__efr |= MOXA_MUST_EFR_BANK0;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+MOXA_MUST_XOFF1_REGISTER,(UCHAR)(Value));	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define SET_MOXA_MUST_FIFO_VALUE(info) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((info)->iobase+UART_LCR);	\
	outb((info)->iobase+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((info)->iobase+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_BANK_MASK;	\
	__efr |= MOXA_MUST_EFR_BANK1;	\
	outb((info)->iobase+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((info)->iobase+MOXA_MUST_THRTL_REGISTER,(UCHAR)0);	\
	outb((info)->iobase+MOXA_MUST_RBRTL_REGISTER,(UCHAR)8);	\
	outb((info)->iobase+MOXA_MUST_RBRTH_REGISTER,(UCHAR)(58));	\
	outb((info)->iobase+MOXA_MUST_RBRTI_REGISTER,(UCHAR)58);	\
	outb((info)->iobase+UART_LCR,__oldlcr);	\
}		

#define GET_MOXA_MUST_HARDWARE_ID(baseio, pId) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_BANK_MASK;	\
	__efr |= MOXA_MUST_EFR_BANK2;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	*pId = inb((baseio)+MOXA_MUST_HWID_REGISTER);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define ENABLE_MOXA_MUST_TX_SOFTWARE_FLOW_CONTROL(baseio) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_SF_TX_MASK;	\
	__efr |= MOXA_MUST_EFR_SF_TX1;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define ENABLE_MOXA_MUST_RX_SOFTWARE_FLOW_CONTROL(baseio) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_SF_RX_MASK;	\
	__efr |= MOXA_MUST_EFR_SF_RX1;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define DISABLE_MOXA_MUST_RX_SOFTWARE_FLOW_CONTROL(baseio) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_SF_RX_MASK;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define ENABLE_MOXA_MUST_TX_RX_SOFTWARE_FLOW_CONTROL(baseio) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_SF_MASK;	\
	__efr |= (MOXA_MUST_EFR_SF_RX1|MOXA_MUST_EFR_SF_TX1);	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define DISABLE_MOXA_MUST_TX_SOFTWARE_FLOW_CONTROL(baseio) {	\
	UCHAR	__oldlcr, __efr;	\
	__oldlcr = inb((baseio)+UART_LCR);	\
	outb((baseio)+UART_LCR,MOXA_MUST_ENTER_ENCHANCE);	\
	__efr = inb((baseio)+MOXA_MUST_EFR_REGISTER);	\
	__efr &= ~MOXA_MUST_EFR_SF_TX_MASK;	\
	outb((baseio)+MOXA_MUST_EFR_REGISTER,__efr);	\
	outb((baseio)+UART_LCR,__oldlcr);	\
}

#define ENABLE_MOXA_MUST_XON_ANY_FLOW_CONTROL(baseio) { \
	UCHAR	__oldmcr;	\
	__oldmcr = inb((baseio)+UART_MCR);	\
	__oldmcr |= MOXA_MUST_MCR_XON_ANY;	\
	outb((baseio)+UART_MCR,__oldmcr);	\
}

#define DISABLE_MOXA_MUST_XON_ANY_FLOW_CONTROL(baseio) { \
	UCHAR	__oldmcr;	\
	__oldmcr = inb((baseio)+UART_MCR);	\
	__oldmcr &= ~MOXA_MUST_MCR_XON_ANY;	\
	outb((baseio)+UART_MCR,__oldmcr);	\
}

static int CheckIsMoxaMust(int io)
{
	UCHAR	oldmcr, hwid;

	outb(io+UART_LCR,0);
	DISABLE_MOXA_MUST_ENCHANCE_MODE(io);
	oldmcr = inb(io+UART_MCR);
	outb(io+UART_MCR,0);
	SET_MOXA_MUST_XON1_VALUE(io, 0x11);
	if ( (hwid=inb(io+UART_MCR)) != 0 ) {
		outb(io+UART_MCR, oldmcr);
		return(0);
	}
	GET_MOXA_MUST_HARDWARE_ID(io, &hwid);
	if ( hwid != MOXA_MUST_HARDWARE_ID ){
		return(0);
	}
	return(1);
}
/* above is modified by George Liu. 10-09-2003*/
/********************************************************************/


static void mx_isa_identify(driver_t *driver, device_t parent)
{
	devclass_t	dc;
	device_t	child;
	u_long		ioaddr, count;
	int 		retval, i;

	hwconf = malloc(4 * sizeof(struct mxser_hwconf), M_DEVBUF, M_NOWAIT);
	if (hwconf == NULL) {
		return;
	}

	bzero(hwconf, (4*sizeof(struct mxser_hwconf)));

	isa_board = 0;
	for (i=0; i<MXSER_BOARDS; i++) {
		if (!(ioaddr=mxserBoardCAP[i]))
			continue;
	
		retval = mxser_get_ISA_conf(ioaddr, &hwconf[i]);
		if (retval <= 0) {
			if (retval == MXSER_ERR_IRQ)
				uprintf("Invalid interrupt number, board not configured\n");
			else if (retval == MXSER_ERR_IRQ_CONFLIT)
				uprintf("Invalid interrupt conflit, board not configured\n");
			else if (retval == MXSER_ERR_VECTOR)
				uprintf("Invalid interrupt vector, board not configured\n");
			else if (retval == MXSER_ERR_IOADDR)
				uprintf("Invalid I/O address, board not configured\n");
			continue;
		}
		isa_board++;
	}

	if (!isa_board) {
		free(hwconf, M_DEVBUF);
		return;
	}

	dc = devclass_find(driver_name);
	if (devclass_get_device(dc, mx_pci_numunits) == NULL) {
		for(i=0; i<isa_board; i++) {
			child = BUS_ADD_CHILD(parent, 0, driver_name, -1);
			count = hwconf[i].ports * 8;

			bus_set_resource(child, SYS_RES_IOPORT, 0, 
					hwconf[i].ioaddr[0], count);
			bus_set_resource(child, SYS_RES_IOPORT, 1, 
					hwconf[i].vector, 1);
			bus_set_resource(child, SYS_RES_IRQ, 0, 
					hwconf[i].irq, 1);
		}
	}
}

static int mx_isa_probe(device_t dev)
{
	int	type_idx;

	type_idx = hwconf[mx_isa_numunits].board_type - 1;
	device_set_desc(dev, mxser_brdname[type_idx]);

	return (mxprobe(dev, 0, 1));
}

static int mx_isa_attach(device_t dev)
{
	struct board_s	*info_board;
	int	idx;

	info_board = device_get_softc(dev);
	if (info_board == NULL) {
		return ENXIO;
	}
	bzero(info_board, sizeof(struct board_s));

	info_board->number_ports = hwconf[mx_isa_numunits].ports;
	info_board->rid = 0;
	if (hwconf[mx_isa_numunits].baud_base[0] == 115200)
		info_board->rclk = 1843200;
	else 
		info_board->rclk = 14745600;

	info_board->vector_mask = hwconf[mx_isa_numunits].vector_mask;
	info_board->vector_rid = 1;

	info_board->uart_type = hwconf[mx_isa_numunits].uart_type;

	idx = mx_isa_numunits + mx_pci_numunits;
	mxsercfg[idx].board_type = hwconf[mx_isa_numunits].board_type;
	mxsercfg[idx].ports = info_board->number_ports;
	mxsercfg[idx].baud_base[0] = hwconf[mx_isa_numunits].baud_base[0];   

	mx_isa_numunits++;
	if (mx_isa_numunits == isa_board) {
		free(hwconf, M_DEVBUF);
	}
	
	info_board->bus_type = ISA_BUS;

	
	return (mxattach(dev));
}

static int mxprobe(device_t dev, int port_rid, int vector_rid)
{
	struct resource	*portres, *v_res;
	int	rid, v_rid;


	rid = port_rid;
	portres = bus_alloc_resource(dev, SYS_RES_IOPORT, &rid, 0, ~0,
							0x08, RF_ACTIVE);
	if (!portres) {
		return ENXIO;
	}

	v_rid = vector_rid;
	v_res = bus_alloc_resource(dev, SYS_RES_IOPORT, &v_rid, 0, ~0,
							0x01, RF_ACTIVE);
	if (!v_res) {
		return ENXIO;
	}

	bus_release_resource(dev, SYS_RES_IOPORT, rid, portres);
	bus_release_resource(dev, SYS_RES_IOPORT, v_rid, v_res);

	return 0;	
}

static int mxattach(device_t dev)
{
	int 			rid, minor, i, uart_type, vector_rid;
	int 			ret, unit;
	struct resource		*portres, *irqres, *vector_res;
	u_long			count;
	u_int			iobase;
	struct board_s		*info_board;
	struct port_s		*info_port;
    u_int32_t		devid;

	info_board = device_get_softc(dev);
	if (info_board == NULL) {
		return ENXIO;
	}
	rid = info_board->rid;

	count = info_board->number_ports * 8;
	portres = bus_alloc_resource(dev, SYS_RES_IOPORT, &rid, 
					0, ~0, count, RF_ACTIVE);
	if (!portres) {
		return ENXIO;
	}
	info_board->ioportres = portres;

	/*portres = info_board->ioportres;*/

	vector_rid = info_board->vector_rid;
	if (vector_rid == 1) {
		vector_res = bus_alloc_resource(dev, SYS_RES_IOPORT,
				&vector_rid, 0, ~0, 1, RF_ACTIVE);
	} else {
		vector_res = bus_alloc_resource(dev, SYS_RES_IOPORT,
				&vector_rid, 0, ~0, 0x10, RF_ACTIVE);
	}
										
	if (!vector_res) {
		uprintf("bus_alloc_resource IOPORT vector failed\n");
		return ENXIO;
	}

	info_board->vector_res = vector_res;
	info_board->vector = rman_get_start(vector_res);

	unit = device_get_unit(dev);

	if (unit >= mx_numunits)
		mx_numunits = unit + 1;

	iobase = rman_get_start(portres);

	mxsercfg[unit].ioaddr[0] = iobase; 
	
	devid = pci_get_devid(dev); /* Added by Danny Lin. 2004.9.14. */

	if(info_board->bus_type==PCI_BUS){
		if(CheckIsMoxaMust(iobase)){
		    /* Starting add by Danny Lin. 2004.9.14. */
			if(devid != CP102_PCI_DEVICE_ID)
				mxsercfg[unit].baud_base[0] = B230400;
			else
				mxsercfg[unit].baud_base[0] = 921600L;
		    /* Ending add by Danny Lin. 2004.9.14. */
 	    }else
			mxsercfg[unit].baud_base[0] = 921600L;
	}
	
	

	uart_type = info_board->uart_type;
	info_port = &info_board->port[0];

	for (i=0; i<info_board->number_ports; i++, info_port++) {
		info_port->bst = rman_get_bustag(portres);
		info_port->bsh = rman_get_bushandle(portres);
		info_port->iobase = iobase + 0x08 * i;
		/*
		This will cause next open blocked.
		info_port->dtr_wait = 3 * hz;
		*/
		info_port->dtr_wait = 0;
		info_port->loses_outints = 0;
		info_port->uart_type = uart_type;
		info_port->no_irq = bus_get_resource(dev, SYS_RES_IRQ, 0, NULL, NULL) != 0;		
		info_port->IsMoxaMustChipFlag = CheckIsMoxaMust(info_port->iobase);
		if ((uart_type == PORT_16450) || (uart_type == PORT_8250))
			info_port->tx_fifo_size = 1;
		else
			info_port->tx_fifo_size = 16;

		if ((uart_type == PORT_16450) || (uart_type == PORT_8250))
			info_port->rx_trigger = 1;
		else
			info_port->rx_trigger = 14;

		if(info_port->IsMoxaMustChipFlag){
			ENABLE_MOXA_MUST_ENCHANCE_MODE(info_port->iobase);
			info_port->tx_fifo_size = 64;
		}


		info_port->obufs[0].l_head = info_port->obuf1;
		info_port->obufs[1].l_head = info_port->obuf2;

		info_port->it_in.c_iflag = 0;
		info_port->it_in.c_oflag = 0;
		info_port->it_in.c_cflag = TTYDEF_CFLAG;
		info_port->it_in.c_lflag = 0;
		if (unit == comconsole) {
		} else {
			info_port->it_in.c_ispeed = 
			info_port->it_in.c_ospeed = TTYDEF_SPEED;
		}
		if (mxsetwater(info_port, info_port->it_in.c_ispeed) != 0) {
			enable_intr();
			if (info_port->iobase != siocniobase)
				bus_release_resource(dev, SYS_RES_IOPORT, rid, portres);
			return ENOMEM;
		}
		enable_intr();
		termioschars(&info_port->it_in);
		info_port->it_out = info_port->it_in;
		outb((info_port->iobase + UART_FCR), 0);

		info_port->pps.ppscap = PPS_CAPTUREASSERT | PPS_CAPTURECLEAR;
		pps_init(&info_port->pps);

	}

	minor = unit * 8;
/*	if (unit == 0)
		mxser_dev = make_dev(&mx_cdevsw, MXSER_PORTS, UID_ROOT, 
					GID_WHEEL, 0600, "mxser");*/

	for (i=0; i<info_board->number_ports; i++, minor++) {
		info_board->port[i].sdev_in = 
			make_dev(&mx_cdevsw, minor, 
				UID_ROOT, GID_WHEEL, 0600, "ttyM%d", minor);

		info_board->port[i].sdev_in_i = 
			make_dev(&mx_cdevsw, minor | CONTROL_INIT_STATE, 
				UID_ROOT, GID_WHEEL, 0600, "ttyiM%d", minor);

		info_board->port[i].sdev_in_l = 
			make_dev(&mx_cdevsw, minor | CONTROL_LOCK_STATE, 
				UID_ROOT, GID_WHEEL, 0600, "ttylM%d", minor);

		info_board->port[i].sdev_out = 
			make_dev(&mx_cdevsw, minor | CALLOUT_MASK,
				UID_UUCP, GID_DIALER, 0660, "cum%d", minor);

		info_board->port[i].sdev_out_i = 
			make_dev(&mx_cdevsw, minor | CALLOUT_MASK | CONTROL_INIT_STATE,
				UID_UUCP, GID_DIALER, 0660, "cuim%r", minor);

		info_board->port[i].sdev_out_l = 
			make_dev(&mx_cdevsw, minor | CALLOUT_MASK | CONTROL_LOCK_STATE,
				UID_UUCP, GID_DIALER, 0660, "culm%r", minor);
	}

	rid = 0;
	irqres = bus_alloc_resource(dev, SYS_RES_IRQ, &rid, 0ul, ~0ul, 1,
	    			RF_ACTIVE | RF_SHAREABLE);

	if (!irqres) 
		uprintf("bus_alloc_resource IRQ failed\n");
	    
	if (irqres) {
		if(info_board->bus_type == ISA_BUS){
		       ret = BUS_SETUP_INTR(device_get_parent(dev), dev, irqres,
#if __FreeBSD_version < 500000
				     INTR_TYPE_TTY | INTR_TYPE_FAST,
#else				 
			             INTR_TYPE_TTY | INTR_FAST,  
#endif			            
				     mxintr, info_board, &info_board->cookie);
		}else{
		       ret = BUS_SETUP_INTR(device_get_parent(dev), dev, irqres,
#if __FreeBSD_version < 500000
				     INTR_TYPE_TTY | INTR_TYPE_FAST,
#else				 
			             INTR_TYPE_TTY,  
#endif			            
				     mxintr, info_board, &info_board->cookie);
		}
		if (ret) {
			ret = BUS_SETUP_INTR(device_get_parent(dev), dev,
				     irqres, INTR_TYPE_TTY,
				     mxintr, info_board, &info_board->cookie);
			if (ret == 0)
				uprintf("unable to activate interrupt in fast mode - using normal mode\n");
		}
		if (ret)
			uprintf("could not activate interrupt\n");

	}
	info_board->irqres = irqres;

	return 0; 
}

static int mx_isa_detach(device_t dev)
{
	return (mxdetach(dev));
}

static int mxdetach(device_t dev)
{
	struct board_s	*info_board;
	int	i;
	struct resource *res;

	info_board = device_get_softc(dev);
	if (info_board == NULL) {
		return 0;
	}
	res = info_board->ioportres;
	if (res)
		bus_release_resource(dev, SYS_RES_IOPORT, info_board->rid, res);
		
	res = info_board->vector_res;
	if (res)
		bus_release_resource(dev, SYS_RES_IOPORT, info_board->vector_rid, res);
	
	for (i=0; i<info_board->number_ports; i++) {
 		destroy_dev(info_board->port[i].sdev_in);
 		destroy_dev(info_board->port[i].sdev_in_i);
 		destroy_dev(info_board->port[i].sdev_in_l);
 		destroy_dev(info_board->port[i].sdev_out);
 		destroy_dev(info_board->port[i].sdev_out_i);
 		destroy_dev(info_board->port[i].sdev_out_l);
	}
	
	res = info_board->irqres;
	if (res) {
		bus_teardown_intr(dev, res, info_board->cookie);
		bus_release_resource(dev, SYS_RES_IRQ, 0, res);
	}

	/*destroy_dev(mxser_dev);*/
/*
#if __FreeBSD_version < 500000
  	unregister_swi(SWI_TTY, mxpoll);
#else  	
	ithread_remove_handler (mx_fast_ih);
	ithread_remove_handler (mx_slow_ih); 	
#endif	*/
	return 0;
}

struct mxpci_ids {
	u_int32_t 	type;
	int		board_type;
	int		rid;
	int		num_ports;
};

static struct mxpci_ids mxpci_ids[] = {
	{ C168_PCI_DEVICE_ID, MXSER_BOARD_C168_PCI, PCI_BASE_ADDRESS_2, 8}, 
	{ C104_PCI_DEVICE_ID, MXSER_BOARD_C104_PCI, PCI_BASE_ADDRESS_2, 4}, 
	{ CP132_PCI_DEVICE_ID, MXSER_BOARD_CP132, PCI_BASE_ADDRESS_2, 2}, 	
	{ CP114_PCI_DEVICE_ID, MXSER_BOARD_CP114, PCI_BASE_ADDRESS_2, 4}, 	
	{ CT114_PCI_DEVICE_ID, MXSER_BOARD_CT114, PCI_BASE_ADDRESS_2, 4}, 	
	{ CP104U_PCI_DEVICE_ID, MXSER_BOARD_CP104U, PCI_BASE_ADDRESS_2, 4},
	{ CP168U_PCI_DEVICE_ID, MXSER_BOARD_CP168U, PCI_BASE_ADDRESS_2, 8},
	{ CP132U_PCI_DEVICE_ID, MXSER_BOARD_CP132U, PCI_BASE_ADDRESS_2, 2},
	{ CP134U_PCI_DEVICE_ID, MXSER_BOARD_CP134U, PCI_BASE_ADDRESS_2, 4},
	{ CP104JU_PCI_DEVICE_ID, MXSER_BOARD_CP104JU, PCI_BASE_ADDRESS_2, 4},
	{ CP102_PCI_DEVICE_ID, MXSER_BOARD_CP102, PCI_BASE_ADDRESS_2, 2},
	/* Starting add by Danny Lin. 2004.7.21 */
	{ CP118U_PCI_DEVICE_ID, MXSER_BOARD_CP118U, PCI_BASE_ADDRESS_2, 8},
	{ CP102UL_PCI_DEVICE_ID, MXSER_BOARD_CP102UL, PCI_BASE_ADDRESS_2, 2},
	{ CP102U_PCI_DEVICE_ID, MXSER_BOARD_CP102U, PCI_BASE_ADDRESS_2, 2},	
	/* Ending add by Danny Lin. 2004.7.21 */
	/* Starting add by Danny Lin. 2005.9.13 */
	{ CP118EL_PCI_DEVICE_ID, MXSER_BOARD_CP118EL, PCI_BASE_ADDRESS_2, 8},
	{ CP168EL_PCI_DEVICE_ID, MXSER_BOARD_CP168EL, PCI_BASE_ADDRESS_2, 8},
	{ CP104EL_PCI_DEVICE_ID, MXSER_BOARD_CP104EL, PCI_BASE_ADDRESS_2, 4},	
	/* Ending add by Danny Lin. 2005.9.13 */
	{ 0x00000000, 0, 0, 0}
};

static int mx_pci_probe(device_t dev)
{
	u_int32_t		type;
	struct mxpci_ids	*id;
	type = pci_get_devid(dev);
	id = mxpci_ids;
	while (id->type && id->type != type)
		id++;
	if (id->board_type == 0){
		return ENXIO;
	}
	device_set_desc(dev, mxser_brdname[id->board_type-1]);
	
	return (mxprobe(dev, PCI_BASE_ADDRESS_2, PCI_BASE_ADDRESS_3));
}

static int mx_pci_attach(device_t dev)
{
	u_int32_t		type;
	struct mxpci_ids	*id;
	struct board_s		*info_board;
	int			i, vector_mask;

	type = pci_get_devid(dev);
	id = mxpci_ids;
	while (id->type && id->type != type)
		id++;
	if (id->board_type == 0)
		return ENXIO;

	info_board = device_get_softc(dev);
	if (info_board == NULL) {
		return ENXIO;
	}
	bzero(info_board, sizeof *info_board);

	info_board->number_ports = id->num_ports;
	vector_mask = 0;
	for (i=0; i<id->num_ports; i++)
		vector_mask |= (1<<i);
	info_board->vector_mask = vector_mask; 
	info_board->uart_type = PORT_16550A;
	info_board->rid = id->rid;
	info_board->vector_rid = PCI_BASE_ADDRESS_3;
	info_board->rclk = 14745600;
	
	mxsercfg[mx_pci_numunits].board_type = id->board_type;
	mxsercfg[mx_pci_numunits].ports = id->num_ports;
	
	mx_pci_numunits++;
	
	info_board->bus_type = PCI_BUS;

	return (mxattach(dev));

}

static int mx_pci_detach(device_t dev)
{
	return (mxdetach(dev));
}

static int mxsetwater(struct port_s *port, speed_t speed)
{
	int		cp4ticks, ibufsize;
	u_char		*ibuf;
	struct tty	*tp;

	cp4ticks = speed / 10 / hz * 4;
	for (ibufsize=128; ibufsize<cp4ticks; )
		ibufsize <<= 1;
	if (ibufsize == port->ibufsize) {
		disable_intr();
		return 0;
	}

	ibuf = malloc(2 * ibufsize, M_DEVBUF, M_NOWAIT);
	if (ibuf == NULL) {
		disable_intr();
		return ENOMEM;
	}

	/* Initialize non-critical variables */
	port->ibufold = port->ibuf;
	port->ibufsize = ibufsize;
	tp = port->tp;
	if (tp != NULL) {
		tp->t_ififosize = 2 * ibufsize;
		tp->t_ispeedwat = (speed_t)-1;
		tp->t_ospeedwat = (speed_t)-1;	
	}

	disable_intr();
	if (port->iptr != port->ibuf)
		mxinput(port);

	port->iptr = port->ibuf = ibuf;
	port->ibufend = ibuf + ibufsize;
	port->ierroff = ibufsize;
	port->ihighwater = ibuf + 3 * ibufsize / 4;
	port->ilowwater = ibuf + 1 * ibufsize / 4;

	return 0;
}

static void mxinput(struct port_s *port)
{
	u_char		*buf, line_status;
	int		incc, recv_data;
	struct tty	*tp;

	buf = port->ibuf;
	tp = port->tp;
	if (!(tp->t_state & TS_ISOPEN) || !(tp->t_cflag & CREAD)) {
		ser_events -= (port->iptr - port->ibuf);
		port->iptr = port->ibuf;
		return;
	}

	if (tp->t_state & TS_CAN_BYPASS_L_RINT) {
		do {
			enable_intr();
			incc = port->iptr - buf;
			if (tp->t_rawq.c_cc + incc > tp->t_ihiwat
				&& (port->state & CS_RTS_IFLOW || tp->t_iflag & IXOFF)
				&& !(tp->t_state & TS_TBLOCK))
				ttyblock(tp);
			
			port->delta_error_counts[CE_TTY_BUF_OVERFLOW] +=
				b_to_q((char *)buf, incc, &tp->t_rawq);
			buf += incc;
			tk_nin += incc;
			tk_rawcc += incc;
			ttwakeup(tp);
			if (tp->t_state & TS_TTSTOP &&
				(tp->t_iflag & IXANY || tp->t_cc[VSTART] == tp->t_cc[VSTOP])) {
				tp->t_state &= ~TS_TTSTOP;
				tp->t_lflag &= ~FLUSHO;
				ser_start(tp);
			}
			disable_intr();
		} while (buf < port->iptr);
	} else {
		do {
			enable_intr();
			line_status = buf[port->ierroff];
			recv_data = *buf++;
			if (line_status & (LSR_BI | LSR_FE | LSR_OE | LSR_PE)) {
				if (line_status & LSR_BI)
					recv_data |= TTY_BI;
				if (line_status & LSR_FE)
					recv_data |= TTY_FE;
				if (line_status & LSR_OE)
					recv_data |= TTY_OE;
				if (line_status & LSR_PE)
					recv_data |= TTY_PE;
			}
#if __FreeBSD_version < 500000
			(*linesw[tp->t_line].l_rint)(recv_data, tp);
#else
			(*linesw[tp->t_line]->l_rint)(recv_data, tp); 
#endif			
			disable_intr();
		} while (buf < port->iptr);
	}
	ser_events -= (port->iptr - port->ibuf);
	port->iptr = port->ibuf;
	if ((port->state & CS_RTS_IFLOW) && !(port->MCR & MCR_RTS) &&
		!(tp->t_state & TS_TBLOCK))
		outb((port->iobase + UART_MCR), port->MCR |= MCR_RTS);
	/* added by casper */	
	if( port->IsMoxaMustChipFlag ){
		char ret;
		ret = inb(port->iobase+UART_IER);
		if((port->tp->t_iflag & IXOFF) &&
			!(ret & UART_IER_RDI) &&
			!(tp->t_state & TS_TBLOCK)){
			outb((port->iobase+UART_IER),ret|UART_IER_RDI);
		}
	}
		
}

static void mxintr(void *arg)
{
	struct board_s	*info_board;
	struct port_s	*port;
	int		irqbits, max, bits, i;
	int		vector_mask, pass_counter=0;
	u_int		vector;

	info_board = (struct board_s *)arg;
	if (info_board == NULL) {
		return;
	}
	max = info_board->number_ports;
	vector = info_board->vector;
	vector_mask = info_board->vector_mask;
	
	while (1) {
		irqbits = inb(vector) & vector_mask;
		if (irqbits == vector_mask)
			break;

		for (i=0, bits=1; i<max; i++, irqbits |= bits, bits <<= 1) {
			if (irqbits == vector_mask)
				break;
			if (bits & irqbits)
				continue;
			port = &info_board->port[i];
			if (port != NULL && !port->gone
				&& (inb(port->iobase + UART_IIR) & IIR_IMASK) != IIR_NOPEND)
				mxintr1(port);
		}
		if (pass_counter++ > MXSER_ISR_PASS_LIMIT) {
			break;
		}
	}
}

static void mxintr1(struct port_s *port)
{
	u_char	line_status;
	u_char	modem_status;
	u_char	*ioptr;
	u_char	recv_data;
	u_char	int_ctl;
	u_char	int_ctl_new;
#if __FreeBSD_version < 500000
	struct	timecounter *tc;
	u_int	count;
#endif
	struct tty	*tp;
 	u_char rxd;

	tp = port->tp;

	int_ctl = inb(port->iobase + UART_IER);
	int_ctl_new = int_ctl;

	rxd = inb(port->iobase+UART_IIR);

	while (!port->gone) {
		if (port->pps.ppsparam.mode & PPS_CAPTUREBOTH) {
			modem_status = inb(port->iobase + UART_MSR);
		        if ((modem_status ^ port->last_modem_status)&MSR_DCD) {
#if __FreeBSD_version < 500000
				tc = timecounter;
				count = tc->tc_get_timecount(tc);
				pps_event(&port->pps, tc, count, 
#else				
				pps_event(&port->pps, 				
#endif				
				    (modem_status & MSR_DCD) ? 
				    PPS_CAPTUREASSERT : PPS_CAPTURECLEAR);
			}
		}
		line_status = inb(port->iobase + UART_LSR);

		/* input event? (check first to help avoid overruns) */
		if (rxd & IIR_RXRDY)
		while(1){	
			if(! (line_status & LSR_RCV_MASK)) 
				break;
			/* break/unnattached error bits or real input? */
			if(!(line_status & LSR_RXRDY))
				recv_data = 0;
			else
				recv_data = inb(port->iobase + UART_RX);

			if (line_status & (LSR_BI | LSR_FE | LSR_PE)) {
				/*
				 * Don't store BI if IGNBRK or FE/PE if IGNPAR.
				 * Otherwise, push the work to a higher level
				 * (to handle PARMRK) if we're bypassing.
				 * Otherwise, convert BI/FE and PE+INPCK to 0.
				 *
				 * This makes bypassing work right in the
				 * usual "raw" case (IGNBRK set, and IGNPAR
				 * and INPCK clear).
				 *
				 * Note: BI together with FE/PE means just BI.
				 */
				if (line_status & LSR_BI) {
					if (port->tp == NULL
					    || port->tp->t_iflag & IGNBRK)
						goto cont;
				} else {
					if (port->tp == NULL
					    || port->tp->t_iflag & IGNPAR)
						goto cont;
				}
				if (port->tp->t_state & TS_CAN_BYPASS_L_RINT
				    && (line_status & (LSR_BI | LSR_FE)
					|| port->tp->t_iflag & INPCK))
					recv_data = 0;
			}
			++port->bytes_in;
			if (port->hotchar != 0 && recv_data == port->hotchar)
#if __FreeBSD_version < 500000
				setsofttty();
#else
				swi_sched (mx_fast_ih, 0); 
#endif				
			ioptr = port->iptr;
			if (ioptr >= port->ibufend){
				CE_RECORD(port, CE_INTERRUPT_BUF_OVERFLOW);
			}else {
				if (port->do_timestamp)
					microtime(&port->timestamp);
				++ser_events;
#if __FreeBSD_version < 500000
				schedsofttty();
#else				
				swi_sched(mx_slow_ih, SWI_DELAY); 				
#endif
				
#if 0 /* for testing input latency vs efficiency */
if (port->iptr - port->ibuf == 8)
#if __FreeBSD_version < 500000
	setsofttty();
#else	
	swi_sched (mx_fast_ih, 0); 	
#endif	
#endif
				ioptr[0] = recv_data;
				ioptr[port->ierroff] = line_status;
				port->iptr = ++ioptr;

				if (ioptr == port->ihighwater
				    && port->state & CS_RTS_IFLOW){
					outb((port->iobase + UART_MCR),
					     port->MCR &= ~MCR_RTS);
				}
					   
				/* added by casper */	  
				if (ioptr == port->ihighwater){
					if((port->IsMoxaMustChipFlag) && 
						(port->tp->t_iflag & IXOFF)){
						outb((port->iobase+UART_IER),inb(port->iobase+UART_IER) & ~UART_IER_RDI);
					}
				}
					     
				if (line_status & LSR_OE)
					CE_RECORD(port, CE_OVERRUN);
			}
cont:
			/*
			 * "& 0x7F" is to avoid the gcc-1.40 generating a slow
			 * jump from the top of the loop to here
			 */
			line_status = inb(port->iobase + UART_LSR) & 0x7F;
		}

		/* modem status change? (always check before doing output) */
		modem_status = inb(port->iobase + UART_MSR);
		if (modem_status != port->last_modem_status) {
			if (port->do_dcd_timestamp
			    && !(port->last_modem_status & MSR_DCD)
			    && modem_status & MSR_DCD)
				microtime(&port->dcd_timestamp);

			/*
			 * Schedule high level to handle DCD changes.  Note
			 * that we don't use the delta bits anywhere.  Some
			 * UARTs mess them up, and it's easy to remember the
			 * previous bits and calculate the delta.
			 */
			port->last_modem_status = modem_status;
			if (!(port->state & CS_CHECKMSR)) {
				ser_events += LOTS_OF_EVENTS;
				port->state |= CS_CHECKMSR;
#if __FreeBSD_version < 500000
				setsofttty();
#else				
				swi_sched (mx_fast_ih, 0); 	
#endif	
			}

			/* handle CTS change immediately for crisp flow ctl */
			if (port->state & CS_CTS_OFLOW) {
				if (modem_status & MSR_CTS)
					port->state |= CS_ODEVREADY;
				else
					port->state &= ~CS_ODEVREADY;
			}
		}

		/* output queued and everything ready? */
		if (line_status & LSR_TXRDY
		    && port->state >= (CS_BUSY | CS_TTGO | CS_ODEVREADY)) {
			ioptr = port->obufq.l_head;
			if ((port->tx_fifo_size > 1)  && 
				(port->IsMoxaMustChipFlag || 
				(!(tp->t_iflag & (IXON|IXOFF))))
				/* tx 1 on sw flowcontorl */
				){
				u_int	ocount;

				ocount = port->obufq.l_tail - ioptr;
				if (ocount > port->tx_fifo_size)
					ocount = port->tx_fifo_size;

				port->bytes_out += ocount;
				do{
					/* added by casper, skip xon xoff */
					if((tp->t_iflag & IXOFF) && (port->IsMoxaMustChipFlag)){
						if((*ioptr) == tp->t_cc[VSTART]){
							outb((port->iobase+UART_IER),inb(port->iobase+UART_IER) | UART_IER_RDI);
							ioptr++;
							continue;
						}else if((*ioptr) == tp->t_cc[VSTOP]){
							outb((port->iobase+UART_IER),inb(port->iobase+UART_IER) & ~UART_IER_RDI);
							ioptr++;
							continue;
						}
					}
					outb((port->iobase+UART_TX),*ioptr++);
				}while (--ocount != 0);
			} else {
				/* added by casper, skip xon, xoff */
				if((tp->t_iflag & IXOFF) && (port->IsMoxaMustChipFlag)){
					if((*ioptr) == tp->t_cc[VSTART]){

						outb((port->iobase+UART_IER),inb(port->iobase+UART_IER) | UART_IER_RDI);

						ioptr++;
					}else if((*ioptr) == tp->t_cc[VSTOP]){
						outb((port->iobase+UART_IER),inb(port->iobase+UART_IER) & ~UART_IER_RDI);
						ioptr++;
					}else{
					        outb((port->iobase + UART_TX), *ioptr++);
					        ++port->bytes_out;
					}
				}else{

					outb((port->iobase + UART_TX), *ioptr++);
					++port->bytes_out;
				}
			}
			port->obufq.l_head = ioptr;
/*			
			if (COM_IIR_TXRDYBUG(com->flags)) {
				int_ctl_new = int_ctl | IER_ETXRDY;
			}
*/			
			if (ioptr >= port->obufq.l_tail) {
				struct lbq	*qp;

				qp = port->obufq.l_next;
				qp->l_queued = FALSE;
				qp = qp->l_next;
				if (qp != NULL) {
					port->obufq.l_head = qp->l_head;
					port->obufq.l_tail = qp->l_tail;
					port->obufq.l_next = qp;
				} else {
					/* output just completed */
/*					
					if (COM_IIR_TXRDYBUG(com->flags)) {
						int_ctl_new = int_ctl & ~IER_ETXRDY;
					}
*/					
					port->state &= ~CS_BUSY;
				}
				if (!(port->state & CS_ODONE)) {
					ser_events += LOTS_OF_EVENTS;
					port->state |= CS_ODONE;
#if __FreeBSD_version < 500000
					setsofttty();	
#else					
					swi_sched (mx_fast_ih, 0);	 
#endif					
					/* handle at high level ASAP */
				}
			}
/*			
			if (COM_IIR_TXRDYBUG(com->flags) && (int_ctl != int_ctl_new)) {
				outb(com->intr_ctl_port, int_ctl_new);
			}
*/			
		}

		/* finished? */
		return;
	}
}

#if __FreeBSD_version < 500000 
int mxopen(dev_t dev, int flag, int mode, struct proc *p)
#else
int mxopen(dev_t dev, int flag, int mode, struct thread *p)
#endif
{
	struct port_s	*port;
	int		error;
	int		mynor;
	int		s;
	struct tty	*tp;
	int		unit;

	mynor = minor(dev);
	if (mynor == MXSER_PORTS)
		return 0;

	unit = (mynor & 0x1F) / 8;
 	port = getportinfo(unit, mynor); 	
 	if (port == NULL) {
		return (ENXIO); 		
	}

	if (port->gone) {
		return (ENXIO);
	}

	if (mynor & CONTROL_MASK)
		return (0);

	tp = dev->si_tty = port->tp = ttymalloc(port->tp);
	s = spltty();
	/*
	 * We jump to this label after all non-interrupted sleeps to pick
	 * up any changes of the device state.
	 */
open_top:

	while (port->state & CS_DTR_OFF) {
		error = tsleep(&port->dtr_wait, TTIPRI | PCATCH, "siodtr", 0);
 		port = getportinfo(unit, mynor); 	
 		if (port == NULL) {
			return (ENXIO); 		
		}		
		if (error != 0 || port->gone) {
			goto out;
		}
	}

	if (tp->t_state & TS_ISOPEN) {
		/*
		 * The device is open, so everything has been initialized.
		 * Handle conflicts.
		 */
		if (mynor & CALLOUT_MASK) {
			if (!port->active_out) {
				error = EBUSY;
				goto out;
			}
		} else {
			if (port->active_out) {
				if (flag & O_NONBLOCK) {
					error = EBUSY;
					goto out;
				}
				error =	tsleep(&port->active_out,
					       TTIPRI | PCATCH, "siobi", 0);

 				port = getportinfo(unit, mynor); 	
 				if (port == NULL) {
					return (ENXIO); 		
				}
				if (error != 0 || port->gone)
					goto out;
				goto open_top;
			}
		}
		if (tp->t_state & TS_XCLUDE &&
		    suser(p)) {
			error = EBUSY;
			goto out;
		}
	} else {
		/*
		 * The device isn't open, so there are no conflicts.
		 * Initialize it.  Initialization is done twice in many
		 * cases: to preempt sleeping callin opens if we are
		 * callout, and to complete a callin open after DCD rises.
		 */
		tp->t_oproc = ser_start;
		tp->t_param = ser_param;
		tp->t_stop = ser_stop;
		tp->t_dev = dev;
		tp->t_termios = mynor & CALLOUT_MASK
				? port->it_out : port->it_in;
		(void)ser_mctl(port, TIOCM_DTR | TIOCM_RTS, DMSET);
		port->poll = port->no_irq;
		port->poll_output = port->loses_outints;
		++port->wopeners;
		error = ser_param(tp, &tp->t_termios);
		--port->wopeners;

		if (error != 0) {
			goto out;
		}
		/*
		 * XXX we should goto open_top if comparam() slept.
		 */
			/*
			 * (Re)enable and drain fifos.
			 *
			 * Certain SMC chips cause problems if the fifos
			 * are enabled while input is ready.  Turn off the
			 * fifo if necessary to clear the input.  We test
			 * the input ready bit after enabling the fifos
			 * since we've already enabled them in comparam()
			 * and to handle races between enabling and fresh
			 * input.
			 */
		while (TRUE) {
			outb((port->iobase + UART_FCR), FIFO_RCV_RST | FIFO_XMT_RST
							| port->FIFO);
				/*
				 * XXX the delays are for superstitious
				 * historical reasons.  It must be less than
				 * the character time at the maximum
				 * supported speed (87 usec at 115200 bps
				 * 8N1).  Otherwise we might loop endlessly
				 * if data is streaming in.  We used to use
				 * delays of 100.  That usually worked
				 * because DELAY(100) used to usually delay
				 * for about 85 usec instead of 100.
				 */
			DELAY(50);
			if (!(inb(port->iobase + UART_LSR) & LSR_RXRDY)) {
				break;
			}
			outb((port->iobase + UART_FCR), 0);				
			DELAY(50);
			(void) inb(port->iobase + UART_RX);
		}

		disable_intr();
		(void) inb(port->iobase + UART_LSR);
		(void) inb(port->iobase + UART_RX);
		port->prev_modem_status = port->last_modem_status
		    = inb(port->iobase + UART_MSR);

		outb((port->iobase + UART_IER), IER_ERXRDY | 
				IER_ETXRDY | IER_ERLS | IER_EMSC);
		enable_intr();
		/*
		 * Handle initial DCD.  Callout devices get a fake initial
		 * DCD (trapdoor DCD).  If we are callout, then any sleeping
		 * callin opens get woken up and resume sleeping on "siobi"
		 * instead of "siodcd".
		 */
		/*
		 * XXX `mynor & CALLOUT_MASK' should be
		 * `tp->t_cflag & (SOFT_CARRIER | TRAPDOOR_CARRIER) where
		 * TRAPDOOR_CARRIER is the default initial state for callout
		 * devices and SOFT_CARRIER is like CLOCAL except it hides
		 * the true carrier.
		 */
		if (port->prev_modem_status & MSR_DCD || mynor & CALLOUT_MASK)
#if __FreeBSD_version < 500000
			(*linesw[tp->t_line].l_modem)(tp, 1);
#else			
			(*linesw[tp->t_line]->l_modem)(tp, 1); 			
#endif			
	}
	/*
	 * Wait for DCD if necessary.
	 */

	if (!(tp->t_state & TS_CARR_ON) && !(mynor & CALLOUT_MASK)
	    && !(tp->t_cflag & CLOCAL) && !(flag & O_NONBLOCK)) {
		++port->wopeners;
		error = tsleep(TSA_CARR_ON(tp), TTIPRI | PCATCH, "siodcd", 0);
 		port = getportinfo(unit, mynor); 	
 		if (port == NULL) {
			return (ENXIO); 		
		}		
		--port->wopeners;
		if (error != 0 || port->gone) {
			goto out;
		}
		goto open_top;
	}
#if __FreeBSD_version < 500000
	error =	(*linesw[tp->t_line].l_open)(dev, tp);
#else	
	error =	(*linesw[tp->t_line]->l_open)(dev, tp); 
#endif	
	
	disc_optim(tp, &tp->t_termios, port);
	if (tp->t_state & TS_ISOPEN && mynor & CALLOUT_MASK)
		port->active_out = TRUE;
	mxsettimeout();

out:
	splx(s);
	if (!(tp->t_state & TS_ISOPEN) && port->wopeners == 0) {
		ser_hardclose(port);
	}

	return (error);
}

#if __FreeBSD_version < 500000 
static int mxclose(dev_t dev, int flag, int mode, struct proc *p)
#else
static int mxclose(dev_t dev, int flag, int mode, struct thread *p)
#endif
{
	struct port_s	*port;
	int		minor_no, unit;
	int		s;
	struct tty	*tp;

	minor_no = minor(dev);
	if (minor_no == MXSER_PORTS)
		return 0;
	if (minor_no & CONTROL_MASK)
		return 0;

	unit = (minor_no & 0x1F) / 8;
	port = getportinfo(unit, minor_no);
	if (port == NULL) {
		return (ENODEV);
	}

	tp = port->tp;
	s = spltty();
#if __FreeBSD_version < 500000
	(*linesw[tp->t_line].l_close)(tp, flag);
#else	
	(*linesw[tp->t_line]->l_close)(tp, flag); 	
#endif
	disc_optim(tp, &tp->t_termios, port);
	ser_stop(tp, FREAD | FWRITE);
	ser_hardclose(port);
	
#if __FreeBSD_version < 500000
	ttyclose(tp);
#else
	tty_close(tp); 
#endif
	
	mxsettimeout();
	splx(s);
	if (port->gone) {
		s = spltty();
		if (port->ibuf != NULL)
			free(port->ibuf, M_DEVBUF);
		bzero(tp, sizeof *tp);
		splx(s);
	}
	return 0;
}

static int mxread(dev_t dev, struct uio *uio, int flag)
{
	int		minor_no, unit;
	struct port_s	*port;
	int		rlen;

	minor_no = minor(dev);
	if (minor_no & CONTROL_MASK)
		return (ENODEV);
	unit = (minor_no & 0x1F) / 8;
	port = getportinfo(unit, minor_no);
	if (port == NULL || port->gone)
		return (ENODEV);
#if __FreeBSD_version < 500000
	rlen = (*linesw[port->tp->t_line].l_read)(port->tp, uio, flag);
#else	
	rlen = (*linesw[port->tp->t_line]->l_read)(port->tp, uio, flag); 
#endif	
	
	return rlen;
}

static int mxwrite(dev_t dev, struct uio *uio, int flag)
{
	int		minor_no;
	struct port_s	*port;
	int		unit;

	minor_no = minor(dev);
	if (minor_no & CONTROL_MASK)
		return (ENODEV);
	unit = (minor_no & 0x1F) / 8;
	port = getportinfo(unit, minor_no);
	if (port == NULL || port->gone)
		return (ENODEV);
#if __FreeBSD_version < 500000
	return ((*linesw[port->tp->t_line].l_write)(port->tp, uio, flag));
#else
	return ((*linesw[port->tp->t_line]->l_write)(port->tp, uio, flag)); 
#endif
}

#if __FreeBSD_version < 500000 
static int mxioctl(dev_t dev,u_long cmd,caddr_t data,int flag,struct proc *p)
#else
static int mxioctl(dev_t dev,u_long cmd,caddr_t data,int flag,struct thread *p)
#endif
{
	struct port_s	*port;
	int		error;
	int		minor_no, unit;
	int		s;
	struct tty	*tp;
#if defined(COMPAT_43) || defined(COMPAT_SUNOS)
	u_long		oldcmd;
	struct termios	term;
#endif	

	minor_no = minor(dev);

	if (minor_no == MXSER_PORTS)
		return(mxser_ioctl_special(cmd, data));

	unit = (minor_no & 0x1F) / 8;
	port = getportinfo(unit, minor_no);
	if (port == NULL || port->gone)
		return (ENODEV);
	if (minor_no & CONTROL_MASK) {
		struct termios	*ct;
		switch (minor_no & CONTROL_MASK) {
		case CONTROL_INIT_STATE:
			ct = minor_no & CALLOUT_MASK ? &port->it_out : &port->it_in;
			break;
		case CONTROL_LOCK_STATE:
			ct = minor_no & CALLOUT_MASK ? &port->lt_out : &port->lt_in;
			break;
		default:
			return (ENODEV);
		}
		switch (cmd) {
		case TIOCSETA:
			error = suser(p);
			if (error != 0)
				return (error);
			*ct = *(struct termios *)data;
			return (0);
		case TIOCGETA:
			*(struct termios *)data = *ct;
			return (0);
		case TIOCGETD:
			*(int *)data = TTYDISC;
			return (0);
		case TIOCGWINSZ:
			bzero(data, sizeof(struct winsize));
			return (0);
		default:
			return (ENOTTY);
		}
	}

	tp = port->tp;
	
#if defined(COMPAT_43) || defined(COMPAT_SUNOS)
	term = tp->t_termios;
	oldcmd = cmd;
	error = ttsetcompat(tp, &cmd, data, &term);
	if (error != 0)
		return (error);
	if (cmd != oldcmd)
		data = (caddr_t)&term;
#endif
	if (cmd == TIOCSETA || cmd == TIOCSETAW || cmd == TIOCSETAF) {
		int	cc;
		struct termios *dt = (struct termios *)data;
		struct termios *lt = minor_no & CALLOUT_MASK
				     ? &port->lt_out : &port->lt_in;

		dt->c_iflag = (tp->t_iflag & lt->c_iflag)
			      | (dt->c_iflag & ~lt->c_iflag);
		dt->c_oflag = (tp->t_oflag & lt->c_oflag)
			      | (dt->c_oflag & ~lt->c_oflag);
		dt->c_cflag = (tp->t_cflag & lt->c_cflag)
			      | (dt->c_cflag & ~lt->c_cflag);
		dt->c_lflag = (tp->t_lflag & lt->c_lflag)
			      | (dt->c_lflag & ~lt->c_lflag);
		for (cc = 0; cc < NCCS; ++cc)
			if (lt->c_cc[cc] != 0)
				dt->c_cc[cc] = tp->t_cc[cc];
		if (lt->c_ispeed != 0)
			dt->c_ispeed = tp->t_ispeed;
		if (lt->c_ospeed != 0)
			dt->c_ospeed = tp->t_ospeed;
	}
#if __FreeBSD_version < 500000
	error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, data, flag, p);
#else
	error = (*linesw[tp->t_line]->l_ioctl)(tp, cmd, data, flag, p); 
#endif	
	if (error != ENOIOCTL)
		return (error);
	s = spltty();
	error = ttioctl(tp, cmd, data, flag);
	disc_optim(tp, &tp->t_termios, port);
	if (error != ENOIOCTL) {
		splx(s);
		return (error);
	}
	switch (cmd) {
	case TIOCSBRK:
		outb((port->iobase + UART_LCR), port->LCR |= CFCR_SBREAK);
		break;
	case TIOCCBRK:
		outb((port->iobase + UART_LCR), port->LCR &= ~CFCR_SBREAK);
		break;
	case TIOCSDTR:
		(void)ser_mctl(port, TIOCM_DTR, DMBIS);
		break;
	case TIOCCDTR:
		(void)ser_mctl(port, TIOCM_DTR, DMBIC);
		break;
	/*
	 * XXX should disallow changing MCR_RTS if CS_RTS_IFLOW is set.  The
	 * changes get undone on the next call to comparam().
	 */
	case TIOCMSET:
		(void)ser_mctl(port, *(int *)data, DMSET);
		break;
	case TIOCMBIS:
		(void)ser_mctl(port, *(int *)data, DMBIS);
		break;
	case TIOCMBIC:
		(void)ser_mctl(port, *(int *)data, DMBIC);
		break;
	case TIOCMGET:
		*(int *)data = ser_mctl(port, 0, DMGET);
		break;
	case TIOCMSDTRWAIT:
		/* must be root since the wait applies to following logins */
		error = suser(p);
		if (error != 0) {
			splx(s);
			return (error);
		}
		port->dtr_wait = *(int *)data * hz / 100;
		break;
	case TIOCMGDTRWAIT:
		*(int *)data = port->dtr_wait * 100 / hz;
		break;
	case TIOCTIMESTAMP:
		port->do_timestamp = TRUE;
		*(struct timeval *)data = port->timestamp;
		break;
#if __FreeBSD_version < 500000
	case TIOCDCDTIMESTAMP:
		port->do_dcd_timestamp = TRUE;
		*(struct timeval *)data = port->dcd_timestamp;
		break;
#endif
	default:
		splx(s);
		error = pps_ioctl(cmd, data, &port->pps);
		if (error == ENODEV)
			error = ENOTTY;
		return (error);
	}
	splx(s);

	return (0);
}

static int mxser_ioctl_special(u_long cmd, caddr_t data)
{
	struct mxser_hwconf	*ioctl_hwconf;

	switch (cmd) {
	case MOXA_GET_CONF:
		ioctl_hwconf = (struct mxser_hwconf *)data;
		bcopy(&mxsercfg[0], ioctl_hwconf, sizeof(mxsercfg)); 
		break;
	case MOXA_GET_MAJOR:
#if __FreeBSD_version < 500000
		*(int *)data = CDEV_MAJOR;
#else
		*(int *)data = major(mxser_dev); 		
#endif
		break;
	default:
		return EINVAL;
	}
	return 0;
}

static struct port_s * getportinfo(int unit, int minor_no) 
{
	struct board_s *info_board;
	struct port_s	*port;
	int		portno;
	
	info_board = (struct board_s *)devclass_get_softc(mx_devclass, unit);
	if (info_board == NULL) {
		return (0);
	}
	portno = (minor_no & 0x1F) - unit * 8;
	port = &info_board->port[portno];

	return (port);			
}		

static void ser_start(struct tty *tp)
{
	struct board_s	*info_board;
	struct port_s	*port;
	int		s, unit, myminor, portno;

	myminor = minor(tp->t_dev);
	unit = (myminor & 0x1F) / 8;
	info_board = (struct board_s *)devclass_get_softc(mx_devclass, unit);
	if (info_board == NULL) {
		return;
	}
	portno = (myminor & 0x1F) - unit * 8;
	port = &info_board->port[portno];
	s = spltty();
	disable_intr();
	if (tp->t_state & TS_TTSTOP)
		port->state &= ~CS_TTGO;
	else
		port->state |= CS_TTGO;
	if (tp->t_state & TS_TBLOCK) {
		if (port->MCR & MCR_RTS &&
			port->state & CS_RTS_IFLOW)
			outb((port->iobase + UART_MCR), port->MCR &= ~MCR_RTS);

		/* added by casper */
		if((port->IsMoxaMustChipFlag) && (tp->t_iflag & IXOFF)){
			char ret;
			ret = inb(port->iobase+UART_IER);
			if(ret & UART_IER_RDI){
				outb((port->iobase+UART_IER),ret & ~UART_IER_RDI);
			}
		}

	} else {
		if (!(port->MCR & MCR_RTS) && port->iptr < port->ihighwater
			&& port->state & CS_RTS_IFLOW)
			outb((port->iobase + UART_MCR), port->MCR |= MCR_RTS);

		/* added by casper */
		if((port->IsMoxaMustChipFlag) && (tp->t_iflag & IXOFF)){
			char ret;
			ret = inb(port->iobase+UART_IER);
			if(!(ret & UART_IER_RDI) && (port->iptr < port->ihighwater)){
				outb((port->iobase+UART_IER),ret | UART_IER_RDI);
			}
		}


	}


	enable_intr();
	if (tp->t_state & (TS_TIMEOUT | TS_TTSTOP)) {
		ttwwakeup(tp);
		splx(s);
		return;
	}
	if (tp->t_outq.c_cc != 0) {
		struct lbq	*qp;
		struct lbq	*next;

		if (!port->obufs[0].l_queued) {
			port->obufs[0].l_tail = port->obuf1 +
			q_to_b(&tp->t_outq, port->obuf1, sizeof port->obuf1); 
			port->obufs[0].l_next = NULL;
			port->obufs[0].l_queued = TRUE;
			disable_intr();
			if (port->state & CS_BUSY) {
				qp = port->obufq.l_next;
				while((next = qp->l_next) != NULL)
					qp = next;
				qp->l_next = &port->obufs[0];
			} else {
				port->obufq.l_head = port->obufs[0].l_head;
				port->obufq.l_tail = port->obufs[0].l_tail;
				port->obufq.l_next = &port->obufs[0];
				port->state |= CS_BUSY;
			}
			enable_intr();
		}
		if (tp->t_outq.c_cc != 0 && !port->obufs[1].l_queued) {
			port->obufs[1].l_tail = port->obuf2 + q_to_b(&tp->t_outq, 
			port->obuf2, sizeof port->obuf2);
			port->obufs[1].l_next = NULL;
			port->obufs[1].l_queued = TRUE;
			disable_intr();
			if (port->state & CS_BUSY) {
				qp = port->obufq.l_next;
				while ((next = qp->l_next) != NULL)
					qp = next;
				qp->l_next = &port->obufs[1];
			} else {
				port->obufq.l_head = port->obufs[1].l_head;
				port->obufq.l_tail = port->obufs[1].l_tail;
				port->obufq.l_next = &port->obufs[1];
				port->state |= CS_BUSY;
			}
			enable_intr();
		}
		tp->t_state |= TS_BUSY;
	}
	disable_intr();
	if (port->state >= (CS_BUSY | CS_TTGO))
		mxintr1(port);
	enable_intr();
	ttwwakeup(tp);
	splx(s);
}

static void mxsettimeout(void)
{
	struct board_s	*info_board;
	struct port_s	*port;
	u_char		someopen;
	int		unit, number_of_ports, i;

	/*
	 * Set our timeout period to 1 second if no polled devices are open.
	 * Otherwise set it to max(1/200, 1/hz).
	 * Enable timeouts iff some device is open.
	 */
	untimeout(ser_wakeup, (void *)NULL, mx_timeout_handle);
	mx_timeout = hz;
	someopen = FALSE;
	for (unit = 0; unit < mx_numunits; ++unit) {
		info_board = (struct board_s *)devclass_get_softc(mx_devclass, unit);
		if (info_board == NULL) {
			return;
		}		
		number_of_ports = info_board->number_ports;
		port = &info_board->port[0];
		for (i=0; i<number_of_ports; i++, port++) {
			if (port != NULL && port->tp != NULL
		    	&& port->tp->t_state & TS_ISOPEN && !port->gone) {
				someopen = TRUE;
				if (port->poll || port->poll_output) {
					mx_timeout = hz > 200 ? hz / 200 : 1;
					break;
				}
			}
		}
	}
	if (someopen) {
		mx_timeouts_until_log = hz / mx_timeout;
		mx_timeout_handle = timeout(ser_wakeup, (void *)NULL,
					     mx_timeout);
	} else {
		/* Flush error messages, if any. */
		mx_timeouts_until_log = 1;
		ser_wakeup((void *)NULL);
		untimeout(ser_wakeup, (void *)NULL, mx_timeout_handle);
	}
}

static void ser_wakeup(void *chan)
{
	struct board_s	*info_board;
	struct port_s	*port;
	int		unit, number_of_ports, i;

	mx_timeout_handle = timeout(ser_wakeup, (void *)NULL, mx_timeout);

	/*
	 * Recover from lost output interrupts.
	 * Poll any lines that don't use interrupts.
	 */
	for (unit = 0; unit < mx_numunits; ++unit) {
		info_board = (struct board_s *)devclass_get_softc(mx_devclass, unit);
		if (info_board == NULL) {
			return;
		}		
		number_of_ports = info_board->number_ports;
		port = &info_board->port[0];
		for (i=0; i<number_of_ports; i++, port++) {
			if (port != NULL && !port->gone
		    	&& (port->state >= (CS_BUSY | CS_TTGO) || port->poll)) {
				disable_intr();
				mxintr1(port);
				enable_intr();
			}
		}
	}

	/*
	 * Check for and log errors, but not too often.
	 */
	if (--mx_timeouts_until_log > 0)
		return;
	mx_timeouts_until_log = hz / mx_timeout;
	for (unit = 0; unit < mx_numunits; ++unit) {
		int	errnum;
		info_board = (struct board_s *)devclass_get_softc(mx_devclass, unit);
		if (info_board == NULL) {
			return;
		}		
		number_of_ports = info_board->number_ports;
		port = &info_board->port[0];
		for (i=0; i<number_of_ports; i++, port++) {
			if (port == NULL)
				continue;
			if (port->gone)
				continue;
			for (errnum = 0; errnum < CE_NTYPES; ++errnum) {
				u_int	delta;
				u_long	total;

				disable_intr();
				delta = port->delta_error_counts[errnum];
				port->delta_error_counts[errnum] = 0;
				enable_intr();
				if (delta == 0)
					continue;
				total = port->error_counts[errnum] += delta;
				uprintf("mx%d: %u more %s%s (total %lu)\n",
			    		unit, delta, error_desc[errnum],
			    		delta == 1 ? "" : "s", total);
			}			    		
		}
	}
}

static int ser_param(struct tty *tp, struct termios *t)
{
	u_int		cfcr;
	int		cflag,iflag;
	struct board_s	*info_board;
	struct port_s	*port;
	u_int		divisor;
	u_char		dlbh;
	u_char		dlbl;
	int		s;
	int		unit, minor_no, portno, uart_type;

	minor_no = minor(tp->t_dev);
	unit = (minor_no & 0x1F) / 8;
	
	info_board = (struct board_s *)devclass_get_softc(mx_devclass, unit);
	if (info_board == NULL) {
		return (0);
	}
	uart_type = info_board->uart_type;
	portno = (minor_no & 0x1F) - unit * 8;
	port = &info_board->port[portno];	

	if (port == NULL) {
		return (ENODEV);
	}

	/* do historical conversions */
	if (t->c_ispeed == 0)
		t->c_ispeed = t->c_ospeed;

	/* check requested parameters */
	if (t->c_ospeed == 0)
		divisor = 0;
	else {
		if (t->c_ispeed != t->c_ospeed)
			return (EINVAL);
		divisor = mxdivisor(info_board->rclk, t->c_ispeed);
		if (divisor == 0)
			return (EINVAL);
	}

	/* parameters are OK, convert them to the com struct and the device */
	s = spltty();
	if (divisor == 0)
		(void)ser_mctl(port, TIOCM_DTR, DMBIC);	/* hang up line */
	else
		(void)ser_mctl(port, TIOCM_DTR, DMBIS);
	cflag = t->c_cflag;
	switch (cflag & CSIZE) {
	case CS5:
		cfcr = CFCR_5BITS;
		break;
	case CS6:
		cfcr = CFCR_6BITS;
		break;
	case CS7:
		cfcr = CFCR_7BITS;
		break;
	default:
		cfcr = CFCR_8BITS;
		break;
	}
	if (cflag & PARENB) {
		cfcr |= CFCR_PENAB;
		if (!(cflag & PARODD))
			cfcr |= CFCR_PEVEN;
	}
	if (cflag & CSTOPB)
		cfcr |= CFCR_STOPB;

	if (divisor != 0) {
		/*
		 * Use a fifo trigger level low enough so that the input
		 * latency from the fifo is less than about 16 msec and
		 * the total latency is less than about 30 msec.  These
		 * latencies are reasonable for humans.  Serial comms
		 * protocols shouldn't expect anything better since modem
		 * latencies are larger.
		 *
		 * Interrupts can be held up for long periods of time
		 * due to inefficiencies in other parts of the kernel,
		 * certain video cards, etc.  Setting the FIFO trigger
		 * point to MEDH instead of HIGH gives us 694uS of slop
		 * (8 character times) instead of 173uS (2 character times)
		 * @ 115200 bps.
		 */
		port->FIFO = t->c_ospeed <= 4800
				  ? FIFO_ENABLE : FIFO_ENABLE | FIFO_RX_MEDH;
		outb((port->iobase + UART_FCR), port->FIFO);				  
	}

	port->FIFO &= ~UART_FCR_TRIGGER_MASK;
	if(port->IsMoxaMustChipFlag){
		port->FIFO |= UART_FCR_TRIGGER_14;
		SET_MOXA_MUST_FIFO_VALUE(port); 
	}else{
		if(t->c_iflag & (IXON|IXOFF)){
			port->FIFO |= UART_FCR_TRIGGER_1;
		}else{
			switch(port->rx_trigger){
			case 14:
				port->FIFO |= UART_FCR_TRIGGER_14;
				break;
			case 8:
				port->FIFO |= UART_FCR_TRIGGER_8;
				break;
			case 4:
				port->FIFO |= UART_FCR_TRIGGER_4;
				break;
			case 1:
				port->FIFO |= UART_FCR_TRIGGER_1;
			break;
			}
		}
		outb((port->iobase + UART_FCR), port->FIFO);
	}

	/* set the software flow control flag */
	disable_intr();
	iflag = t->c_iflag;
	if(port->IsMoxaMustChipFlag){
		SET_MOXA_MUST_XON1_VALUE(port->iobase,tp->t_cc[VSTART]);
		SET_MOXA_MUST_XOFF1_VALUE(port->iobase,tp->t_cc[VSTOP]);
		if(iflag & IXON){
			ENABLE_MOXA_MUST_TX_SOFTWARE_FLOW_CONTROL(port->iobase);
		}else{
			DISABLE_MOXA_MUST_TX_SOFTWARE_FLOW_CONTROL(port->iobase);
		}
		if(iflag & IXOFF){
			ENABLE_MOXA_MUST_RX_SOFTWARE_FLOW_CONTROL(port->iobase);
		}else{
			DISABLE_MOXA_MUST_RX_SOFTWARE_FLOW_CONTROL(port->iobase);
		}
		if(iflag & IXANY){
			port->MCR |= MOXA_MUST_MCR_XON_ANY;
			ENABLE_MOXA_MUST_XON_ANY_FLOW_CONTROL(port->iobase);
		}else{
			port->MCR &= ~MOXA_MUST_MCR_XON_ANY;
			DISABLE_MOXA_MUST_XON_ANY_FLOW_CONTROL(port->iobase);
		}
	}
	enable_intr();
	/*  */

	/*
	 * This returns with interrupts disabled so that we can complete
	 * the speed change atomically.  Keeping interrupts disabled is
	 * especially important while com_data is hidden.
	 */
	(void) mxsetwater(port, t->c_ispeed);

	if (divisor != 0) {
		outb((port->iobase + UART_LCR), cfcr | CFCR_DLAB);
		/*
		 * Only set the divisor registers if they would change,
		 * since on some 16550 incompatibles (UMC8669F), setting
		 * them while input is arriving them loses sync until
		 * data stops arriving.
		 */
		dlbl = divisor & 0xFF;
		outb((port->iobase + UART_DLL), dlbl);
		dlbh = divisor >> 8;
		outb((port->iobase + UART_DLM), dlbh);
	}

	port->LCR = cfcr;
	outb((port->iobase + UART_LCR), cfcr);

	if (!(tp->t_state & TS_TTSTOP))
		port->state |= CS_TTGO;

	port->MCR &= ~UART_MCR_AFE;
	if (cflag & CRTS_IFLOW) {

		port->state |= CS_RTS_IFLOW;
		if (uart_type == PORT_16550A)
			port->MCR |= UART_MCR_AFE;
		/*
		 * If CS_RTS_IFLOW just changed from off to on, the change
		 * needs to be propagated to MCR_RTS.  This isn't urgent,
		 * so do it later by calling comstart() instead of repeating
		 * a lot of code from comstart() here.
		 */
	} else if (port->state & CS_RTS_IFLOW) {
		port->state &= ~CS_RTS_IFLOW;
		/*
		 * CS_RTS_IFLOW just changed from on to off.  Force MCR_RTS
		 * on here, since comstart() won't do it later.
		 */
		outb((port->iobase + UART_MCR), port->MCR |= MCR_RTS);
	}
	/* below added by george_liu 08-07-2003 */
	outb((port->iobase + UART_MCR), port->MCR);
	/* above added by george_liu */

	/*
	 * Set up state to handle output flow control.
	 * XXX - worth handling MDMBUF (DCD) flow control at the lowest level?
	 * Now has 10+ msec latency, while CTS flow has 50- usec latency.
	 */
	port->state |= CS_ODEVREADY;
	port->state &= ~CS_CTS_OFLOW;
	if (cflag & CCTS_OFLOW) {
		port->state |= CS_CTS_OFLOW;
		if (!(port->last_modem_status & MSR_CTS))
			port->state &= ~CS_ODEVREADY;
		if (uart_type == PORT_16550A)
			port->MCR |= UART_MCR_AFE;
		outb((port->iobase + UART_MCR), port->MCR);
	}

	outb((port->iobase + UART_LCR), port->LCR);

	/* XXX shouldn't call functions while intrs are disabled. */
	disc_optim(tp, t, port);
	/*
	 * Recover from fiddling with CS_TTGO.  We used to call siointr1()
	 * unconditionally, but that defeated the careful discarding of
	 * stale input in sioopen().
	 */
	if (port->state >= (CS_BUSY | CS_TTGO))
		mxintr1(port);

	enable_intr();
	splx(s);
	ser_start(tp);
	if (port->ibufold != NULL) {
		free(port->ibufold, M_DEVBUF);
		port->ibufold = NULL;
	}

	return (0);
}

static void ser_stop(struct tty *tp, int rw)
{
	int		minor_no, unit;
	struct port_s	*port;
	
	minor_no = minor(tp->t_dev);
	unit = (minor_no & 0x1F) / 8;
	port = getportinfo(unit, minor_no); 
	if (port == NULL || port->gone) {
		return;
	}
	disable_intr();
	if (rw & FWRITE) {
		outb((port->iobase + UART_FCR), FIFO_XMT_RST | port->FIFO);
		port->obufs[0].l_queued = FALSE;
		port->obufs[1].l_queued = FALSE;
		if (port->state & CS_ODONE)
			ser_events -= LOTS_OF_EVENTS;
		port->state &= ~(CS_ODONE | CS_BUSY);
		port->tp->t_state &= ~TS_BUSY;
	}
	if (rw & FREAD) {
		outb((port->iobase + UART_FCR), FIFO_RCV_RST | port->FIFO);
		ser_events -= (port->iptr - port->ibuf);
		port->iptr = port->ibuf;
	}
	enable_intr();
	ser_start(tp);
}

static int ser_mctl(struct port_s *port, int bits, int how)
{
	int	mcr;
	int	msr;

	if (how == DMGET) {
		bits = TIOCM_LE;	/* XXX - always enabled while open */
		mcr = port->MCR;
		if (mcr & MCR_DTR)
			bits |= TIOCM_DTR;
		if (mcr & MCR_RTS)
			bits |= TIOCM_RTS;
		msr = port->prev_modem_status;
		if (msr & MSR_CTS)
			bits |= TIOCM_CTS;
		if (msr & MSR_DCD)
			bits |= TIOCM_CD;
		if (msr & MSR_DSR)
			bits |= TIOCM_DSR;
		/*
		 * XXX - MSR_RI is naturally volatile, and we make MSR_TERI
		 * more volatile by reading the modem status a lot.  Perhaps
		 * we should latch both bits until the status is read here.
		 */
		if (msr & (MSR_RI | MSR_TERI))
			bits |= TIOCM_RI;
		return (bits);
	}
	mcr = 0;
	if (bits & TIOCM_DTR)
		mcr |= MCR_DTR;
	if (bits & TIOCM_RTS)
		mcr |= MCR_RTS;
	if (port->gone)
		return(0);
	disable_intr();
	switch (how) {
	case DMSET:
		port->MCR = mcr | (port->MCR & MCR_IENABLE);
		break;
	case DMBIS:
		port->MCR |= mcr;
		break;
	case DMBIC:
		port->MCR &= ~mcr;
		break;
	}
	outb((port->iobase+UART_MCR), port->MCR);	
	enable_intr();
	return (0);
}

static void ser_hardclose(struct port_s *port)
{
	int		s;
	struct tty	*tp;

	s = spltty();
	port->poll = FALSE;
	port->poll_output = FALSE;
	port->do_timestamp = FALSE;
	port->do_dcd_timestamp = FALSE;
	port->pps.ppsparam.mode = 0;
	outb((port->iobase + UART_LCR), port->LCR &= ~CFCR_SBREAK);
	tp = port->tp;

	outb((port->iobase + UART_IER), 0);
	if (tp->t_cflag & HUPCL
		    /*
		     * XXX we will miss any carrier drop between here and the
		     * next open.  Perhaps we should watch DCD even when the
		     * port is closed; it is not sufficient to check it at
		     * the next open because it might go up and down while
		     * we're not watching.
		     */
	    || (!port->active_out
	        && !(port->prev_modem_status & MSR_DCD)
	        && !(port->it_in.c_cflag & CLOCAL))
	    || !(tp->t_state & TS_ISOPEN)) {
		(void)ser_mctl(port, TIOCM_DTR, DMBIC);
		if (port->dtr_wait != 0 && !(port->state & CS_DTR_OFF)) {
			timeout(mxdtrwakeup, port, port->dtr_wait);
			port->state |= CS_DTR_OFF;
		}
	}

		/*
		 * Disable fifos so that they are off after controlled
		 * reboots.  Some BIOSes fail to detect 16550s when the
		 * fifos are enabled.
		 */
	outb((port->iobase + UART_FCR), 0);		 
	port->active_out = FALSE;
	wakeup(&port->active_out);
	wakeup(TSA_CARR_ON(tp));	/* restart any wopeners */
	splx(s);
}

static void mxdtrwakeup(void *chan)
{
	struct port_s	*port;

	port = (struct port_s *)chan;
	port->state &= ~CS_DTR_OFF;
	wakeup(&port->dtr_wait);
}

#if __FreeBSD_version < 500000
static void mxpoll()
#else
static void mxpoll(void *unused) 
#endif
{
	int		unit,i;

	if (ser_events == 0) {
		return;
	}		

	do {
		for (unit = 0; unit < mx_numunits; ++unit) {
			struct board_s	*info_board;
			struct port_s	*port;
			int		incc;
			struct tty	*tp;

			info_board =(struct board_s *)devclass_get_softc(mx_devclass, unit);
			if (info_board == NULL) {
				continue;
			}

          	port = &info_board->port[0];		
            	for(i=0; i<info_board->number_ports; i++, port++) {
			tp = port->tp;
			if (tp == NULL || port->gone) {
			/*
			 * Discard any events related to never-opened or
			 * going-away devices.
			 */
				disable_intr();
				incc = port->iptr - port->ibuf;
				port->iptr = port->ibuf;
				if (port->state & CS_CHECKMSR) {
					incc += LOTS_OF_EVENTS;
					port->state &= ~CS_CHECKMSR;
				}
				ser_events -= incc;
				enable_intr();
				continue;
			}
			if (port->iptr != port->ibuf) {
				disable_intr();
				mxinput(port);
				enable_intr();
			}
			if (port->state & CS_CHECKMSR) {
				u_char	delta_modem_status;
				disable_intr();
				delta_modem_status = port->last_modem_status
				     ^ port->prev_modem_status;
				port->prev_modem_status = port->last_modem_status;
				ser_events -= LOTS_OF_EVENTS;
				port->state &= ~CS_CHECKMSR;
				enable_intr();
				if (delta_modem_status & MSR_DCD)
#if __FreeBSD_version < 500000
					(*linesw[tp->t_line].l_modem)
#else
					(*linesw[tp->t_line]->l_modem) 
#endif					
					(tp, port->prev_modem_status & MSR_DCD);
			}
			if (port->state & CS_ODONE) {
				disable_intr();
				ser_events -= LOTS_OF_EVENTS;
				port->state &= ~CS_ODONE;
				enable_intr();
				if (!(port->state & CS_BUSY)
		    		&& !(port->extra_state & CSE_BUSYCHECK)) {
					timeout(mxbusycheck, port, hz / 100);
					port->extra_state |= CSE_BUSYCHECK;
				}
#if __FreeBSD_version < 500000
				(*linesw[tp->t_line].l_start)(tp);
#else
				(*linesw[tp->t_line]->l_start)(tp); 
#endif
			}
			if (ser_events == 0)
				break;
		} /* i */
		} /* unit */
	} while(ser_events >= LOTS_OF_EVENTS);	
}

static void mxbusycheck(void *chan)
{
	struct port_s	*port;
	int		s;

	port = (struct port_s *)chan;

	/*
	 * Clear TS_BUSY if low-level output is complete.
	 * spl locking is sufficient because siointr1() does not set CS_BUSY.
	 * If siointr1() clears CS_BUSY after we look at it, then we'll get
	 * called again.  Reading the line status port outside of siointr1()
	 * is safe because CS_BUSY is clear so there are no output interrupts
	 * to lose.
	 */
	s = spltty();
	if (port->state & CS_BUSY)
		port->extra_state &= ~CSE_BUSYCHECK;	/* False alarm. */
	else if ((inb(port->iobase + UART_LSR) & (LSR_TSRE | LSR_TXRDY))
	    == (LSR_TSRE | LSR_TXRDY)) {
		port->tp->t_state &= ~TS_BUSY;
		ttwwakeup(port->tp);
		port->extra_state &= ~CSE_BUSYCHECK;
	} else
		timeout(mxbusycheck, port, hz / 100);
	splx(s);
}

static u_int mxdivisor(u_long rclk, speed_t speed)
{
	long	actual_speed;
	u_int	divisor;
	int		error;

	if (speed == 0 || speed > (ULONG_MAX-1) / 8) {
		return 0;
	}
	divisor = (rclk / (8UL * speed) + 1) / 2;
	if (divisor == 0 || divisor >= 65536) {
		return 0;
	}
	actual_speed = rclk / (16UL * divisor);

	error = ((actual_speed - (long)speed) * 2000 / (long)speed + 1) / 2;

	if (error < -30 || error > 30) {
		return 0;
	}
	return divisor;
}

static void disc_optim(struct tty *tp, struct termios *t, struct port_s *port)
{
	unsigned long	swflag;
	
	swflag = IXON;
	/* onchip sw flow control by pass l_rint*/
	if(port->IsMoxaMustChipFlag)
		swflag = 0;
	
	if (!(t->c_iflag & (ICRNL | IGNCR | IMAXBEL | INLCR | ISTRIP | swflag))
		&& (!(t->c_iflag & BRKINT) || (t->c_iflag & IGNBRK))
		&& (!(t->c_iflag & PARMRK)
		|| (t->c_iflag & (IGNPAR | IGNBRK)) == (IGNPAR | IGNBRK))
		&& !(t->c_lflag & (ECHO | ICANON | IEXTEN | ISIG | PENDIN))
#if __FreeBSD_version < 500000
		&& linesw[tp->t_line].l_rint == ttyinput){
#else
		&& linesw[tp->t_line]->l_rint == ttyinput){ 
#endif	
		tp->t_state |= TS_CAN_BYPASS_L_RINT;
	}else{
		tp->t_state &= ~TS_CAN_BYPASS_L_RINT;
	}
}


static int mxser_get_ISA_conf(int cap, struct mxser_hwconf *hwconf)
{
	int				id, i, bits;
	unsigned short	regs[16], irq;
	unsigned char	scratch, scratch2;

	id = mxser_read_register(cap, regs);
	if (id == C168_ASIC_ID) {
		hwconf->board_type = MXSER_BOARD_C168_ISA;
		hwconf->ports = 8;
	} else if (id == C104_ASIC_ID) {
		hwconf->board_type = MXSER_BOARD_C104_ISA;
		hwconf->ports = 4;
	}else if (id == C102_ASIC_ID){
	    	hwconf->board_type = MXSER_BOARD_C102_ISA;
	    	hwconf->ports = 2;
	}else if (id == CI132_ASIC_ID){
	    	hwconf->board_type = MXSER_BOARD_CI132;
	    	hwconf->ports = 2;
	}else if (id == CI134_ASIC_ID){
	    	hwconf->board_type = MXSER_BOARD_CI134;
	    	hwconf->ports = 4;
	}else if (id == CI104J_ASIC_ID){
	    	hwconf->board_type = MXSER_BOARD_CI104J;
	    	hwconf->ports = 4;		
	} else
		return 0;

	irq = 0;
	if (hwconf->ports == 2) {
	    irq = regs[9] & 0xF000;
	    irq = irq | (irq>>4);
	    if (irq != (regs[9] & 0xFF00)) 
	        return(MXSER_ERR_IRQ_CONFLIT);
	} else if (hwconf->ports == 4) {
		irq = regs[9] & 0xF000;
		irq = irq | (irq>>4);
		irq = irq | (irq>>8);
		if (irq != regs[9])
			return MXSER_ERR_IRQ_CONFLIT;
	} else if (hwconf->ports == 8) {
		irq = regs[9] & 0xF000;
		irq = irq | (irq>>4);
		irq = irq | (irq>>8);
		if ((irq != regs[9]) || (irq != regs[10]))
			return MXSER_ERR_IRQ_CONFLIT;
	}
 
	if (!irq) {
		return MXSER_ERR_IRQ;
	}
	hwconf->irq = ((int)(irq & 0xF000) >> 12);

	for (i=0; i<8; i++) {
		hwconf->ioaddr[i] = (int)regs[i + 1] & 0xFFF8;
	}
	if ((regs[12] & 0x80) == 0) {
		return MXSER_ERR_VECTOR;
	}
	hwconf->vector = (int)regs[11];
	if (id == 1)
		hwconf->vector_mask = 0x00FF;
	else 
		hwconf->vector_mask = 0x000F;
	for (i=7, bits=0x0100; i>=0; i--, bits <<= 1) {
		if (regs[12] & bits)
			hwconf->baud_base[i] = 921600;
		else
			hwconf->baud_base[i] = 115200;
	}
	scratch2 = inb(cap + UART_LCR) & (~UART_LCR_DLAB);
	outb(cap + UART_LCR, scratch2 | UART_LCR_DLAB);
	outb(cap + UART_EFR, 0);
	outb(cap + UART_LCR, scratch2);
	outb(cap + UART_FCR, UART_FCR_ENABLE_FIFO);
	scratch = inb(cap + UART_IIR);
	if (scratch & 0xC0) {
		hwconf->uart_type = PORT_16550A;
	} else {
		hwconf->uart_type = PORT_16450;
	}
	if (id == 1)
		hwconf->ports = 8;
	else 
		hwconf->ports = 4;
		
	return hwconf->ports;
}

static int mxser_read_register(int port, unsigned short *regs)
{
	int				i, k, value, id;
	unsigned int	j;

	id = mxser_program_mode(port);
	if (id < 0)
		return id;
	for (i=0; i<14; i++) {
		k = (i & 0x3f) | 0x180;
		for (j=0x100; j>0; j>>=1) {
			outb(port, CHIP_CS);
			if (k & j) {
				outb(port, CHIP_CS | CHIP_DO);
				outb(port, CHIP_CS | CHIP_DO | CHIP_SK);
			} else {
				outb(port, CHIP_CS);
				outb(port, CHIP_CS | CHIP_SK);
			}
		}
		(void)inb(port);
		value = 0;
		for (k=0, j=0x8000; k<16; k++, j>>=1) {
			outb(port, CHIP_CS);
			outb(port, CHIP_CS | CHIP_SK);
			if (inb(port) & CHIP_DI)
				value |= j;
		}
		regs[i] = value;
		outb(port, 0);
	}
	mxser_normal_mode(port);
	return id;
}

static int mxser_program_mode(int port)
{
	int id, i, j, n;

disable_intr();
	outb(port, 0);
	outb(port, 0);
	outb(port, 0);
	(void)inb(port);
	(void)inb(port);
	outb(port, 0);
	(void)inb(port);
enable_intr();

	id = inb(port+1) & 0x1f;
	if ((id != C168_ASIC_ID) &&
		(id != C104_ASIC_ID) &&
		(id != C102_ASIC_ID) && 
		(id != CI132_ASIC_ID) && 
		(id != CI134_ASIC_ID) && 
		(id != CI104J_ASIC_ID) )		
		return -1;

	for(i=0,j=0; i<4; i++) {
		n = inb(port + 2);
		if (n == 'M') {
			j = 1;
		} else if ( (j == 1) && (n == 1) ) {
			j = 2;
			break;
		} else
			j = 0;
	} 
	if (j != 2)
		id = -2;

	return id;
}

static void mxser_normal_mode(int port)
{
	int i, n;

	outb(port + 1, 0xA5);
	outb(port + 3, 0x80);
	outb(port + 0, 12);
	outb(port + 1, 0);
	outb(port + 3, 0x03);
	outb(port + 4, 0x13);
	for (i=0; i<16; i++) {
		n = inb(port + 5);
		if ((n & 0x61) == 0x60)
			break;
		if ((n & 1) == 1)
			(void)inb(port);
	}
	outb(port + 4, 0x00);
}
