/*
  Copyright (C) MOXA Inc. All rights reserved.
  This software is distributed under the terms of the
  MOXA License.  See the file COPYING-MOXA for details.
*/

/**
 * @file mxDev.h -- interface of the Moxa proprietary device library
 *
 * @version 1.0.0.0 - 2013/07/05
 *
 * @author holsety.chen@moxa.com
 */

#ifndef _MXDEV_H_
#define _MXDEV_H_

#ifdef MXDEV_EXPORTS
#define MXDEVLIB_API __declspec(dllexport)
#else
#define MXDEVLIB_API __declspec(dllimport)
#endif

#ifdef __cplusplus    // If used by C++ code, 
extern "C" {          // we need to export the C interface
#endif

//
// Enumeration serial interface mode for SUPERIO serial interface
enum _SERIALIF_MODE_
{
    SIF_MODE_RS232 = 0,
    SIF_MODE_RS485_2W,
    SIF_MODE_RS422,
    SIF_MODE_RS485_4W,

	SIF_MODE_UNKNOWN,
};

//==============================================================================
//
// DIO functions
//

/**
 * Open DIO port
 * @param  None
 * @return Pointer to DIO handle. Return -1 on failure.
 */
MXDEVLIB_API HANDLE mxdgio_open(void);

/**
 * Close DIO port
 * @param  [in] fd - the handle
 * @return None
 */
MXDEVLIB_API void mxdgio_close(HANDLE fd);

/**
 * Set the value HIGH to DO port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxdgio_set_output_signal_high(HANDLE fd, unsigned int port_no);

/**
 * Set the value LOW to DO port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxdgio_set_output_signal_low(HANDLE fd, unsigned int port_no);

/**
 * Set the value (HIGH or LOW) to DO port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @param  [in] data - the port data, 1:HIGH, 0:LOW
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxdgio_set_output_signal(HANDLE fd, unsigned int port_no, unsigned int data);

/**
 * Get the value (HIGH or LOW) from DI port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @return 1 indicates HIGH, 0 indicates LOW, otherwise the function fails
 */
MXDEVLIB_API int mxdgio_get_input_signal(HANDLE fd, int port_no);

/**
 * Get the value (HIGH or LOW) from DO port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @return 1 indicates HIGH, 0 indicates LOW, otherwise the function fails
 */
MXDEVLIB_API int mxdgio_get_output_signal(HANDLE fd, int port_no);

//==============================================================================
//
// GPIO (LED) functions
//

/**
 * Open GPIO(LED) port
 * @param  None
 * @return Pointer to GPIO(LED) handle. Return -1 on failure.
 */
MXDEVLIB_API HANDLE mxgpio_open(void);

/**
 * Close GPIO(LED) port
 * @param  [in] fd - the handle
 * @return None
 */
MXDEVLIB_API void mxgpio_close(HANDLE fd);

/**
 * Set GPIO(LED) port mode
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @param  [in] mode - 1:Input, 0:Output
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxgpio_set_mode(HANDLE fd, unsigned int port_no, int mode);

/**
 * Get GPIO(LED) port mode
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @return 1 indicates Input, 0 indicates Output, otherwise the function fails
 */
MXDEVLIB_API int mxgpio_get_mode(HANDLE fd, unsigned int port_no);

/**
 * Set the value (HIGH or LOW) to GPIO(LED) port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @param  [in] data - the port data, 1:HIGH, 0:LOW
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxgpio_set_data(HANDLE fd, unsigned int port_no, int data);

/**
 * Get the value (HIGH or LOW) to GPIO(LED) port
 * @param  [in] fd - the handle
 * @param  [in] port_no - the port number (started from 0)
 * @return 1 indicates HIGH, 0 indicates LOW, otherwise the function fails
 */
MXDEVLIB_API int mxgpio_get_data(HANDLE fd, unsigned int port_no);

//==============================================================================
//
// UART mode functions
//

/**
 * Open serial interface mode driver for SUPERIO
 * @param  None
 * @return Pointer to serial mode handle. Return -1 on failure.
 */
MXDEVLIB_API HANDLE mxsp_open();

/**
 * Close serial interface mode driver
 * @param  [in] fd - the handle
 * @return None
 */
MXDEVLIB_API void mxsp_close(HANDLE fd);

/**
 * Get serial interface mode
 * @param  [in] fd - the handle
 * @param  [in] port_index - port index
 * @return 0 - RS232 Mode, 1 - RS485-2W Mode, 2 - RS422 Mode, 3 - RS485-4W Mode
 */
MXDEVLIB_API int mxsp_get_interface(HANDLE fd, int port_index);

/**
 * Set serial interface mode
 * @param  [in] fd - the handle
 * @param  [in] port_index - port index
 * @param  [in] 0 - RS232 Mode, 1 - RS485-2W Mode, 2 - RS422 Mode, 3 - RS485-4W Mode
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxsp_set_interface(HANDLE fd, int port_index, int port_interface);

//==============================================================================
//
// Watchdog functions
//

/**
 * Open Watchdog
 * @param  [in] time - initial refresh time, the unit is second
 * @return Pointer to Watchdog handle. Return -1 on failure.
 */
MXDEVLIB_API PVOID mxwdg_open(unsigned long time);

/**
 * Refresh the watchdog timer
 * @param  [in] fd - the handle
 * @return 0 on success, otherwise the function fails
 */
MXDEVLIB_API int mxwdg_refresh(PVOID fd);

/**
 * Close Watchdog
 * @param  [in] fd - the handle
 * @return None
 */
MXDEVLIB_API void mxwdg_close(PVOID fd);

#ifdef __cplusplus    // If used by C++ code, 
}
#endif

#endif // _MXETHCAPLIB_H_
