/*
 * Copyright (C) MOXA Inc. All rights reserved.
 * Authors:
 *     2022  Elvis Yao <ElvisCW.Yao@moxa.com>
 * This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
 * See the file MOXA-SOFTWARE-NOTICE for details.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <getopt.h>
#include "../moxa-scaler.h"

#define BIT(nr) (1UL << (nr))

/* CMD format: chk bit(2) + cmd id(3) + data len(1) + hdr checksum(1) + data(data len) + data checksum(1) */
/* checksum = 0xff - (data sum & 0xff) */
#define FWR_VERSION_QUERY_HDR	"\x07\xFF\x53\x57\x49"
#define SYSTEM_STATUS_QUERY_HDR "\x07\xFF\x53\x59\x3F"
#define MODEL_NAME_QUERY_HDR	"\x07\xFF\x54\x59\x50"
#define BRIGHTNESS_CTL_HDR	"\x07\xFF\x42\x4C\x49"
#define TOUCH_PANEL_CTL_HDR	"\x07\xFF\x50\x54\x43"
#define OSD_CONFIG_HDR	        "\x07\xFF\x4F\x53\x44"

int isNumber(char *n) {
	int i = strlen(n);
	int isnum = (i > 0);

	while (i-- && isnum) {
		if (!(n[i] >= '0' && n[i] <= '9')) {
			isnum = 0;
		}
	}

	return isnum;
}

int fwr_version_send(char *argv[], int optind, unsigned char *send_buf) {
	int ret;
	ret = make_packet(send_buf, FWR_VERSION_QUERY_HDR, 0, NULL);
	return ret;
}

void fwr_version_recv(unsigned char *recv_buf) {
	int plen = recv_buf[5];
	printf("Firmware version: %.*s\n", plen, recv_buf+7);
}

int system_status_send(char *argv[], int optind, unsigned char *send_buf) {
	int ret;
	ret = make_packet(send_buf, SYSTEM_STATUS_QUERY_HDR, 0, NULL);
	return ret;
}

void system_status_recv(unsigned char *recv_buf) {
	int plen = recv_buf[5];
	uint8_t status = recv_buf[7];

	printf("Backlight Power %s\n", status & BIT(1) ? "ON" : "OFF");
	printf("%s\n", status & BIT(2) ? "Display output" : "No display output");
}

int model_name_send(char *argv[], int optind, unsigned char *send_buf) {
	int ret;
	ret = make_packet(send_buf, MODEL_NAME_QUERY_HDR, 0, NULL);
	return ret;
}

void model_name_recv(unsigned char *recv_buf) {
	int plen = recv_buf[5];
	printf("Model Name: %.*s\n", plen, recv_buf+7);
}

int brightness_control_send(char *argv[], int optind, unsigned char *send_buf) {
	int ret, val;
	unsigned char buf;
	if (argv[optind]) {
		if (!isNumber(argv[optind])) {
			printf("\nError: The input value '%s' is invalid.\n\n", argv[optind]);
			return -1;
		}

		val = (int)strtol(argv[optind], NULL, 10);
		if (0 <= val && val <= 10) {
			buf = (unsigned char)val;
			ret = make_packet(send_buf, BRIGHTNESS_CTL_HDR, 1, &buf);
		} else {
			printf("\nError: The input value %d is out of range.\n\n", val);
			return -1;
		}
	} else {
		ret = make_packet(send_buf, BRIGHTNESS_CTL_HDR, 0, NULL);
	}

	return ret;
}

void brightness_control_recv(unsigned char *recv_buf) {
	printf("Current brightness is %d\n", recv_buf[7]);
}

int touch_panel_control_send(char *argv[], int optind, unsigned char *send_buf) {
	int ret, val;
	unsigned char buf;
	if (argv[optind]) {
		if (!isNumber(argv[optind])) {
			printf("\nError: The input value '%s' is invalid.\n\n", argv[optind]);
			return -1;
		}

		val = (int)strtol(argv[optind], NULL, 10);
		if (0 <= val && val <= 1) {
			if (val == 1) {
				buf = '\xFF';
			}
			buf = (unsigned char)val;
			ret = make_packet(send_buf, TOUCH_PANEL_CTL_HDR, 1, &buf);
		} else {
			printf("\nError: The input value %d is out of range.\n\n", val);
			return -1;
		}
	} else {
		ret = make_packet(send_buf, TOUCH_PANEL_CTL_HDR, 1, "\x3F");
	}

	return ret;
}

void touch_panel_control_recv(unsigned char *recv_buf) {
	printf("Current panel touch control is %d\n", recv_buf[7]);
}

int osd_config_send(char *argv[], int optind, unsigned char *send_buf) {
	int ret, val;
	unsigned char buf;
	if (argv[optind]) {
		if (!isNumber(argv[optind])) {
			printf("\nError: The input value '%s' is invalid.\n\n", argv[optind]);
			return -1;
		}

		val = (int)strtol(argv[optind], NULL, 10);
		if (0 <= val && val <= 1) {
			if (val == 1) {
				buf = '\xFF';
			}
			buf = (unsigned char)val;
			ret = make_packet(send_buf, OSD_CONFIG_HDR, 1, &buf);
		} else {
			printf("\nError: The input value %d is out of range.\n\n", val);
			return -1;
		}
	} else {
		ret = make_packet(send_buf, OSD_CONFIG_HDR, 1, "\x3F");
	}

	return ret;
}

void osd_config_recv(unsigned char *recv_buf) {
	uint8_t osd = recv_buf[7] & BIT(0);
	printf("Current OSD is %s\n", osd ? "enable" : "disable");
}

int custom_cmd_send(char *argv[], int optind, unsigned char *send_buf) {
	int val, idx;
	idx = 0;
	char *token = strtok(argv[optind-1], " ");
	while(token!=NULL){
		val = (int)strtol(token,NULL,16);
		send_buf[idx] = val;
		idx++;
		token = strtok(NULL, " ");
	}
	if (!validate_cmd_checksum(send_buf, idx)){
		printf("Command format error\n");
		return -1;
	}
	return idx;
}
