#!/bin/bash
#
# Copyright (C) 2023 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#
# Authors:
#       2024  Wilson YS Huang  <wilsonys.huang@moxa.com>

util::is_root_user() {
        [[ "$(id -u)" -eq 0 ]]
}

util::is_dir() {
        [[ -d "$1" ]]
}

util::is_empty_str() {
        [[ -z "$1" ]]
}

util::is_exists() {
        [[ -e "$1" ]]
}

util::is_file() {
        [[ -f "$1" ]]
}

util::parse_product_file() {
        local file_path
        local kernel_major_ver
        local kernel_minor_ver
        local process_fn
        local repo_name
        local repo_tag
        local ret

        file_path="$1"
        kernel_major_ver="$2"
        kernel_minor_ver="$3"
        process_fn="$4"
        ret=1

        if util::is_exists "${file_path}"; then
                # Read from fd=3 instead of the stdin
                while read -u 3 -r line; do
                        # \[         Match an opening square bracket '['
                        # ([0-9]+)   Match and capture one or more digits (major version)
                        # (?:        Start of a non-capturing group (for optional minor version)
                        #   \.       Match a dot '.' (denoting a decimal point)
                        #   ([0-9]+) Match and capture one or more digits (minor version)
                        # )?         End of the non-capturing group, make it optional (minor version)
                        # \]         Match a closing square bracket ']'
                        if [[ $line =~ \[([0-9]+)(\.([0-9]+))?\] ]]; then
                                major_ver="${BASH_REMATCH[1]}"
                                minor_ver="${BASH_REMATCH[3]}"
                        elif ! util::is_empty_str "$line" && [[ "${kernel_major_ver}" == "${major_ver}" ]]; then
                                IFS=', ' read -r -a fields <<<"$line"

                                if [ ${#fields[@]} -ne 2 ]; then
                                        log::error "Invalid config format in line: $line"
                                        exit 1
                                fi

                                repo_name=${fields[0]}
                                repo_tag=${fields[1]}

                                log::debug "repo_name=${repo_name}"
                                log::debug "repo_tag=${repo_tag}"

                                if util::is_empty_str "${minor_ver}" || [[ "${kernel_minor_ver}" == "${minor_ver}" ]]; then
                                        $process_fn "${repo_name}" "${repo_tag}"
                                        ((ret *= $?))
                                fi
                        fi
                done 3<"${file_path}"
        fi

        return $ret
}

util::confirm_prompt() {
        local prompt_msg
        local auto_yes
        prompt_msg="$1"
        auto_yes="$2"

        if [[ $auto_yes = false ]]; then
                read -r -p "${prompt_msg} [Y/n]" response
                if [[ "$response" =~ ^([yY][eE][sS]|[yY])$ ]]; then
                        return 0
                else
                        return 1
                fi
        else
                return 0
        fi
}
