#!/bin/bash
#
# Copyright (C) 2023 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#
# Authors:
#       2023  Wilson YS Huang  <wilsonys.huang@moxa.com>

# shellcheck disable=SC2317 # Don't warn about unreachable commands in this file

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
source "${INSTL_SCRIPT_DIR}/lib/version.sh"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

set -e

process_file_install() {
        local repo_name="$1"
        local repo_tag="$2"
        local repo_ver
        local src_ver
        local cmp_ver

        if util::is_exists "${INSTL_TOP_DIR}/src/${repo_name}-${repo_tag}/version"; then
                repo_ver="$(cat "${INSTL_TOP_DIR}"/src/"${repo_name}"-"${repo_tag}"/version)"
        else
                repo_ver="unknown"
        fi

        if [[ $INSTL_FORCE_INSTALL = true ]]; then
                if util::confirm_prompt "Do you want to install ${repo_name} (${repo_ver})?" "${INSTL_PROMPT_AUTO_YES}"; then
                        source "${INSTL_SCRIPT_DIR}/install.d/${repo_name}.install.sh" "${repo_tag}" "${repo_ver}"

                        if grep -q "${repo_name}" "$INSTL_SRC_VERSION_FILE"; then
                                sed -i "/^$repo_name/s/:\([^:]*\)$/:$repo_ver/" "${INSTL_SRC_VERSION_FILE}"
                        else
                                echo "${repo_name}:${repo_ver}" >>"$INSTL_SRC_VERSION_FILE"
                        fi

                        return 0
                else
                        log::info "Skip install ${repo_name}"
                        return 1
                fi
        else
                if util::is_exists "${INSTL_SRC_VERSION_FILE}" && grep -wq "${repo_name}" "${INSTL_SRC_VERSION_FILE}"; then
                        src_ver=$(grep -w "${repo_name}" "${INSTL_SRC_VERSION_FILE}" | awk -F ':' '{print $2}')

                        cmp_ver=$(ver::compare "$repo_ver" "$src_ver")

                        if [[ $cmp_ver -eq 0 ]]; then
                                log::info "$repo_name is already the latest version ($repo_ver)"
                                return 1
                        elif [[ $cmp_ver -lt 0 ]]; then
                                log::info "$repo_name is already the newest version ($src_ver)"
                                log::info " An OLD version of $repo_name is found ($repo_ver)"
                                log::info " If you want to install, please use the --force option"
                                return 1
                        elif [[ $cmp_ver -gt 0 ]]; then
                                log::info "A NEW version of $repo_name is available ($src_ver -> $repo_ver)"
                                if util::confirm_prompt "Do you want to install?" "${INSTL_PROMPT_AUTO_YES}"; then
                                        source "${INSTL_SCRIPT_DIR}/install.d/${repo_name}.install.sh" "${repo_tag}" "${repo_ver}"
                                        sed -i "/^$repo_name/s/:\([^:]*\)$/:$repo_ver/" "${INSTL_SRC_VERSION_FILE}"

                                        return 0
                                else
                                        log::info "Skip install newer ${repo_name}"

                                        return 1
                                fi
                        fi
                else
                        if util::confirm_prompt "Do you want to install ${repo_name} (${repo_ver})?" "${INSTL_PROMPT_AUTO_YES}"; then
                                source "${INSTL_SCRIPT_DIR}/install.d/${repo_name}.install.sh" "${repo_tag}" "${repo_ver}"
                                echo "${repo_name}:${repo_ver}" >>"$INSTL_SRC_VERSION_FILE"

                                return 0
                        else
                                log::info "Skip install ${repo_name}"

                                return 1
                        fi
                fi
        fi

        return 1
}

main() {
        local drivers_list
        local tools_list
        local driver_available
        local tool_available

        driver_available=true
        tool_available=true

        if util::is_exists "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/preinst.sh"; then
                log::info "=== Run pre-install"
                source "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/preinst.sh"
        fi

        if ! util::is_exists "${INSTL_CONF_DIR}"; then
                mkdir "${INSTL_CONF_DIR}"
        fi

        if ! util::is_exists "${INSTL_SRC_VERSION_FILE}"; then
                touch "${INSTL_SRC_VERSION_FILE}"
        fi

        log::info "=== Install driver"
        drivers_list="${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/drivers"
        if ! util::parse_product_file "${drivers_list}" "${LINUX_KERNEL_MAJOR_VER}" "${LINUX_KERNEL_MINOR_VER}" process_file_install; then
                driver_available=false
        fi

        log::info "=== Install tool"
        tools_list="${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/tools"
        if ! util::parse_product_file "${tools_list}" "${LINUX_KERNEL_MAJOR_VER}" "${LINUX_KERNEL_MINOR_VER}" process_file_install; then
                tool_available=false
        fi

        if [[ $driver_available = false ]] && [[ $tool_available = false ]]; then
                exit 3
        else
                if util::is_exists "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/postinst.sh"; then
                        log::info "=== Run post-install"
                        source "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/postinst.sh"
                fi

                exit 0
        fi
}

main
