/*
Copyright (C) MOXA Inc. All rights reserved.

This software is distributed under the terms of the
MOXA License.  See the file COPYING-MOXA for details.
*/
#include <iostream>
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>  
#pragma comment(lib,"getopt.lib") 
#include "mxsim.h"
using namespace std;

#define UNSET 	-1

enum action_type {
	GET_MODE = 0,
	SET_MODE = 1
};

struct action_struct {
	int type;
	int port;
	int slot;
};

void usage(FILE *fp)
{
	fprintf(fp, "Usage:\n");
	fprintf(fp, "	mx-sim-ctl -i <SIM port index>  [-m <sim slot>]\n");
	fprintf(fp, "OPTIONS:\n");
	fprintf(fp, "	-i <#SIM port index>\n");
	fprintf(fp, "		Set target index: [0..]\n");
	fprintf(fp, "	-m <#sim slot>\n");
	fprintf(fp, "		Set sim slot to\n");
	fprintf(fp, "		0 --> sim slot 0\n");
	fprintf(fp, "		1 --> sim slot 1\n");
	fprintf(fp, "\n");
	fprintf(fp, "Example:\n");
	fprintf(fp, "	Set sim port 1 to sim slot 1\n");
	fprintf(fp, "	# mx-sim-ctl -i 1 -m 1\n");
}

int my_atoi(const char *nptr, int *number)
{
	int tmp;

	tmp = atoi(nptr);
	if (tmp != 0) {
		*number = tmp;
		return 0;
	}
	else {
		if (!strcmp(nptr, "0")) {
			*number = 0;
			return 0;
		}
	}
	return -1;
}

void do_action(struct action_struct action)
{
	switch (action.type) {
	case GET_MODE:
		action.slot = GetSIMSlot(action.port);
		if (action.slot < 0) {
			fprintf(stderr, "Failed to get SIM index\n");
			exit(1);
		}
		printf("SIM port %d slot: %d\n", action.port, action.slot);
		break;
	case SET_MODE:
		if (SetSIMSlot(action.port, action.slot) < 0) {
			fprintf(stderr, "Failed to set SIM index\n");
			exit(1);
		}
		printf("Set OK.\n");		
		break;
	}
}

int main(int argc, char**argv)
{
	struct action_struct action = { GET_MODE, UNSET, UNSET };
	int c;

	while (1) {
		c = getopt(argc, argv, "hi:m:");
		if (c == -1)
			break;

		switch (c) {
		case 'h':
			usage(stdout);
			exit(0);
		case 'i':
			if (my_atoi(optarg, &action.port) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 'm':
			action.type = SET_MODE;
			if (my_atoi(optarg, &action.slot) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		default:
			printf("SIM %d type: %d\n", action.port, action.type);
			usage(stderr);
			exit(99);
		}
	}

	if (optind < argc) {
		fprintf(stderr, "non-option arguments found: ");
		while (optind < argc)
			fprintf(stderr, "%s ", argv[optind++]);
		fprintf(stderr, "\n");

		usage(stderr);
		exit(99);
	}
	if (action.port == UNSET) {
		fprintf(stderr, "port number is unset\n");
		usage(stderr);
		exit(99);
	}
	do_action(action);
	exit(0);
}