/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    moxattyd.c

    Routines to support networking ................

    2008-09-03  James Wang
        Release version 3.3
    2009-09-09  James Wang
        Release version 3.4
    2011-03-18  Ken Huang
        Release version 3.5        
*/
/*
  	Moxa NPort/Async Server UNIX Fixed TTY daemon program.
  	Usage: moxattyd [-t reset-time] [-f file-name] [-d alarm-flag] [-k] [-x] [-u user_defined.cf]
  		reset-time : reconnection successful time
   		file-name  : Fixed TTY program configuration file name. 
  		alarm-flag : enable(1)/disable(0) read/write/select alarm(10)
  		-k         : multi-sessions for terminal users	
  		-x         : disable log function
  		user_defined.cf : User self-defined master/slave tty name configuration file.
  
  	Compilation instructions:
  		SCO:       cc -O -DSCO -o moxattyd moxattyd.c -lsocket
  		LINUX:     cc -O -DLINUX -o moxattyd moxattyd.c
  		ALPHA:     cc -O -DQNX4 -o moxattyd moxattyd.c
  		SVR4.2:    cc -O -DSTREAM -o moxattyd moxattyd.c -lx -lsocket
  		SVR5:      cc -O -DSTREAM -o moxattyd moxattyd.c -lsocket
  		SCO6:      cc -O -DSTREAM -o moxattyd moxattyd.c -lsocket -lnsl
  		AIX:       cc -DAIX -o moxattyd moxattyd.c
  		HP-UX 11i: cc -O -DHPUX -o moxattyd moxattyd.c               
  		Solaris:   cc -O -DSTREAM moxattyd.c -o moxattyd -lnsl -lsocket -laio
  		QNX6:      cc -O -DQNX6 -o moxattyd moxattyd.c -lsocket
  		QNX4:      cc -O -DQNX4 -o moxattyd moxattyd.c -lsocket -lunix
  		FREEBSD:   cc -O -DFREEBSD -o moxattyd moxattyd.c
        FREEBSD8:  cc -O -DBSD8 -o moxattyd moxattyd.c
        MAC:       cc -O -DMAC -o moxattyd moxattyd.c
*/


#define VERSION     "3.5 Build 11032510"

#define	FD_SETSIZE	768          /* system define 150 fd available,now set 768 fd can be used.*/

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/param.h>
#include	<sys/stat.h>
#ifdef	QNX4				
#include	<sys/select.h>	
#include	<sys/wait.h>	
#endif
#include	<netinet/in.h>
#include	<arpa/inet.h>
#include	<netdb.h>
#include	<stdlib.h>
#include	<stdio.h>
#include	<errno.h>
#include	<time.h>
#include	<string.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<signal.h>
#include	<sys/ioctl.h>
#ifdef  BSD8
#define FREEBSD
#endif
#ifndef QNX4				
#ifndef	QNX6				
#ifndef FREEBSD
#ifndef MAC
#include	<stropts.h>
#endif
#endif
#endif
#endif
#ifdef	STREAM
#ifndef AIX
#include	<sys/ptms.h>
#endif
#endif

/*
 * Moxa tty definition
 */
typedef struct _TTYINFO {
	char		mpt_name[40];	/* Master pseudo TTY name */
       u_long  		ip_addr;	/* Server IP address */
	int		tcp_port;	/* Server TCP port number */
	int		mpt_fd; 	/* Master pseudo tty file handle */
	int		sock_fd;	/* TCP Socket handle */
	int		state;		/* TCP connection state */
	char *		mpt_bufptr;	/* Master pseudo TTY data buffer*/
	int		mpt_datakeep;
	int		mpt_dataofs;
	char *		sock_bufptr;	/* TCP socket data buffer */
	int		sock_datakeep;
	int		sock_dataofs;
	int		error_flags;
	int		time_out;	/* Using for TCP connection check */
	int		serv_index;
	int		tcp_state;
	int		tty_opens;
	int		tcp_opens;
	time_t		last_state_time;
	time_t		last_t2s_time;
	time_t		last_s2t_time;
	time_t		wait_id;
	int			tflag;
	int			sameflag;
#ifdef STREAM
	int		slave_fd;
#endif
} TTYINFO;

typedef struct _ConnMsg {
	TTYINFO *	infop;
	int		status;
	time_t		wait_id;
} ConnMsg;

typedef struct _SERVERINFO {
	u_long		ip_addr;
	u_short 	dev_type;
	u_short 	serial_no;
	time_t		last_servertime;
	time_t		next_sendtime;
	int		retry;	/* for power off check.Add by ying */

    /* Below is for DSCI protocol. */
    u_int           ap_id;
    u_short         hw_id;
    unsigned char   mac[6];
} SERVINFO;


typedef struct _DSCI_HEADER {
    unsigned char   opcode;
    unsigned char   result;
    u_short         length;
    u_int           id;
} DSCI_HEADER;

typedef struct _DSCI_RET_HEADER {
    u_int           ap_id;
    u_short         hw_id;
    unsigned char   mac[6];
} DSCI_RET_HEADER;

typedef struct _DSCI_DA_DATA {
    u_int           ap_id;
    u_short         hw_id;
    unsigned char   mac[6];
} DSCI_DA_DATA;

typedef struct _DSCI_NET_STAT {
    u_int           remote_ip;
    u_int           local_ip;
    u_short         remote_port;
    u_short         local_port;
    char            sock_type;
    unsigned char   status;
    unsigned char   reverse[2];    
} DSCI_NET_STAT;


#define 	STATE_INIT		0
#define 	STATE_MPT_OPEN		1
#define 	STATE_CONN_FAIL 	2
#define 	STATE_TCP_OPEN		3
#define 	STATE_TCP_CONN		4
#define 	STATE_TCP_CLOSE 	5
#define 	STATE_TCP_WAIT		6
#define 	STATE_MPT_REOPEN	7
#define 	STATE_RW_DATA		8
#define 	STATE_REMOTE_LISTEN	9

#define 	CONNECT_OK	0
#define 	CONNECT_FAIL	1
#define 	CLOSE_OK	2

#define 	TCP_LISTEN	1
#define     TCP_CONNECTED 4

#define 	ERROR_MPT_OPEN	0x0001
#define 	ERROR_TCP_OPEN	0x0002
#define 	ERROR_TCP_CONN	0x0004
#define 	ERROR_FORK	0x0008

#ifdef		AIX
#define 	BUFFER_SIZE	8192	/* temporary data buffer size for AIX */
#else
#define 	BUFFER_SIZE	1024	/* temporary data buffer size */
#endif
#define 	MAX_TTYS	256	/* max. handling TTYs number */
#define 	MAX_PORTS	16	/* max. port of server is 16 */

int		ttys, servers;
TTYINFO 	ttys_info[MAX_TTYS];
TTYINFO		chttys_info[MAX_TTYS];
SERVINFO	serv_info[MAX_TTYS];
char		EventLog[160];		/* Event log file name */
char		StateLog[160];		/* State log file name */
int		pipefd[2];
int		maxfd;
int     timeout_time = 0;
int		polling_time=0; 	/* default disable polling function */
int		reconn_time;
int		polling_fd;
int     polling_nport_fd;
int		Restart_daemon;
int		Multi_session=0;	/* for multi-session. add by ying */
static	int	No_tty_defined;
#ifdef	STREAM
int		fds;
extern	char	*ptsname();
#endif
time_t		last_log_time;
int		err1_count, err2_count;
int		task_state;
int		log_tty_state[16], log_tcp_state[16];
int		Alarm_Flag = 0;
int		ConfigFile_Flag = 0;
int		UserDefine_Flag = 0;
int		Log_Enable = 1;
time_t		last_rw_time;
int		Timeout = 0;
/*time_t	chlasttime;*/		/* time of last modification in moxattyd.cf 2004/5/17 */
char	cffile[160];	/* moxattyd.cf file path 2004/5/17 */
int		Config_changed;
char	filename[50];

#ifndef STREAM
void	restart_handle ();
void	wait_handle ();
void	timeout_handle ();
void	config_changed_handle ();
#endif
#ifdef	STREAM
void	restart_handle (int);
void	wait_handle (int);
void	timeout_handle (int);
void	config_changed_handle (int);
#endif
void OpenTty(TTYINFO *infop);
void OpenTcpSocket(TTYINFO *infop);
void ConnectTcp(TTYINFO *infop);
void CloseTcp(TTYINFO *infop);
void ConnectCheck(void);
u_long lib_name2ip(char * ipaddr);
int moxattyd_read_config(char * cmdpath);
int poll_async_server_init(void);
void poll_async_server_send(SERVINFO *servp);
int poll_async_server_recv(void);
void log_state(void);
void log_event(char * msg);
void statelog_fname(char *logfname,char *workpath);
void state_change(TTYINFO *infop, int new_state);
int stream_pty_open(char *pts_name);
void recv_fail_message(TTYINFO *infop);
void moxattyd_daemon_start(void);
void moxattyd_handle_ttys(void);
/*
 *	MOXA TTY daemon main program
 */
main(int argc, char * argv[])
{
	TTYINFO *	infop;
	int		i;
    char            buf[80];
     
	for ( i=0; i<16; i++ )
		log_tty_state[i] = log_tcp_state[i] = -1;
	task_state = 0;
	Restart_daemon = 0;
	No_tty_defined = 0;
	polling_fd = -1;
    polling_nport_fd = -1;
	while ( 1 ) {
		task_state = 10;
		if ( Restart_daemon ) {
			close(pipefd[0]);
			close(pipefd[1]);
			infop = ttys_info;
			
			if ( polling_fd >= 0 ) {
				close (polling_fd);
				polling_fd = -1;
			}

            /* Scott 2004-11-01 */
			if (polling_nport_fd >= 0) {
				close(polling_nport_fd);
				polling_nport_fd = -1;
			}
			
			for ( i=0; i<ttys; i++ ) {
				if ( infop->sock_fd >= 0 ){
					close (infop->sock_fd);
					infop->sock_fd = -1;
				}
				if ( infop->mpt_fd >= 0 ){
					close (infop->mpt_fd);
					infop->mpt_fd = -1;
				}
				infop++;
			}
		}
		task_state = 20;
		if ( Restart_daemon == 1 ) {
#ifndef STREAM
			signal (SIGTERM, ( (void (*)()) wait_handle) );
#else
			signal (SIGTERM, wait_handle);
#endif
			task_state = 21;
			log_state();
			pause();
		}

		task_state = 30;
		for(i = 1; i < argc; i++){
			if(strcmp(argv[i], "-t") == 0){
				i++;
				reconn_time = 60 * atoi(argv[i]);
				timeout_time = reconn_time;
				if ( reconn_time >= 60 ) {
					polling_time = (reconn_time - 20) / 4;
					reconn_time -= (polling_time + 20);
				}
			} else if(strcmp(argv[i], "-f") == 0){
				i++;
				ConfigFile_Flag = 1;
				strcpy(filename, argv[i]);
			} else if(strcmp(argv[i], "-d") == 0){
				i++;
				if(atoi(argv[i]) == 1)
					Alarm_Flag = 1;
			} else if(strcmp(argv[i], "-k") == 0){
				Multi_session = 1;
			} else if(strcmp(argv[i], "-x") == 0){
				Log_Enable = 0;
			} else if(strcmp(argv[i], "-u") == 0){
				i++;
				UserDefine_Flag = 1;
				strcpy(filename, argv[i]);
			} else {
				break;
			}
		}

		task_state = 40;
		if ( moxattyd_read_config(argv[0]) <= 0 ) {
			if (!No_tty_defined) {
				sprintf(buf,"Not any pseudo tty defined");
				log_event (buf);
				No_tty_defined = 1;
			}
			log_state();
			continue;
		}
		No_tty_defined = 0;
		task_state = 50;
		if ( !Restart_daemon ) {
			log_state();
	      	 	moxattyd_daemon_start();
			task_state = 51;
			log_state();
		}

		task_state = 60;
		if ( polling_time && (poll_async_server_init() < 0) ) {
			log_state();
			continue;
		}

		task_state = 70;
		if ( pipe(pipefd) < 0 ) {
			sprintf(buf,"pipe error");
			log_event(buf);
			log_state();
			continue;
		}
#ifdef	O_NDELAY
		fcntl(pipefd[0], F_SETFL, fcntl(pipefd[0], F_GETFL) | O_NDELAY);
#elif defined(QNX4) || defined(QNX6)
		fcntl(pipefd[0], F_SETFL, fcntl(pipefd[0], F_GETFL) | O_NONBLOCK);
#endif

#ifndef	QNX4
#ifndef FREEBSD	
#ifndef MAC
#ifndef HPUX
		signal(SIGCLD, SIG_IGN);
#endif
#else
		signal(SIGCHLD, SIG_IGN);		/* Because QNX4/FreeBSD no SIGCLD */
#endif
#endif
#endif

		Restart_daemon = 0;
		Timeout = 0;
		sprintf(buf,"MOXA Fixed TTY daemon program starting (Ver%s) ...", VERSION);
		log_event(buf);
#ifndef STREAM
		signal (SIGTERM, ( (void (*)())restart_handle) );
		if (Alarm_Flag)
			signal (SIGALRM, ( (void (*)())timeout_handle) );
#endif
#ifdef	STREAM
		signal (SIGTERM, restart_handle);
		if (Alarm_Flag)
			signal (SIGALRM, timeout_handle);
#endif
		log_state();


		task_state = 80;
		
#ifndef STREAM
			signal (SIGUSR1, ( (void (*)()) config_changed_handle) );
#endif
#ifdef	STREAM
			signal (SIGUSR1, config_changed_handle);
#endif
		Config_changed = 0;
		moxattyd_handle_ttys();
	}
}

u_long	lib_name2ip(char * ipaddr)
{
	struct hostent *host;
	u_long		ip;

	host = gethostbyname(ipaddr);
	if ( host ){
		ip = ((struct in_addr *)host->h_addr)->s_addr;
	} else{
		ip = inet_addr(ipaddr);
        }
	return(ip);
}

int	moxattyd_read_config(char * cmdpath)
{
	int		n;
	FILE *		ConfigFd;
	struct hostent *host;
	TTYINFO *	infop;
	char		workpath[160], buf[160], curpath[160],showbuf[80];
	char		ttyname[160], ipaddr[160], tcpport[160];

	getcwd(curpath, sizeof(curpath));
	strcpy(buf, cmdpath);
	n = strlen(buf) - 1;
	while ( n > 0 ) {
		if ( buf[n] == '/' )
			break;
		n--;
	}
	buf[n] = 0;
	if ( n )
		chdir(buf);
	getcwd(workpath, sizeof(workpath));
	chdir(curpath);

	if(ConfigFile_Flag || UserDefine_Flag)
		sprintf(buf,"%s/%s", workpath, filename);        /* Config file name */
	else
		sprintf(buf,"%s/moxattyd.cf", workpath);        /* Config file name */
	sprintf(EventLog,"%s/moxattyd.log", workpath);  /* Log file name */
	if(Log_Enable)
		statelog_fname(StateLog, workpath);
/*	stat(buf, &filestat);*/							/* Add for moxattyd.cf changed 2004/5/17 */
	strcpy(cffile, buf);
/*	chlasttime = filestat.st_mtime;*/

	ConfigFd = fopen(buf, "r");
	if ( ConfigFd == NULL ) {
		if(ConfigFile_Flag || UserDefine_Flag)
			sprintf(showbuf,"Can't open configuration file (%s) !", buf);
		else
			sprintf(showbuf,"Can't open configuration file (moxattyd.cf) !");
		log_event(showbuf);
		exit(0);		
	}


	ttys = 0;
	infop = ttys_info;
	while ( ttys < MAX_TTYS ) {
		if ( fgets(buf, sizeof(buf), ConfigFd) == NULL )
			break;			
		if(UserDefine_Flag){
			if ( sscanf(buf, "%s%s%s%s", ttyname, ttyname, ipaddr, tcpport) != 4 )
				continue;
		}else{
			if ( sscanf(buf, "%s%s%s", ttyname, ipaddr, tcpport) != 3 )
				continue;
		}

		if(UserDefine_Flag){
			strcpy(infop->mpt_name, ttyname);
		} else { 
#ifndef STREAM

#ifdef HPUX
			if ( strncmp(ttyname, "pty/tty", 7) == 0 ){
				strcpy(infop->mpt_name, "ptym/pty");
				strcat(infop->mpt_name, &ttyname[7]);
			} else if(strncmp(ttyname, "tty", 3) == 0){
				ttyname[0] = 'p';      
				strcpy(infop->mpt_name, ttyname);
			} else
				continue;
#else
			if ( strncmp(ttyname, "tty", 3) != 0 )
				continue;
			ttyname[0] = 'p';      
			strcpy(infop->mpt_name, ttyname);
#endif

#endif
#ifdef	STREAM
			if ( strncmp(ttyname, "pts", 3) != 0 )
				continue;
			sprintf (infop->mpt_name, "/dev/%s", ttyname);
#endif
		}
		
		infop->ip_addr = lib_name2ip(ipaddr);
		if ( infop->ip_addr == (u_long)0xFFFFFFFF )
			continue;
		if ( (n = atoi(tcpport)) <= 0 || n >= 10000 )
			continue;
		if ( n >= 1 && n <= 16 )
			n += 949;
		infop->tcp_port = n;
		infop->mpt_fd = -1;
		infop->sock_fd = -1;
		infop->state = STATE_INIT;
		infop->mpt_bufptr = (char *)malloc(BUFFER_SIZE * 2);
		if ( infop->mpt_bufptr == (char *)NULL ) {
		    sprintf(showbuf,"Alocate memory fail !");
	        log_event(showbuf);
			break;
		}
		infop->sock_bufptr = infop->mpt_bufptr + BUFFER_SIZE;
		infop->mpt_datakeep = 0;
		infop->mpt_dataofs = 0;
		infop->sock_datakeep = 0;
		infop->sock_dataofs = 0;
		infop->error_flags = 0;
		infop->tcp_state = 0;
		infop->tty_opens = 0;
		infop->tcp_opens = 0;
		infop->last_state_time = time(0);
		infop->tflag = 0;
		infop->sameflag = 0;
		infop++;
		ttys++;
	}

	fclose(ConfigFd);
	
	return(ttys);
}

int moxattyd_change_config(){
	
	int			i, j, n, chttys;
	FILE *		ConfigFd;
	TTYINFO 	*infop, *orip;
	char		buf[160], showbuf[80];
	char		ttyname[160], ipaddr[160], tcpport[160];

	ConfigFd = fopen(cffile, "r");
	if ( ConfigFd == NULL ) {
		if(ConfigFile_Flag || UserDefine_Flag)
			sprintf(showbuf,"Can't open configuration file (%s) !", buf);
		else
			sprintf(showbuf,"Can't open configuration file (moxattyd.cf) !");
		log_event(showbuf);
		return(-1);		
	}

	chttys = 0;
	infop = chttys_info;
	while ( chttys < MAX_TTYS ) {
		if ( fgets(buf, sizeof(buf), ConfigFd) == NULL )
			break;			
		if ( sscanf(buf, "%s%s%s", ttyname, ipaddr, tcpport) != 3 )
			continue;

#ifndef STREAM

#ifdef HPUX
		if ( strncmp(ttyname, "pty/tty", 7) == 0 ){
			strcpy(infop->mpt_name, "ptym/pty");
			strcat(infop->mpt_name, &ttyname[7]);
		} else if(strncmp(ttyname, "tty", 3) == 0){
			ttyname[0] = 'p';      
			strcpy(infop->mpt_name, ttyname);
		} else
			continue;
#else
		if ( strncmp(ttyname, "tty", 3) != 0 )
			continue;
		ttyname[0] = 'p';      
		strcpy(infop->mpt_name, ttyname);
#endif

#endif
#ifdef	STREAM

		if ( strncmp(ttyname, "pts", 3) != 0 )
			continue;
		sprintf (infop->mpt_name, "/dev/%s", ttyname);
#endif
		infop->ip_addr = lib_name2ip(ipaddr);
		if ( infop->ip_addr == (u_long)0xFFFFFFFF )
			continue;
		if ( (n = atoi(tcpport)) <= 0 || n >= 10000 )
			continue;
		if ( n >= 1 && n <= 16 )
			n += 949;
		infop->tcp_port = n;
		infop->mpt_fd = -1;
		infop->sock_fd = -1;
		infop->state = STATE_INIT;
		infop->mpt_bufptr = (char *)malloc(BUFFER_SIZE * 2);
		if ( infop->mpt_bufptr == (char *)NULL ) {
		    sprintf(showbuf,"Alocate memory fail !");
	        log_event(showbuf);
			break;
		}
		infop->sock_bufptr = infop->mpt_bufptr + BUFFER_SIZE;
		infop->mpt_datakeep = 0;
		infop->mpt_dataofs = 0;
		infop->sock_datakeep = 0;
		infop->sock_dataofs = 0;
		infop->error_flags = 0;
		infop->tcp_state = 0;
		infop->tty_opens = 0;
		infop->tcp_opens = 0;
		infop->last_state_time = time(0);
		infop->tflag = 0;
		infop->sameflag = 0;
		infop++;
		chttys++;
	}
	
	fclose(ConfigFd);

	orip = ttys_info;
	for(i = 0; i < ttys; i++, orip++){
		infop = chttys_info;
		for(j = 0; j < chttys; j++, infop++){
			if(strcmp(orip->mpt_name, infop->mpt_name) == 0){
				if(orip->ip_addr == infop->ip_addr){
					if(orip->tcp_port == infop->tcp_port){
						free(chttys_info[j].mpt_bufptr);
						memcpy(&chttys_info[j], &ttys_info[i], sizeof(TTYINFO));
						orip->sameflag = 1;
/*						sprintf(buf,"not changed %d", i);
						log_event(buf);
*/						break;
					}
				}
			}
		}
	}
	orip = ttys_info;
	for(i = 0; i < ttys; i++, orip++){
		if(orip->sameflag != 1){
			free(orip->mpt_bufptr);
#ifdef STREAM
			close(orip->slave_fd);
#endif
			close(orip->mpt_fd);
			close(orip->sock_fd);
		}
	}
	for(i = 0; i < chttys; i++){
		memcpy(&ttys_info[i], &chttys_info[i], sizeof(TTYINFO));
	}
	ttys = chttys;

	return(chttys);
}

void moxattyd_daemon_start(void)
{
	register int	childpid, fd;
        char            buf[80];


	if ( getppid() == 1 )
		goto next;


#ifdef	SIGTTOU
	signal(SIGTTOU, SIG_IGN);
#endif
#ifdef	SIGTTIN
	signal(SIGTTIN, SIG_IGN);
#endif
#ifdef	SIGTSTP
	signal(SIGTSTP, SIG_IGN);
#endif

	if ( (childpid = fork()) < 0 ) {
	    sprintf(buf,"Can't fork first child !");
	    log_event(buf);
		exit(0);
	} else if ( childpid > 0 )
		exit(0);	


#ifdef	TIOCNOTTY

	if ( (fd = open("/dev/tty", O_RDWR)) >= 0 ) {
		ioctl(fd, TIOCNOTTY, (char *)NULL);
		close(fd);
	}

#else

	if ( setpgrp() == -1 ) {
	    sprintf(buf,"Can't change process group !");
	    log_event(buf);
		exit(0);
	}
	signal(SIGHUP, SIG_IGN);
	if ( (childpid = fork()) < 0 ) {
	    sprintf(buf,"Can't fork second child !");
		log_event(buf);
		exit(0);
	} else if ( childpid > 0 )
		exit(0);	

#endif

next:

	close(0);
	close(1);
	close(2);

	errno = 0;

	chdir("/");

	umask(0);
}

int  poll_async_server_init(void)
{
	int			i, n, udp_port;
	struct sockaddr_in	sin;
        char buf[80];

	servers = 0;
	for ( i=0; i<ttys; i++ ) {
		for ( n=0; n<servers; n++ )
			if ( serv_info[n].ip_addr == ttys_info[i].ip_addr )
				break;
		if ( n == servers ) {
			ttys_info[i].serv_index = servers;
			serv_info[servers].ip_addr = ttys_info[i].ip_addr;
			serv_info[servers].dev_type = 0;
			serv_info[servers].serial_no = 0;
			serv_info[servers].last_servertime = time(0) - 2;
			serv_info[servers].next_sendtime = time(0) - 1;
			serv_info[servers].retry = 0;
            		serv_info[servers].ap_id = 0;   /* Scott 2004-11-01 */
            		serv_info[servers].hw_id = 0;   /* Scott 2004-11-01 */
			servers++;
		}
		else
			ttys_info[i].serv_index = n; /* Scott added: 2005-03-02 */
	}
	if ( (polling_nport_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0 ) {
		log_event("Can not open the polling_nport_fd socket !");
		return(-1);
	}
	sin.sin_family = AF_INET;
	sin.sin_port = 0;
	sin.sin_addr.s_addr = INADDR_ANY;
	if (bind(polling_nport_fd,(struct sockaddr *)&sin,sizeof(sin)) == 0) {
#ifdef	FIONBIO
			fcntl(polling_nport_fd, FIONBIO);
#endif
	} else {
    		close(polling_nport_fd);
    		polling_nport_fd = -1;
    		log_event("Can not bind the polling NPort UDP port !");
    		return(-1);
    	}
    
	if ( (polling_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0 ) {
	     sprintf(buf,"Can not open the polling UDP socket !");
		log_event(buf);
		return(-1);
	}

	sin.sin_family = AF_INET;
	sin.sin_port = 0;
	sin.sin_addr.s_addr = INADDR_ANY;
	if (bind(polling_fd,(struct sockaddr *)&sin,sizeof(sin)) == 0) {
#ifdef	FIONBIO
			fcntl(polling_fd, FIONBIO);
#endif
			return(0);
	}

	close(polling_fd);
	sprintf(buf,"Can not bind the polling UDP port !");
	log_event(buf);
	return(-1);
}

void poll_async_server_send(SERVINFO * servp)
{
	struct sockaddr_in	to;
	int			len;
	time_t			t;
	unsigned char		msg[32];

    if (servp->ap_id)
        return;
    
#ifndef STREAM
	bzero(msg, 28);
#endif
#ifdef	STREAM
	memset (msg, 0, 28);
#endif
	t = time(0);
	if ( servp->dev_type == 0 ) {
		servp->next_sendtime = t + 5;
		msg[0] = 1;
		msg[3] = 6;
		len = 6;
	} else {
		servp->next_sendtime = t + polling_time;
		msg[0] = 0x71;
		msg[3] = 30;
		*(u_short *)&msg[22] = servp->dev_type;
		*(u_short *)&msg[24] = servp->serial_no;
		*(u_int *)&msg[26] = htonl(t);
		len = 30;
	}
	to.sin_family = AF_INET;
	to.sin_port = htons(0x405);
	to.sin_addr.s_addr = servp->ip_addr;
	sendto(polling_fd, msg, len, 0, (struct sockaddr *)&to, sizeof(to));
}

int poll_async_server_recv(void)
{
	struct sockaddr_in	from;
	int			len, n, m, i, connected, listening;
	time_t			t;
	SERVINFO *		servp;
	TTYINFO *		infop;
	unsigned char		msg[100];

	len = sizeof(from);
#ifdef	AIX
if ( recvfrom(polling_fd, msg, 86, 0, (struct sockaddr *)&from, (socklen_t *)&len)
#else
#ifdef	SCO
if ( recvfrom(polling_fd, msg, 86, 0, (struct sockaddr *)&from, &len)
#endif
#ifndef SCO
#ifdef HPUX
if ( recvfrom(polling_fd, msg, 86, 0, (void *)&from, &len)
#else
#ifdef MAC
if ( recvfrom(polling_fd, msg, 86, 0, (struct sockaddr *)&from, (socklen_t*)&len)
#else
#ifdef BSD8
if ( recvfrom(polling_fd, msg, 86, 0, (struct sockaddr *)&from, (socklen_t*)&len)
#else
if ( recvfrom(polling_fd, msg, 86, 0, (struct sockaddr *)&from, (size_t*)&len)
#endif
#endif
#endif
#endif
#endif
	     != 86 ) {
		err2_count++;
		return 1;
	}
	if ( ((msg[0] != 0x81) && (msg[0] != 0xF1)) || (msg[3] != 86) )
		return 2;
	if ( msg[1] || msg[2] || msg[4] || msg[5] ||
	     (from.sin_port != ntohs(0x405)) )
		return 3;
	for ( n=0, servp=serv_info; n<servers; n++, servp++ ) {
		if ( from.sin_addr.s_addr == servp->ip_addr )
			break;
	}
	if ( n == servers )
		return 4;
	if ( msg[0] == 0x81 ) {
		n = 0;
		if ( (msg[10]==0x08 && msg[11]==0x21) || (msg[10]==0x16 && msg[11]==0x21) ) {
		   if ( (msg[25] > 1) || (msg[24] > (unsigned char)0x25) ) {
			servp->dev_type = *(u_short *)&msg[10];
			servp->serial_no = *(u_short *)&msg[12];
			servp->last_servertime = time(0) - 1;
			n = 1;
		   }
		}else{
			servp->dev_type = *(u_short *)&msg[10];
			servp->serial_no = *(u_short *)&msg[12];
			servp->last_servertime = time(0) - 1;
			n = 1;
		}
		if ( (servp->serial_no == 0) || (n == 1) ) {		/*Casper change*/
			servp->serial_no = *(u_short *)&msg[12];
			*(u_long *)(&msg[96]) = servp->ip_addr;
			msg[93] = msg[23];
			msg[94] = msg[24];
			msg[95] = msg[25];
			if(msg[93]) /* x.x.[x] */
				sprintf((char *)msg, 
				"IP=%d.%d.%d.%d, Ver=%x.%x.%x(0x%02x%02x%02x) is alive.",
				(int)(msg[96]), (int)(msg[97]), (int)(msg[98]),
				(int)(msg[99]), (int)(msg[95]), (int)(msg[94]),
				(int)(msg[93]), (int)(msg[95]), (int)(msg[94]),
				(int)(msg[93]));
			else
				sprintf((char *)msg, 
				"IP=%d.%d.%d.%d, Ver=%x.%x(0x%02x%02x) is alive.",
				(int)(msg[96]), (int)(msg[97]), (int)(msg[98]),
				(int)(msg[99]), (int)(msg[95]), (int)(msg[94]),
				(int)(msg[95]), (int)(msg[94]));
			log_event((char *)msg);
/*
			if (msg[94] < 0x10)
				sprintf((char *)msg, "NPort(Async) Server (%d.%d.%d.%d) firmware version is %d.%02x .",
				(int)(msg[96]), (int)(msg[97]), (int)(msg[98]),
				(int)(msg[99]), (int)(msg[95]), (int)(msg[94]));
			else
				sprintf((char *)msg, "NPort(Async) Server (%d.%d.%d.%d) firmware version is %d.%2x .",
				(int)(msg[96]), (int)(msg[97]), (int)(msg[98]),
				(int)(msg[99]), (int)(msg[95]), (int)(msg[94]));
*/
		}
		return 5;
	}
	t = ntohl(*(int *)&msg[18]);
	if ( (int)(t - servp->last_servertime) <= 0 ) {
		err1_count++;
		return 6;
	}
	if ( (servp->dev_type != *(u_short *)&msg[10]) ||
	     (servp->serial_no != *(u_short *)&msg[12]) ) {
		servp->dev_type = 0;
		servp->next_sendtime = time(0) - 1;
		servp->retry = 0;
		return 7;
	}
	m = 0;
	servp->last_servertime = t;
	for ( n=0, infop=ttys_info; n<ttys; n++, infop++ ) {
		if ( infop->ip_addr != servp->ip_addr ){
			continue;
		}
        for(i=0, connected=0, listening=0; i<MAX_PORTS; i++) {
		    if ( infop->tcp_port != ntohs(*(u_short *)&msg[22+i*2]))
			    continue;
		    /*infop->tcp_state = (int)msg[54+i];*/
            if ( msg[54+i] == TCP_CONNECTED)
            {
                connected = 1;
                break;
            }
		    if ( msg[54+i] == TCP_LISTEN ) {
			    if ( infop->state == STATE_RW_DATA )
                    listening = 1;
		    }
	    }
       
    	if ( !connected && listening == 1 ) {
    		m++;
    		state_change(infop, STATE_REMOTE_LISTEN);
    		infop->time_out = time(0);
    	/* for Multi sessions reconnetion */
    		if ( Multi_session ) {
                servp->retry = 6;  
    			return 9;    		
    		}	
    	}
	}
	if ( m ) {
		*(u_long *)(&msg[96]) = servp->ip_addr;
		sprintf((char *)msg,"Ports reset of Async Server %d.%d.%d.%d !",
			(int)(msg[96]), (int)(msg[97]), (int)(msg[98]),
			(int)(msg[99]));
		log_event((char *)msg);
	}
	servp->retry = 0;
	return 8;
}

void poll_nport_send(servp)
SERVINFO	*servp;
{
	struct sockaddr_in	to;
	int			    len;
	unsigned char	msg[32];
    DSCI_HEADER     *dsci_headerp;
    DSCI_DA_DATA    *dscidata_p;

    if (servp->dev_type)
        return;

#ifndef	STREAM
	bzero(msg, 28);
#endif
#ifdef	STREAM
	memset (msg, 0, 28);
#endif
	if ( servp->ap_id == 0 ) {   /* send dsc_search */
		servp->next_sendtime = time(0) + 5;
        dsci_headerp=(DSCI_HEADER*)&msg[0];
		dsci_headerp->opcode = 0x01; /* dsc_search */
		dsci_headerp->result = 0;
		dsci_headerp->length = htons(8);
		dsci_headerp->id = 0;
		len = 8;
	} else {        /* send dsc_GetNetstat */
		servp->next_sendtime = time(0) + polling_time;
        dsci_headerp=(DSCI_HEADER*)&msg[0];
		dsci_headerp->opcode = 0x14; /* dsc_GetNetstat */
		dsci_headerp->result = 0;
		dsci_headerp->length = htons(22);
		dsci_headerp->id = htonl(time(0));
		
        dscidata_p=(DSCI_DA_DATA*)&msg[8];
		dscidata_p->ap_id = htonl(servp->ap_id);
		dscidata_p->hw_id = htons(servp->hw_id);
		memcpy((void*)dscidata_p->mac, (void*)servp->mac, 6);
        *(u_short *)(&msg[20]) = 128;   /* max number of sockets */
		len = 22;
	}
	to.sin_family = AF_INET;
	to.sin_port = htons(4800);
	to.sin_addr.s_addr = servp->ip_addr;
	sendto(polling_nport_fd, msg, len, 0, (struct sockaddr *)&to, sizeof(to));
}

void poll_nport_recv()
{
    struct sockaddr_in  from;
    int             retlen, len, n, m, i, nstat, connected, listening;
    time_t            t;
    SERVINFO *      servp;
    TTYINFO *       infop;
    unsigned char   msg[2100];
    u_short			tmp_ip;
    DSCI_HEADER     *dsci_headerp;
    DSCI_RET_HEADER *dsci_retp;
    DSCI_NET_STAT   *dsci_netstatp;
	
	len = sizeof(from);
#ifdef	AIX
	if ( (retlen=recvfrom(polling_nport_fd, msg, sizeof(msg), 0, (struct sockaddr *)&from, (socklen_t *)&len))
#else
#ifdef	SCO
	if ( (retlen=recvfrom(polling_nport_fd, msg, sizeof(msg), 0, (struct sockaddr *)&from, &len))
#endif
#ifndef	SCO
#ifdef HPUX
	if ( (retlen=recvfrom(polling_nport_fd, msg, sizeof(msg), 0, (void *)&from, &len))
#else
#ifdef MAC
	if ( (retlen=recvfrom(polling_nport_fd, msg, sizeof(msg), 0, (struct sockaddr *)&from, (socklen_t  *)&len))
#else
#ifdef BSD8
	if ( (retlen=recvfrom(polling_nport_fd, msg, sizeof(msg), 0, (struct sockaddr *)&from, (socklen_t  *)&len))
#else
	if ( (retlen=recvfrom(polling_nport_fd, msg, sizeof(msg), 0, (struct sockaddr *)&from, (size_t  *)&len))
#endif    
#endif
#endif
#endif
#endif
	     != 24 && ((retlen-24)%16))
		return;

    dsci_headerp=(DSCI_HEADER*)&msg[0];
	if ( (dsci_headerp->opcode == 0x81 && (ntohs(dsci_headerp->length) != 24 && ntohs(dsci_headerp->length) != 40)) || (dsci_headerp->opcode == 0x94 && ((ntohs(dsci_headerp->length)-24)%16) != 0) )
		return;
		
	if ( dsci_headerp->result!=0 || (from.sin_port != ntohs(4800)) )
		return;

	for ( n=0, servp=serv_info; n<servers; n++, servp++ ) {
		if ( from.sin_addr.s_addr == servp->ip_addr )
			break;
	}
	if ( n == servers )
		return;

    dsci_retp=(DSCI_RET_HEADER*)&msg[8];
	if ( dsci_headerp->opcode == 0x81 ) {     /* dsc_search respons */
		servp->ap_id = ntohl(dsci_retp->ap_id);
		servp->hw_id = ntohs(dsci_retp->hw_id);
        memcpy((void*)servp->mac, (void*)dsci_retp->mac, 6);
		servp->last_servertime = time(0) - 1;

		*(u_long *)(&msg[96]) = servp->ip_addr;
		sprintf((char *)msg, "IP=%d.%d.%d.%d is alive.",
		(int)(msg[96]), (int)(msg[97]), (int)(msg[98]), (int)(msg[99]));
		log_event((char*)msg);
		return;
	}

    t = ntohl(dsci_headerp->id);
    if ( (int)(t - servp->last_servertime) <= 0 )
        return;
	if ( (servp->ap_id != ntohl(dsci_retp->ap_id)) ||
	     (servp->hw_id != ntohs(dsci_retp->hw_id)) ) {
		servp->ap_id = 0;
		servp->next_sendtime = time(0) - 1;
		return;
	}
	m = 0;
	servp->last_servertime = t;
/*    nstat = *(u_short*)(&msg[20]);*/
    nstat = (int)msg[21];	/* for big&little endian machine */
    nstat = (nstat << 8) | ((int)msg[20] & 0xff);
    if(nstat > 128){            /* the value can not over 128 */
        nstat = 128;
    }
    for ( n=0, infop=ttys_info; n<ttys; n++, infop++ ) {
        if ( infop->ip_addr != servp->ip_addr )
            continue;

        for(i=0, connected = 0, listening = 0; i<nstat; i++) {
            dsci_netstatp=(DSCI_NET_STAT*)&msg[24+i*16];

			tmp_ip = (u_short)msg[24+i*16+11];  /* for big&little endian machine */
			dsci_netstatp->local_port = (tmp_ip << 8) | ((u_short)msg[24+i*16+10] & 0xff);

            if ( infop->tcp_port != dsci_netstatp->local_port )
                continue;
             
            if ( dsci_netstatp->status == TCP_CONNECTED ) {
                connected = 1;
                break;
            }
            if ( dsci_netstatp->status == TCP_LISTEN ) {
                if ( infop->state == STATE_RW_DATA )
                    listening = 1;
            }
        }
        if ( !connected && listening == 1) {
            m++;
            state_change(infop, STATE_REMOTE_LISTEN);
            infop->time_out = time(0);
        }
	}
	if ( m ) {
		*(u_long *)(&msg[96]) = servp->ip_addr;
		sprintf((char *)msg, "Ports reset of NPort(Async) Server %d.%d.%d.%d !",
			(int)(msg[96]), (int)(msg[97]), (int)(msg[98]),
			(int)(msg[99]));
		log_event((char*)msg);		
	}
}

void moxattyd_handle_ttys(void)
{
	int		i, n, m, maxfd, maxfd2, t0, j;
	TTYINFO *	infop;
	SERVINFO *	servp;
	fd_set		rfd, wfd;
	struct timeval	tm, polltm;
	char		buf[80];
/*	struct stat	filestat;*/		/* Terence add 2004/5/17 */
	double	tmpval;

	last_log_time = time(0) - 50;
	last_rw_time = time(0);
	err1_count = err2_count = 0;
	signal(SIGPIPE, SIG_IGN);			/* add for "broken pipe" error 2004/1/7*/
	while ( 1 ) {
		if ( Restart_daemon )
			return;
		log_state();
		tm.tv_sec = 3;
		tm.tv_usec = 0;
		polltm.tv_sec = 0;
		polltm.tv_usec = 0;
		FD_ZERO(&rfd);
		FD_ZERO(&wfd);
		maxfd = -1;
		
		if(Config_changed > 0){
			Config_changed = 0;
/*			sprintf(buf,"file changed !");
			log_event(buf);*/
		}
	/* Modify for power off check */
		if ( polling_time ) {
			for (i=0; i<servers; i++) {
				t0 = time(0);
				if ((t0-(int)serv_info[i].next_sendtime) > 0) {
					poll_async_server_send(&serv_info[i]);
                    poll_nport_send(&serv_info[i]);
					serv_info[i].retry++;
				}
				FD_SET(polling_fd, &rfd);
                FD_SET(polling_nport_fd, &rfd);
                if (polling_fd > polling_nport_fd)
                    maxfd2=polling_fd;
                else
                    maxfd2=polling_nport_fd;
				if (select(maxfd2+1, &rfd, &wfd, NULL, &polltm)> 0) {
					if (FD_ISSET(polling_fd, &rfd)) 
	    				poll_async_server_recv(); 
                    if (FD_ISSET(polling_nport_fd, &rfd))
                        poll_nport_recv();
				}
				if ((time(0)-serv_info[i].last_servertime)>timeout_time) {
				/* if (serv_info[i].retry > 3) { */
					serv_info[i].retry = 0;
					for (j=0,infop=ttys_info; j<ttys; j++,infop++) {
						if (infop->ip_addr != serv_info[i].ip_addr)
							continue;
						if (infop->state==STATE_RW_DATA) {
							state_change(infop, STATE_REMOTE_LISTEN);
							infop->time_out = time(0);
						}
					}
				}
			}
		}
		
		t0 = time(0);
		for ( i=0, infop=ttys_info; i<ttys; i++, infop++ ) {
			if ( infop->state == STATE_INIT ||
			     infop->state == STATE_MPT_OPEN ||
			     infop->state == STATE_MPT_REOPEN ) {
				OpenTty(infop);
			}

			if ( infop->state == STATE_CONN_FAIL ) {
				if ( (t0 - infop->time_out) >= 5 )
					state_change(infop, STATE_TCP_OPEN);
			}

			if ( infop->state == STATE_TCP_OPEN ) {
				OpenTcpSocket(infop);
			}

			if ( infop->state == STATE_TCP_CONN )
				ConnectTcp(infop);

			if ( infop->state == STATE_TCP_CLOSE )
				CloseTcp(infop);
                        
			if ( infop->state == STATE_TCP_WAIT )
				ConnectCheck();


			if ( infop->state < STATE_TCP_WAIT ) {		/*Terence change STATE_RW_DATA to STATE_TCP_WAIT*/
				tm.tv_sec = 1;							/*2004/1/2*/
				continue;			
			} else if ( (infop->state == STATE_REMOTE_LISTEN) /*&&
				((t0 - infop->time_out) >= reconn_time) */) {
				CloseTcp(infop);
				continue;
			}
			if ( infop->mpt_fd > maxfd )
				maxfd = infop->mpt_fd;
			if ( infop->sock_fd > maxfd )
				maxfd = infop->sock_fd;
			
			if ( infop->mpt_datakeep ){
				if(infop->tflag == 0){
					FD_SET(infop->sock_fd, &wfd);
				} 
			}else {
				if( infop->mpt_fd >= 0)
					FD_SET(infop->mpt_fd, &rfd);
			}
			
			if ( infop->sock_datakeep ) {
				if( infop->mpt_fd >= 0)
					FD_SET(infop->mpt_fd, &wfd);
			}else {
				if(infop->tflag == 0)
					FD_SET(infop->sock_fd, &rfd);
			}
		}
		if ( maxfd < 0 ) {
			task_state = 81;
			sleep(1);
			task_state = 82;
			continue;
		}
		if (Alarm_Flag) { 
			Timeout = 0;
			alarm(10);
		}

		if ( select(maxfd+1, &rfd, &wfd, NULL, &tm) <= 0 ) {	/*SVR5: Terence change &efd -> NULL*/
			if (Alarm_Flag) {
				alarm(0);
				if (Timeout) {
					sprintf(buf,"Warning: select don't complete for 10 seconds, its timeout is %ld seconds\n",tm.tv_sec);
					log_event(buf);
				}
			}
			continue;
		}
		if (Alarm_Flag) { 
			alarm(0);
		}
		last_rw_time = time(0);

		for ( i=0, infop=ttys_info; i<ttys; i++, infop++ ) {
			if(Config_changed > 0)
				break;
			if ( infop->state < STATE_TCP_WAIT )		/*Terence change STATE_RW_DATA to STATE_TCP_WAIT*/
				continue;								/*2004/1/2*/

        /* Scott add if condition 04-05-2005
           Prevent program halt error when infop->mpt_fd=-1. */
			if ( infop->state != STATE_MPT_REOPEN ) {
    			if ( FD_ISSET(infop->mpt_fd, &rfd) ) {
    				infop->last_t2s_time = time(0);
    				m = infop->mpt_datakeep + infop->mpt_dataofs;
    				if (Alarm_Flag) { 
    					Timeout = 0;
    					alarm(10);
    				}
    				n = read(infop->mpt_fd,
    					 infop->mpt_bufptr + m,
    					 BUFFER_SIZE - m);

    				if (Alarm_Flag) { 
    					alarm (0);
    					if (Timeout) {
    						sprintf(buf,"Warning: %s read tty timeout for 10 seconds\n",infop->mpt_name);
    						log_event(buf);
    					}
    				}
    				if ( n > 0 ){
    					if(infop->tflag == 0)
    						infop->mpt_datakeep += n;
    			/* Scott: 04-05-2005 */
    			/*    When read()=0, don't reopen the master tty */
    		    /*  }else if ( n <= 0 ) { */ /*Add "=" for SCO,Ying*/
    				}else if ( n < 0 ) { 
    					state_change(infop, STATE_MPT_REOPEN);
    					continue;
    				}
#if defined(FREEBSD) || defined(SCO) || defined(QNX6) || defined(MAC)
					else 
						usleep(100);
#endif
    			}
            }
			if ( infop->state != STATE_TCP_WAIT ) {			/*Terence add if condition 2004/1/6*/
				if ( FD_ISSET(infop->sock_fd, &rfd) ) {
					m = infop->sock_datakeep + infop->sock_dataofs;
					if (Alarm_Flag) { 
						Timeout = 0;
						alarm (10);
					}
					infop->last_s2t_time = time(0);
			
					n = read(infop->sock_fd,
						 infop->sock_bufptr + m,
						 BUFFER_SIZE - m);
			
					if (Alarm_Flag) { 
						alarm (0);
						if (Timeout) {
							sprintf(buf,"Warning: %s read socket timeout for 10 seconds\n",infop->mpt_name);
							log_event(buf);
						}
					}
					if ( n > 0 ) {
						infop->sock_datakeep += n;
						if ( infop->state != STATE_MPT_REOPEN )
						    state_change(infop, STATE_RW_DATA);  /* when mpt_fd < 0, it could not be change to STATE_RW_DATA */
					} else {
						state_change(infop, STATE_TCP_OPEN);	/*Terence change MPT->TCP 2004/1/2*/
						continue;
					}
				}
			}
        /* Scott add if condition 04-05-2005
           Prevent program halt error when infop->mpt_fd=-1. */
			if ( infop->state != STATE_MPT_REOPEN ) {
    			if ( FD_ISSET(infop->mpt_fd, &wfd) ) {
    				if (Alarm_Flag) { 
    					Timeout = 0;
    					alarm(10);
    				}
  					
    				n = write(infop->mpt_fd,
    					infop->sock_bufptr+infop->sock_dataofs,
    					infop->sock_datakeep);

    				if (Alarm_Flag) { 
    					alarm (0);
    					if (Timeout) {
    						sprintf(buf,"Warning: %s write tty timeout for 10 seconds\n",infop->mpt_name);
    						log_event(buf);
    					}
    				}
    				if ( n > 0 ) {
    					infop->sock_datakeep -= n;
    					if ( infop->sock_datakeep )
    						infop->sock_dataofs += n;
    					else
    						infop->sock_dataofs = 0;
    				} else {	/*SVR5: Terence add else{} */
#ifndef QNX4		/* Terence add. QNX4 does not have I_FLUSH & FLUSHRW 2004/5/6 */
#ifndef	QNX6		/* Terence add for QNX6 2004/4/21 */
#ifndef	FREEBSD
#ifndef MAC
//    					ioctl(infop->sock_fd, I_FLUSH, FLUSHRW);
#endif
#endif
#endif
#endif

//    					infop->sock_datakeep = 0;
//   					infop->sock_dataofs = 0;
    				}
                }
            }
			if ( infop->state != STATE_TCP_WAIT ) {			/*Terence add if condition 2004/1/6*/
				if ( FD_ISSET(infop->sock_fd, &wfd) ) {
					if (Alarm_Flag) {
						Timeout = 0;
						alarm(10);
					}
				
					n = write(infop->sock_fd,
						  infop->mpt_bufptr+infop->mpt_dataofs,
						  infop->mpt_datakeep); 

					if (Alarm_Flag) { 
						alarm (0);
						if (Timeout) {
							sprintf(buf,"Warning: %s write tty timeout for 10 seconds\n",infop->mpt_name);
							log_event(buf);
						}
					}
					if ( n > 0 ) {
						infop->mpt_datakeep -= n;
						if ( infop->mpt_datakeep )
							infop->mpt_dataofs += n;
						else
							infop->mpt_dataofs = 0;
					} else {						/*Terence add else{} 2003/12/30*/
						state_change(infop, STATE_TCP_OPEN);
						continue;
					}
				}
			}
		}
	}
}

void OpenTty(TTYINFO * infop)
{
	char	ttyname[160], buf[160];
	int	flag;

	infop->tty_opens++;
	flag = 0;
	if ( infop->mpt_fd >= 0 ) {
		if(infop->state == STATE_MPT_REOPEN) {		/* Terence add if{}-else & flag=2, 2004/1/5 */
			close(infop->mpt_fd);
			infop->mpt_fd = -1;   /* Add by Ying */
			flag = 2;
		} else
			flag = 1;
	} else
		flag = 0;

	if ( infop->mpt_fd < 0 ) {						/* Terence add if{} 2004/1/5 */
#ifndef STREAM
		sprintf(ttyname, "/dev/%s", infop->mpt_name);
#ifdef	O_NOCTTY
#ifndef QNX4										/* Terence add for QNX4 2004/5/6 */
		infop->mpt_fd = open(ttyname, O_RDWR | O_NDELAY | O_NOCTTY);
#else
		infop->mpt_fd = open(ttyname, O_RDWR | O_NONBLOCK | O_NOCTTY);
#endif
#else
		infop->mpt_fd = open(ttyname, O_RDWR | O_NDELAY);
#endif
#endif
#ifdef	STREAM
		infop->mpt_fd = stream_pty_open (infop->mpt_name);
		if(infop->mpt_fd)
			infop->slave_fd = fds;
#endif
	}
	
	if ( infop->mpt_fd >= 0 ) {
		if ( infop->state == STATE_MPT_REOPEN ) {
			if(infop->tflag == 0)						/* Terence add if & else{} 2004/1/5*/
				state_change(infop, STATE_RW_DATA);
			else
				state_change(infop, STATE_TCP_WAIT);
		}else{
			state_change(infop, STATE_TCP_OPEN);
		}
	} else if ( (flag == 0) || (flag == 2) ) {				/*Terence add flag == 2 2004/1/5*/
		if ( !(infop->error_flags & ERROR_MPT_OPEN) ) {
			sprintf(ttyname, "Master pseudo tty open fail (%s) !",
				infop->mpt_name);
			log_event(ttyname);
			infop->error_flags |= ERROR_MPT_OPEN;
		}
	}
}

void OpenTcpSocket(TTYINFO * infop)
{
	char	buf[80];
	int	on=1;

	if( infop->sock_fd >= 0 ){				/*Terence add if{} 2004/1/6*/
		close(infop->sock_fd);
		infop->sock_fd = -1;
	}
	infop->tflag = 1;
	infop->tcp_opens++;
	infop->sock_fd = socket(AF_INET, SOCK_STREAM, 0);
	if ( infop->sock_fd >= 0 ) {
/*
		if ( setsockopt(infop->sock_fd, SOL_SOCKET, SO_KEEPALIVE,
				(char *)&on, sizeof(on)) < 0 )
			log_event("Set TCP keep alive fail !");
*/
		state_change(infop, STATE_TCP_CONN);
	} else {
		if ( !(infop->error_flags & ERROR_TCP_OPEN) ) {
			struct in_addr tmp_addr;
			tmp_addr.s_addr = infop->ip_addr;
			sprintf(buf, "Socket open fail (%s, TCP port %d) !",
				inet_ntoa(tmp_addr), infop->tcp_port);
			log_event(buf);

			infop->error_flags |= ERROR_TCP_OPEN;
		}
	}
}

void ConnectTcp(TTYINFO * infop)
{
	struct sockaddr_in	sin;
	int			childpid;
	ConnMsg 		msg;
        char                    buf[80];

	state_change(infop, STATE_TCP_WAIT);
	infop->wait_id = time(0);
	if ( (childpid = fork()) == 0 ) {	
		close(pipefd[0]);
		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = infop->ip_addr;
		sin.sin_port = htons(infop->tcp_port);
		if ( connect(infop->sock_fd, (struct sockaddr *)&sin,
			sizeof(sin)) < 0 ) {
			msg.status = CONNECT_FAIL;
		}else{
			msg.status = CONNECT_OK;
		}
		msg.infop = infop;
		msg.wait_id = infop->wait_id;
		write(pipefd[1], (char *)&msg, sizeof(ConnMsg));
		close(pipefd[1]);
		exit(0);
	} else if ( childpid < 0 ) {
		state_change(infop, STATE_TCP_CONN);
		if ( !(infop->error_flags & ERROR_FORK) ) {
	            sprintf(buf,"Can't fork child process !");
			log_event(buf);
			infop->error_flags |= ERROR_FORK;
		}
	}
}

void CloseTcp(TTYINFO * infop)
{
	struct sockaddr_in	sin;
	int			childpid;
	ConnMsg 		msg;
	char                    buf[80];

	state_change(infop, STATE_TCP_WAIT);
	infop->wait_id = time(0);
	if ( (childpid = fork()) == 0 ) {	
		close(pipefd[0]);
		close(infop->sock_fd);			/*2004/1/5 fork can not set sock-fd = -1*/
		sleep(1);
		msg.status = CLOSE_OK;
		msg.infop = infop;
		msg.wait_id = infop->wait_id;
		write(pipefd[1], (char *)&msg, sizeof(ConnMsg));
		close(pipefd[1]);
		exit(0);
	} else if ( childpid < 0 ) {
		state_change(infop, STATE_TCP_CLOSE);
		if ( !(infop->error_flags & ERROR_FORK) ) {
		    sprintf(buf,"Can't fork child process !");
			log_event(buf);
			infop->error_flags |= ERROR_FORK;
		}
	}

	if ( infop->state != STATE_TCP_CLOSE ){
		close(infop->sock_fd);
	}
}

void ConnectCheck(void)
{
	ConnMsg 	msg;
	TTYINFO *	infop;

	if (read(pipefd[0], (char *)&msg, sizeof(ConnMsg)) == sizeof(ConnMsg)) {
		infop = msg.infop;
		if ( (infop->state == STATE_TCP_WAIT) &&
		     (infop->wait_id == msg.wait_id) ) {
			if ( msg.status == CONNECT_OK ) {
				state_change(infop, STATE_RW_DATA);
				infop->tflag = 0;
				infop->error_flags = 0;
				infop->last_s2t_time = time(0);
				infop->last_t2s_time = time(0); 
#if defined(QNX4) || defined (HPUX) || defined(FREEBSD)
				wait(0);
#endif
			} else if ( msg.status == CLOSE_OK ) {
				infop->error_flags = 0;
				infop->sock_fd = -1;
				/*state_change(infop, STATE_MPT_OPEN);*/
				state_change(infop, STATE_TCP_OPEN);			/*Terence "mark MPT" & "change TCP"*/
#if defined(QNX4) || defined (HPUX) || defined(FREEBSD)
				wait(0);
#endif
			} else {
#if defined(QNX4) || defined (HPUX) || defined(FREEBSD)
				wait(0);
#endif
				recv_fail_message(infop);
			}
		}
	}
}

void recv_fail_message(TTYINFO *infop)
{
	char		buf[80];

	close(infop->sock_fd);
	infop->sock_fd = -1;
	state_change(infop, STATE_CONN_FAIL);
	infop->time_out = time(0);
	if ( !(infop->error_flags & ERROR_TCP_CONN) ) {
		struct in_addr tmp_addr;
		tmp_addr.s_addr = infop->ip_addr;
		sprintf(buf, "Socket connect fail (%s, TCP port %d) !",
			inet_ntoa(tmp_addr), infop->tcp_port);
		log_event(buf);
		infop->error_flags |= ERROR_TCP_CONN;
	}
}

void log_event(char * msg)
{
	FILE *		fd;
	time_t	t;
	struct tm	*tt;
	char		tmp[80];

	if ( Restart_daemon )
		return;

	if(Log_Enable){
		t = time(0);
		tt = localtime(&t);
		fd = fopen(EventLog, "a+");
		if ( fd ) {
			sprintf(tmp, "%02d-%02d-%4d %02d:%02d:%02d  ",
				     tt->tm_mon + 1, tt->tm_mday, tt->tm_year+1900,
				     tt->tm_hour, tt->tm_min, tt->tm_sec);
			fputs(tmp, fd);
			fputs(msg, fd);
			fputs("\n", fd);
			fclose(fd);
		}
	}
}
#ifdef	STREAM
int	stream_pty_open (char *pts_name)
{
	int	i, j, fdm[MAX_TTYS];
	char	slavename[40], buf[160];

	i = 0;
	while (i < MAX_TTYS) {

#ifdef	O_NOCTTY
#ifdef	AIX
		fdm[i] = open ("/dev/ptc", O_RDWR | O_NDELAY | O_NOCTTY);
#else
	    fdm[i] = open ("/dev/ptmx", O_RDWR | O_NDELAY | O_NOCTTY);
#endif
#else
#ifdef	AIX
		fdm[i] = open ("/dev/ptc", O_RDWR | O_NDELAY);
#else
	    fdm[i] = open ("/dev/ptmx", O_RDWR | O_NDELAY);
#endif
#endif
	    if (fdm[i] < 0)
		break;
	    grantpt (fdm[i]);
	    unlockpt (fdm[i]);
	    sprintf (slavename, "%s", ptsname (fdm[i]));
	    chmod (slavename, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);  
  	    if (!strcmp (slavename ,pts_name)) {
#if 1
		    if ( (fds = open(slavename,O_RDWR)) < 0 ) {
			    sprintf(buf,"stream_pty_open fail");
			    log_event(buf);
		    } else {
		    	ioctl(fds, I_PUSH, "ptem");
		    	ioctl(fds, I_PUSH, "ldterm");
		    }
#endif		
   		    break;
 	    }
	    i++;
	}
	for (j = i-1;j >= 0;j--)
	    close (fdm[j]);
	if(i >= MAX_TTYS)
		return -1;
	else
		return (fdm[i]);
}
#endif

#ifndef STREAM
void	restart_handle ()
#endif
#ifdef	STREAM
void	restart_handle (int sig)
#endif
{
	Restart_daemon = 1;
#ifndef STREAM
	signal (SIGTERM, ( (void (*)()) wait_handle) );
#endif
#ifdef	STREAM
	sig = sig;
	signal (SIGTERM, wait_handle);
#endif
}


#ifndef STREAM
void	timeout_handle ()
#endif
#ifdef	STREAM
void	timeout_handle (int sig)
#endif
{
	Timeout = 1;
#ifndef STREAM
	signal (SIGALRM, ( (void (*)()) timeout_handle) );
#endif
#ifdef	STREAM
	sig = sig;
	signal (SIGALRM, timeout_handle);
#endif
}


#ifndef STREAM
void	wait_handle ()
#endif
#ifdef	STREAM
void	wait_handle (int sig)
#endif
{
	Restart_daemon = 2;
#ifndef STREAM
	signal (SIGTERM, ( (void (*)()) wait_handle) );
#endif
#ifdef	STREAM
	sig = sig;
	signal (SIGTERM, wait_handle);
#endif
}


#ifndef STREAM
void	config_changed_handle ()
#endif
#ifdef	STREAM
void	config_changed_handle (int sig)
#endif
{
	moxattyd_change_config();
	poll_async_server_init();
	Config_changed = 1;
#ifndef STREAM
	signal (SIGUSR1, ( (void (*)()) config_changed_handle) );
#endif
#ifdef	STREAM
	sig = sig;
	signal (SIGUSR1, config_changed_handle);
#endif
}
/*
#ifndef STREAM
void	chld_handle ()
#endif
#ifdef	STREAM
void	chld_handle (int sig)
#endif
{

	pid_t pid;
	int stat;
	
	while((pid = waitpid(-1, &stat, WNOHANG)) > 0)

}*/

void statelog_fname(char *logfname, char *workpath)
{
	FILE *		fd;
	char		tmp[8], fname[256], tttt[80];

	sprintf(fname, "%s/logindex", workpath);
	if ( (fd = fopen(fname, "r+")) == NULL ) {
		sprintf(logfname,"%s/state_0.log", workpath);
		if ( (fd = fopen(fname, "w+")) != NULL ) {
			fputs("1\n", fd);
			fclose(fd);
		}
		return;
	}
	fgets(tmp, 2, fd);
	if ( tmp[0] < '0' || tmp[0] > '9' )
		tmp[0] = '0';
	tmp[1] = tmp[0] + 1;
	tmp[2] = '\n';
	tmp[3] = 0;
	if ( tmp[1] > '9' )
		tmp[1] = '0';
	rewind(fd);
	fputs(&tmp[1], fd);
	fclose(fd);
	sprintf(logfname,"%s/state_%c.log", workpath, tmp[0]);
}

void state_change(TTYINFO *infop, int new_state)
{
	if ( infop->state != new_state ) {
		infop->state = new_state;
		infop->last_state_time = time(0);
		if ( new_state == STATE_RW_DATA ) {
			infop->tty_opens = 0;
			infop->tcp_opens = 0;
		}
	}
}

void log_state(void)
{
	int		i, dttys;
	TTYINFO *	infop;
	FILE *		fd;
	time_t		t, t2;
	struct tm	*tt;
	char		tmp[128];

	if(Log_Enable){
		t = time(0);
		if ( ((t - last_log_time) < 60) && (task_state >= 80) ) {
			if ( (dttys = ttys) > 16 )
				dttys = 16;
			infop = ttys_info;
			for ( i=0; i<dttys; i++, infop++ ) {
				if ( log_tty_state[i] != infop->state )
					break;
				if ( log_tcp_state[i] != infop->tcp_state )
					break;
			}
			if ( i >= dttys )
				return;
		}
	
		if ( (fd = fopen(StateLog, "w+")) == NULL )
			return;
		last_log_time = t;
		tt = localtime(&t);
		sprintf(tmp, "%02d-%02d-%4d %02d:%02d:%02d  ",
			     tt->tm_mon + 1, tt->tm_mday, tt->tm_year+1900,
			     tt->tm_hour, tt->tm_min, tt->tm_sec);
		fputs(tmp, fd);
		sprintf(tmp, "Total_ttys=%d  ", ttys);
		fputs(tmp, fd);
		sprintf(tmp, "Err1_count=%d Err2_count=%d\n", err1_count, err2_count);
		fputs(tmp, fd);
		sprintf(tmp, "Task state = %d\n", task_state);
		fputs(tmp, fd);
		sprintf(tmp, "Last rw for %d seconds\n", t - last_rw_time);
		fputs(tmp, fd);
		if ( task_state < 80 )
			dttys = 0;
		else
			dttys = ttys;

		infop = ttys_info;
		for ( i=0; i<dttys; i++, infop++ ) {
			if ( infop->state == STATE_CONN_FAIL ||
			     infop->state == STATE_REMOTE_LISTEN )
				t2 = t - infop->time_out;
			else
				t2 = 0;
			sprintf(tmp, "%s state=%1d o1=%d o2=%d tcp=%d ts=%d tout=%04d time=%d tk=%d/%d sk=%d/%d\n",
				     infop->mpt_name,
				     infop->state,
				     infop->tty_opens,
				     infop->tcp_opens,
				     infop->tcp_port,
				     infop->tcp_state,
				     (int)t2,
				     t - infop->last_state_time,
				     infop->mpt_datakeep,
				     t - infop->last_t2s_time,
				     infop->sock_datakeep,
				     t - infop->last_s2t_time
				);
			if ( i < 16 ) {
				log_tty_state[i] = infop->state;
				log_tcp_state[i] = infop->tcp_state;
			}
			fputs(tmp, fd);
		}
		fclose(fd);
	}
}
