/***********************************************************************************************
    File name:	Example.cpp
 
    Description:
	This program demo how to use MOXA NECI Lib to configure Network Enabler from PC Host. 
	
	The Hardware connection is following:
	PC Host--- ~~LAN~~ ---[Network Enabler]
	
	This program demo:
		how to search Network Enabler and get information from it. 
		how to set ethernet port IP Address. 
		how to set serial port speed.

	Use function:
		dsc_SetSearchOpt,	dsc_GetSerialInterface,		dsc_GetKernelInfo,
		dsc_EnumSearch,		dsc_GetSerialIOCtl,			dsc_GetAPInfo,	
		dsc_Attach,			dsc_GetIfConfig,			dsc_GetName,		
		dsc_Login,			dsc_SetSerialIOCtl,			dsc_SetIP,	
		dsc_Logout,			dsc_SaveAndRestart,			dsc_Detach
	
    History:    
    	Version		Date			Comment
		1.2			09-19-2007		Add dsc_Detach call.
									Increate buffer for DIO channels.
		1.1			10-13-2003		Network Enabler Support
        1.0			02-18-2003		Create
************************************************************************************************/
#include	<stdio.h>
#include	<conio.h>
#include	<new>
#include	<winsock2.h>
#include	"neciDef.h"

#define		SET_IPADDRESS	0
#define		SET_SERIALSPEED	1

#define		PASS_MASK		0x01

static CFGINFO	SearchInfo[MAX_DEVICE_NUM];
static int		search_count;
static int		BaudrateBuf[17] = { 50, 75, 110, 134, 150, 300, 600, 1200, 1800, 2400, 4800,
									7200, 9600, 19200, 38400, 57600, 115200};

int CALLBACK EnumSearchProc(PDS_INFO dsinfo);
char Menu_Search(void);
int	Searching(int mode, DWORD search_ip);
void ShowSearchResult(void);
void Configuration(int mode);
char Menu_Config(void);
void ShowBasicInfo(CFGINFO *cfg_info);
void ShowDIOInfo(CFGINFO *cfg_info);
void SetIPAddress(CFGINFO *cfg_info);
void SetSerialSpeed(CFGINFO *cfg_info);
int SavetoPDS(CFGINFO *cfg_info, char *passwd, int mode, DWORD ip, BYTE baudrate);
DWORD GetInputIP(int mode);

void main(void)
{
	bool			main_flag;
	DWORD			search_ip;

	printf("Welcome to NECI Lib Example for Console Application!\n\n");
	main_flag = true;
	search_count = 0;

	while (main_flag) {
		switch(Menu_Search()) {
		case 'a':
			printf("Broadcast Searching......");
			if (!Searching(BROADCAST, 0)) {
				printf("%d Network Enabler found.\n\n", search_count);
				ShowSearchResult();
				Configuration(BROADCAST);	
			}
			break;
		case 'b':
			search_ip = GetInputIP(0);
			printf("Search By IP Address......");
			if (!Searching(SEARCH_BY_IP, search_ip)) {
				printf("OK.\n\n");
				ShowSearchResult();
				Configuration(SEARCH_BY_IP);
			}
			break;
		case 'q':
			printf("Example Program End !\n");
			main_flag = false;
			break;
		default:
			printf("There isn't this option.\n\n");
			break;
		}
	}
	exit(0);
}

char Menu_Search(void)
{
	char	ch;

	printf("Please select a method to search Network Enabler, or Exit:\n");
	printf("a :Broadcast Searching\n");
	printf("b :Search by IP\n");
	printf("q :Exit\n\n");

	printf("Your choice: ");
	scanf(" %c", &ch);

	return ch;
}

int	Searching(int mode, DWORD search_ip)
{
	int		i, ret, retcnt;

	for(i = 0; i < search_count; i++)
	{
		if(SearchInfo[i].dshdl >= 0)
			dsc_Detach(SearchInfo[i].dshdl);
	}
	search_count = 0;

	// Set Search timeout and try count
	dsc_SetSearchOpt(SEARCH_TIMEOUT, SEARCH_RETRY_CNT);

	if (mode == BROADCAST) {
		if ((ret=dsc_EnumSearch(INADDR_BROADCAST, EnumSearchProc)) != DSC_OK) {
			printf("dsc_EnumSearch() fail, error code: %d.\n\n",ret);
			return -1;
		}
	} else {
		if ((ret=dsc_EnumSearch(search_ip, EnumSearchProc)) != DSC_OK) {
			printf("dsc_EnumSearch() fail, error code: %d.\n\n",ret);
			return -1;
		}
	}

	if (search_count <= 0) {
		printf("No Network Enabler found !\n\n");
		return -1;
	}

	for (i=0; i<search_count; i++) {
		if ((ret=dsc_Attach(&SearchInfo[i].ds_info, &SearchInfo[i].dshdl)) != DSC_OK) {
			printf("dsc_Attach() fail, error code: %d.\n\n",ret);
			return -1;
		}
		// Get ip address, netmask, gateway
		SearchInfo[i].p_ifconfig = new IFCONFIGINFO;
		if ((ret=dsc_GetIfConfig(SearchInfo[i].dshdl, 1, &retcnt, SearchInfo[i].p_ifconfig)) != DSC_OK) {
			printf("dsc_GetIfConfig() fail, error code: %d.\n\n",ret);
			return -1;
		}
		// Get serial interface
		if ((ret=dsc_GetSerialInterface(SearchInfo[i].dshdl, 1, SERINIT, &SearchInfo[i].port_type)) != DSC_OK) {
			printf("dsc_GetSerialInterface() fail, error code: %d.\n\n",ret);
			return -1;
		}
		// Get serial port baudrate, mode, flowctrl setting
		SearchInfo[i].pserial_info = new SERPARMITEM;
		if ((ret=dsc_GetSerialIOCtl(SearchInfo[i].dshdl, 1, SERINIT, SearchInfo[i].pserial_info)) != DSC_OK) {
			printf("dsc_GetSerialIOCtl() fail, error code: %d.\n\n",ret);
			return -1;
		}
		// Get Server name
		if ((ret=dsc_GetName(SearchInfo[i].dshdl, SearchInfo[i].svr_name)) != DSC_OK) {
			printf("dsc_GetName() fail, error code: %d.\n\n",ret);
			return -1;
		}
		//Get the kernel/firmware information.
		SearchInfo[i].pkernel_info = new KERNELINFO;
		if ((ret=dsc_GetKernelInfo(SearchInfo[i].dshdl, SearchInfo[i].pkernel_info)) != DSC_OK) {
			printf("dsc_GetKernelInfo() fail, error code: %d.\n\n",ret);
			return -1;
		}
		//Get DIO information information.
        if ((ret=dsc_GetMaxDIO(SearchInfo[i].dshdl, &SearchInfo[i].DIO_max)) != DSC_OK) {
			printf("dsc_GetDIOMode() fail, error code: %d.\n\n",ret);
			return -1;
		}
		if ((ret=dsc_GetDIOMode(SearchInfo[i].dshdl, 0,SearchInfo[i].DIO_max,SearchInfo[i].DIO_mode)) != DSC_OK) {
			printf("dsc_GetDIOMode() fail, error code: %d.\n\n",ret);
			return -1;
		}
		if ((ret=dsc_GetDIOState(SearchInfo[i].dshdl, 0,SearchInfo[i].DIO_max,SearchInfo[i].DIO_state)) != DSC_OK) {
			printf("dsc_GetDIOState() fail, error code: %d.\n\n",ret);
			return -1;
		}
	
	}

	return 0;
}

int CALLBACK EnumSearchProc(PDS_INFO dsinfo)
{
	if (dsinfo == NULL)
		return true;
	
	SearchInfo[search_count].dshdl = -1;
	if (dsinfo->hwid == NE4100T) {
		printf("Found: NE4100T\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == NE4110S) {
		printf("Found: NE4110S\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == NE4110A) {
		printf("Found: NE4110A\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == NE4120S) {
		printf("Found: NE4120S\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == NE4120A) {
		printf("Found: NE4120A\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == WE2100T_US) {
		printf("Found: WE2100T_US\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == WE2100T_EU) {
		printf("Found: WE2100T_EU\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}else if (dsinfo->hwid == WE2100T_JP) {
		printf("Found: WE2100T_JP\n");
		memcpy(&SearchInfo[search_count].ds_info, dsinfo, sizeof(DS_INFO));
		search_count++;
	}

	return true;
}

void ShowSearchResult(void)
{
	int				i, j;
	struct in_addr	ip;

	printf("Search Result:\n");
	printf("Num	Model\t\tIP Address\t\tMAC Address\n");
	for(i=0; i<search_count; i++) {
		ip.S_un.S_addr = SearchInfo[i].p_ifconfig->ipaddr;
		if (SearchInfo[i].ds_info.hwid == NE4100T) {
			printf("%d\t%s\t%s\t\t", i, "NE-4100T", inet_ntoa(ip));
		} else if (SearchInfo[i].ds_info.hwid == NE4110S) {
			printf("%d\t%s\t%s\t\t", i, "NE-4110S", inet_ntoa(ip));
        }else if (SearchInfo[i].ds_info.hwid == NE4110A) {
			printf("%d\t%s\t%s\t\t", i, "NE-4110A", inet_ntoa(ip));
        }else if (SearchInfo[i].ds_info.hwid == NE4120S) {
			printf("%d\t%s\t%s\t\t", i, "NE-4120S", inet_ntoa(ip));
        }else if (SearchInfo[i].ds_info.hwid == NE4120A) {
			printf("%d\t%s\t%s\t\t", i, "NE-4120A", inet_ntoa(ip));
        }else if (SearchInfo[i].ds_info.hwid == WE2100T_US) {
			printf("%d\t%s\t%s\t\t", i, "WE-2100T-US", inet_ntoa(ip));
        }else if (SearchInfo[i].ds_info.hwid == WE2100T_EU) {
			printf("%d\t%s\t%s\t\t", i, "WE-2100T_EU", inet_ntoa(ip));
        }else if (SearchInfo[i].ds_info.hwid == WE2100T_JP) {
			printf("%d\t%s\t%s\t\t", i, "WE-2100T-JP", inet_ntoa(ip));
        }
		for(j=0; j<6; j++) {
			printf("%02X", SearchInfo[i].ds_info.mac[j]);
			if (j<5)
				printf(":");
		}
		printf("\n");
	}
	printf("\n");
}

void Configuration(int mode)
{
	int		pds_num;
	char	ch[5];
	bool	Config_flag;

	if (mode == BROADCAST) {
		printf("Please select a target to  control it.(Select the Num column)\n");
		printf("Your choice: ");
		scanf("%s", ch);
		pds_num = atoi(ch);
		if (pds_num == 0) {
			if (strcmp("0", ch) != 0)
				pds_num = -1;
		}
	} else if (mode == SEARCH_BY_IP)
		pds_num = 0;

	if (pds_num < 0 || pds_num >= search_count) {
		printf("There isn't this Num.\n\n");
		return;
	}
	Config_flag = true;

	while (Config_flag) {
		switch(Menu_Config()) {
		case '1':
			ShowBasicInfo(&SearchInfo[pds_num]);
			break;
		case '2':
			SetIPAddress(&SearchInfo[pds_num]);
			Config_flag = false;
			break;
		case '3':
			SetSerialSpeed(&SearchInfo[pds_num]);
			Config_flag = false;
			break;
        case '4':
			ShowDIOInfo(&SearchInfo[pds_num]);
	    break;
		case '5':
			printf("Quit from Configuration.\n\n");
			Config_flag = false;
			break;
		default:
			printf("There isn't this option.\n");
			break;
		}
	}
}

char Menu_Config(void)
{
	char	ch;

	printf("\nConfigurate the Network Enabler or Quit:\n");
	printf("1 :Get device Information\n");
	printf("2 :Set Ethernet port IP Address\n");
	printf("3 :Set Serial port Speed\n");
	printf("4 :Get DIO Settings\n");
	printf("5 :Quit\n\n");

	printf("Your choice: ");
	scanf(" %c", &ch);

	return ch;
}

void ShowBasicInfo(CFGINFO *cfg_info)
{
	struct in_addr	ip;
	char			buf[30];

	printf("\nHW Model\tServer Name\t\tFirmware Version\n");
	if(cfg_info->ds_info.hwid == NE4100T){
		printf("%s\t", "NE-4100T");	
	}else if(cfg_info->ds_info.hwid == NE4110S){
		printf("%s\t", "NE-4110S");	
	}else if(cfg_info->ds_info.hwid == NE4110A){
		printf("%s\t", "NE-4110A");	
	}else if(cfg_info->ds_info.hwid == NE4120S){
		printf("%s\t", "NE-4120S");	
	}else if(cfg_info->ds_info.hwid == NE4120A){
		printf("%s\t", "NE-4120A");	
	}else if(cfg_info->ds_info.hwid == WE2100T_US){
		printf("%s\t", "WE-2100T-US");	
	}else if(cfg_info->ds_info.hwid == WE2100T_EU){
		printf("%s\t", "WE-2100T-EU");	
	}else if(cfg_info->ds_info.hwid == WE2100T_JP){
		printf("%s\t", "WE-2100T-JP");	
	}
	
	printf("%s\t\t", cfg_info->svr_name);

	if((((cfg_info->pkernel_info->firm_version >> 8) & 0xFF) == 0) ||
	   (((cfg_info->pkernel_info->firm_version >> 8) & 0xFF) == 0x30)) {
		sprintf(buf, "%X.%X", ((cfg_info->pkernel_info->firm_version) >> 24) & 0xff,
							  ((cfg_info->pkernel_info->firm_version) >> 16) & 0xff);
	} else {
		sprintf(buf, "%X.%X.%X", ((cfg_info->pkernel_info->firm_version) >> 24) & 0xff,
								 ((cfg_info->pkernel_info->firm_version) >> 16) & 0xff,  
								 ((cfg_info->pkernel_info->firm_version) >> 8) & 0xff);
	}
	printf("\tV%s\n\n", buf);


	printf("IP Address\tSerial Port Type\n");
	ip.S_un.S_addr = cfg_info->p_ifconfig->ipaddr;
	printf("%s\t", inet_ntoa(ip));
	switch(cfg_info->port_type) {
	case IF_RS232:
		printf("RS-232\t");
		break;
	case IF_RS422:
		printf("RS-422\t");
		break;
	case IF_RS4852W:
		printf("RS-485, 2wire\t");
		break;
	case IF_RS4854W:
		printf("RS-485, 4wire\t");
		break;
	default:
		printf("unknown\t");
		break;
	}
	/*sprintf(buf,"%x.%x.%x.%x",((cfg_info->pap_info->version & 0xff000000) >>24),
							  ((cfg_info->pap_info->version & 0x00ff0000) >> 16),
                              ((cfg_info->pap_info->version & 0x0000ff00) >> 8),
                              ((cfg_info->pap_info->version & 0x000000ff)));
	printf("\t\tV%s\n", buf);*/
}

void ShowDIOInfo(CFGINFO *cfg_info)
{
	int i;
	printf("DIO\tMode\tState\n");
	for(i=0; i< MAXDIO; i++)
		printf("%d\t%s\t%s\n", i, cfg_info->DIO_mode[i]?"Output":"Input",cfg_info->DIO_state[i]?"High":"Low"); 
}

void SetIPAddress(CFGINFO *cfg_info)
{
	char	password[30];
	DWORD	ipaddr;
	int		i;

	// Get set new IP Address from keyboard
	ipaddr = GetInputIP(1);

	// Password Checking if password enabled
	sprintf(password, "");
	if (cfg_info->pkernel_info->flag & PASS_MASK) {
		printf("Please input Password: ");
		i = 0;
		while(1) {
			password[i] = getch();
			if (password[i] != 0x0d) {
				printf("*");
				i++;
			} else {
				password[i] = NULL;
				break;
			}
		}
		printf("\n");
	}

	printf("Set IP Address to PDS......");
	if (!SavetoPDS(cfg_info, password, SET_IPADDRESS, ipaddr, NULL)) 
		printf("OK\n");

	printf("\n");
}

int SavetoPDS(CFGINFO *cfg_info, char *passwd, int mode, DWORD ip, BYTE baudrate)
{
	int		ret;

	if ((ret=dsc_Login(cfg_info->dshdl, passwd)) != DSC_OK) {
		printf("dsc_Login() fail, error code: %d.\n",ret);
		return -1;
	}

	if (mode == SET_IPADDRESS) {
		if ((ret=dsc_SetIP(cfg_info->dshdl, cfg_info->p_ifconfig->ifid, ip)) != DSC_OK) {
			printf("dsc_SetIP() fail, error code: %d.\n",ret);
			dsc_Logout(cfg_info->dshdl);
			return -1;
		}
	}

	if (mode == SET_SERIALSPEED) {
		if ((ret=dsc_SetSerialIOCtl(cfg_info->dshdl, 1, baudrate, cfg_info->pserial_info->mode, cfg_info->pserial_info->flowctrl)) != DSC_OK) {
			printf("dsc_SetSerialIOCtl() fail, error code: %d.\n",ret);
			dsc_Logout(cfg_info->dshdl);
			return -1;
		}
	}

	if ((ret=dsc_SaveAndRestart(cfg_info->dshdl)) != DSC_OK) {
		printf("dsc_SaveAndRestart() fail, error code: %d.\n",ret);
		dsc_Logout(cfg_info->dshdl);
		return -1;
	}

	if ((ret=dsc_Logout(cfg_info->dshdl)) != DSC_OK) {
		printf("dsc_Logout() fail, error code: %d.\n",ret);
		return -1;
	}

	return 0;
}

void SetSerialSpeed(CFGINFO *cfg_info)
{
	int		i;
	BYTE	baud;
	char	buf[10], password[30];

	//Get new baudrate to be set
	printf("Please select a baudrate to be set:\n");
	for(i=0; i<17; i++) {
		printf("%d\t:", i);
		sprintf(buf, "%d", BaudrateBuf[i]);
		printf("%s\n", buf);
	}
	printf("\nYour choice: ");
	scanf("%s", buf);
	baud = atoi(buf);

	// Password Checking if password enabled
	sprintf(password, "");
	if (cfg_info->pkernel_info->flag  & PASS_MASK) {
		printf("Please input Password: ");
		i = 0;
		while(1) {
			password[i] = getch();
			if (password[i] != 0x0d) {
				printf("*");
				i++;
			} else {
				password[i] = NULL;
				break;
			}
		}
		printf("\n");
	}

	printf("Set Serial Speed to PDS......");
	if (!SavetoPDS(cfg_info, password, SET_SERIALSPEED, NULL, baud)) 
		printf("OK\n");
	printf("\n");
}

DWORD GetInputIP(int mode)
{
	char	ipbuf[30];

	if (mode) 
		printf("Please input new IP address: ");
	else
		printf("Please input IP Address to be searched: ");

	scanf("%s",ipbuf);
	
	return (inet_addr(ipbuf));
}