﻿using CommandLine;
using CommandLine.Text;
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace mx_sim_ctl
{
    class Program
    {
        class Options
        {
            [Option('s', Required = true, HelpText = "-s <#Module socket index> (Start from 1)")]
            public uint SocketIndex { get; set; }

            [Option('i', Default = null,
                HelpText = "-i <SIM slot>\n" +
                " 		1 --> SIM slot 1\n" +
                "		2 --> SIM slot 2\n"
                )]
            public int? SIMSlot { get; set; }

            [Usage(ApplicationAlias = "mx-sim-ctl")]
            public static IEnumerable<Example> Examples
            {
                get
                {
                    yield return new Example("Get SIM slot from module socket 1", new Options { SocketIndex = 1 });
                    yield return new Example("Set module socket 1 to SIM slot 2", new Options { SocketIndex = 1, SIMSlot = 2 });
                }
            }
        }

        [DllImport("mxsim.dll", EntryPoint = "GetSIMSlot", CharSet = CharSet.Ansi, SetLastError = true)]
        public static extern int GetSIMSlot(int socket);

        [DllImport("mxsim.dll", EntryPoint = "SetSIMSlot", CharSet = CharSet.Ansi, SetLastError = true)]
        public static extern int SetSIMSlot(int socket, int mode);

        public struct Action_Struct
        {
            public int index;
            public int slot;
        };

        static void Main(string[] args)
        {
            CommandLine.Parser.Default.ParseArguments<Options>(args)
                .WithParsed(opts =>
                {
                    if (opts.SIMSlot == null)
                    {
                        GetModeAction(opts);
                    }
                    else
                    {
                        SetModeAction(opts);
                    }
                });
        }

        private static void SetModeAction(Options opts)
        {
            Action_Struct action;
            action.index = (int)opts.SocketIndex;
            if (SetSIMSlot(action.index - 1, (int)opts.SIMSlot - 1) < 0)
            {
                Console.WriteLine("Failed to set SIM slot of module socket {0}.", action.index);
            }
            else
            {
                action.slot = GetSIMSlot(action.index - 1) + 1;
                if (action.slot >= 0)
                {
                    Console.WriteLine("Module socket {0} SIM slot: {1}", action.index, action.slot);
                }
            }
        }

        private static void GetModeAction(Options opts)
        {
            Action_Struct action;
            action.index = (int)opts.SocketIndex;
            action.slot = GetSIMSlot(action.index - 1) + 1;
            if (action.slot >= 0)
            {
                Console.WriteLine(string.Format("Module socket {0} SIM slot: {1}", action.index, action.slot));
            }
        }
    }
}
