/*
Copyright (C) MOXA Inc. All rights reserved.

This software is distributed under the terms of the
MOXA License.  See the file COPYING-MOXA for details.
*/
#include <windows.h>
#include<stdio.h>
#include<stdlib.h>
#include<getopt.h>  
#pragma comment(lib,"getopt.lib") 
#include "mxdgio.h"
using namespace std;

#define UNSET 	-1

enum action_type {
	GET_DIN = 0,
	GET_DOUT = 1,
	SET_DOUT = 2
};

struct action_struct {
	int type;
	int port;
	int mode;
};

void usage(FILE *fp)
{
	fprintf(fp, "Usage:\n");
	fprintf(fp, "	mx-dio-ctl <-i #DIN |-o #DOUT > -m <status>\n\n");
	fprintf(fp, "OPTIONS:\n");
	fprintf(fp, "	-i <#DIN>\n");
	fprintf(fp, "		Get target to DIN port\n");
	fprintf(fp, "		Get DIN index: [0..] \n");
	fprintf(fp, "	-o <#DOUT>\n");
	fprintf(fp, "		Get target to DOUT port\n");
	fprintf(fp, "		Get DOUT index: [0..] \n");
	fprintf(fp, "	-m <status>\n");
	fprintf(fp, "		Set status for target DOUT port\n");
	fprintf(fp, "		0 --> LOW\n");
	fprintf(fp, "		1 --> HIGH\n");
	fprintf(fp, "\n");
	fprintf(fp, "Example:\n");
	fprintf(fp, "	Get value from DIN port 1\n");
	fprintf(fp, "	# mx-dio-ctl -i 1\n");
	fprintf(fp, "\n");
	fprintf(fp, "	Get value from DOUT port 1\n");
	fprintf(fp, "	# mx-dio-ctl -o 1\n");
	fprintf(fp, "\n");
	fprintf(fp, "	Set DOUT port 2 value to LOW\n");
	fprintf(fp, "	# mx-dio-ctl -o 2 -m 0\n");
}

int my_atoi(const char *nptr, int *number)
{
	int tmp;

	tmp = atoi(nptr);
	if (tmp != 0) {
		*number = tmp;
		return 0;
	}
	else {
		if (!strcmp(nptr, "0")) {
			*number = 0;
			return 0;
		}
	}
	return -1;
}

void do_action(struct action_struct action)
{
	switch (action.type) {
	case GET_DIN:
		action.mode = GetDinStatus(action.port);
		if (action.mode < 0) {
			fprintf(stderr, "Failed to get DIN index\n");
			exit(1);
		}
		printf("DIN port %d status: %d\n", action.port, action.mode);
		break;
	case GET_DOUT:
		action.mode = GetDoutStatus(action.port);
		if (action.mode < 0) {
			fprintf(stderr, "Failed to get DOUT index\n");
			exit(1);
		}
		printf("DOUT port %d status: %d\n", action.port, action.mode);
		break;
	case SET_DOUT:
		if (SetDoutStatus(action.port, action.mode) < 0) {
			fprintf(stderr, "Failed to set DOUT port %d value\n", action.port);
			exit(1);
		}
		printf("DOUT port %d status: %d\n", action.port, action.mode);
		break;
	}
}

int main(int argc, char**argv)
{
	struct action_struct action = { UNSET, UNSET, UNSET };
	int c;

	while (1) {
		c = getopt(argc, argv, "hi:o:m:");
		if (c == -1)
			break;

		switch (c) {
		case 'h':
			usage(stdout);
			exit(0);
		case 'i':
			if (action.type != UNSET) {
				fprintf(stderr, "action has already set\n");
				usage(stderr);
				exit(99);
			}
			action.type = GET_DIN;
			if (my_atoi(optarg, &action.port) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 'o':
			if (action.type != UNSET) {
				fprintf(stderr, "action has already set\n");
				usage(stderr);
				exit(99);
			}
			action.type = GET_DOUT;
			if (my_atoi(optarg, &action.port) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 'm':
			action.type = SET_DOUT;
			if (my_atoi(optarg, &action.mode) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		default:
			usage(stderr);
			exit(99);
		}
	}

	if (optind < argc) {
		fprintf(stderr, "non-option arguments found: ");
		while (optind < argc)
			fprintf(stderr, "%s ", argv[optind++]);
		fprintf(stderr, "\n");

		usage(stderr);
		exit(99);
	}

	if (action.port == UNSET) {
		fprintf(stderr, "port number is unset\n");
		usage(stderr);
		exit(99);
	}

	do_action(action);
	exit(0);
}