# MXviewOne API Guide

This guide is for developers who wish to use MXviewOne API for their own system.

## Change Log

2019/06/16 First created for MXview V3.1
2020/10/13 Add more detail for the guide
2025/04/17 Adjust the HTTPS example and Event Types table 

## Document

You can read MXviewOne API document by accessing [/doc](/doc) after you start the MXviewOne service.

## To Start

To start with MXviewOne RESTful API, you must *login* the MXviewOne server, the Get Authenticated part describes how you login and how you get the token.

## Get Authenticated

3rd party system has to get authenticated before requesting MXviewOne RESTful API.

MXviewOne uses JWT(JSON web token) to authenticate users. Developers can either login to MXviewOne to get a limited-time token or generate an API key in API Key Management.
To login an MXviewOne server, you POST [/login](/login) with username and password in the request body. If succeeds, the server would respond with a token. You can use that
as a bearer token in the authentication header in the next requests.

The token you get by login MXview can only be used in 30 minutes. You have to either refresh the token every 30 minutes or if you need a permenant token,
you have to generate your own API key via API Key Management.
Remember that, if you get the token by login MXview, after all data fetching is done, call the API *Logout* to release the MXview connection.

### An example
```
import sys, requests

##Enter credentials
URL = "https://myServerAddress/login/"
username = "myusername"
password = "mypassword"

##Enter post body
Data = {
            "username": username,
            "password": password
        }

##Begin POST request
try:
    r = requests.post(url = URL, data = Data, verify="/path/to/certfile") 
	## r = requests.post(url = URL, data = Data, verify=False ) # self-signed cert or if you want to skip cert check
except requests.exceptions.RequestException as e:
    print(e)
    exit()


##Check for errors
if r:
    ##If request is successful, 
    ##deserialise JSON response into dictionary
    response = r.json()
    
    ##Output the status code and bearer token
    print("Code: ", r.status_code, '\n', "Bearer Token: ", response['mxviewGateway']) 
    
else:
    ##If request is unsuccessful,
    ##output the status code and error message
    print("Code: " + str(r.status_code), '\n', r.text)
```
        
## Get Data from MXviewOne

After you get a valid JWT, you can send requests to an MXviewOne server by putting the token to HTTP's authorization header as a bearer token (Bearse {{token}}).
The Content-type could be either application/json or application/x-www-urlencoded.

### An example
```
URL = "https://myServerAddress/api/devices/"
    
token = response['mxviewGateway'] ##User must login to receive token
    
##Bearer token must be included in header
header = {
            'Authorization': 'Bearer ' + token
         }

#Begin GET request
try:
    r = requests.get(url = URL, headers = header, verify="/path/to/certfile")
	## r = requests.get(url = URL, headers = header, verify=False ) # self-signed cert or if you want to skip cert check
except requests.exceptions.RequestException as e:
    print(e)
    exit()

#Check for errors
if r:
    ##If request is successful,
        
    ##Output the entire result
    print(r.text + "\n\n")
        
    ##OR Deserialise JSON response into dictionary and
    ##output the site id of the first device  
    response = r.json()
    print(response[0]['site_id'])
        
else:
    print("Code: ", str(r.status_code), '\n', r.text)
```

## MXviewOne Site

From MXview V3.0, it supports client-server architecture to connect multiple MXview sites with each other. Each site is represented by a "Site ID". So if you want to query a device within a particular site, you must include the site id in your query. For example, if you want to query device "192.168.127.1" within site "e88dfb9cd7a3e8e6", you have to use the following api

```
"https://myServerAddress/api/devices/ip/192.168.127.1/site/e88dfb9cd7a3e8e6"
```

To get the Site ID, you can query 

```
"https://myServerAddress/api/sites"
```
And the response would be something like the following json object

```js
[
  {
    "status": true,
    "license": {
      "status": 0
    },
    "site_id": "e88dfb9cd7a3e8e6",
    "severity": 3,
    "site_name": "Site DEMO PC",
    "site_description": ""
  }
]
```
## Get device information

To get devices from a site, you can query 
```
https://myServerAddress/api/device/site/{site_id}
```

the response would be an array as follows,

```js
[{
	"id": 7,
	"snmp_username": "admin",
	"write_community": "private",
	"status": 0,
	"group": 0,
	"read_community": "public",
	"model": "EDS-510E",
	"x": 900,
	"ip": "192.168.127.71",
	"severity": 1,
	"redundancy": 0,
	"icmp_polling_interval": 10,
	"snmp_unreachable_warning_threshold": 1,
	"alias": "192.168.127.71--EDS-510E",
	"location": "Switch Location",
	"snmpv3_authentication_type": 1,
	"mac": "0090E84BD034",
	"sysobjid": ".1.3.6.1.4.1.8691.7.84",
	"icon_url": "dev_images/EDS_510E.png",
	"icmp_unreachable_warning_threshold": 1,
	"icon_id": 108,
	"snmpv3_privacy_protocol": 1,
	"snmp_polling_interval": 60,
	"snmp_version": 1,
	"snmp_password": "",
	"snmpv3_security_level": 1,
	"y": 150,
	"router_role": 0,
	"prphsr_role": "",
	"device_components": {
		"Alias": {
			"status": "192.168.127.71--EDS-510E"
		},
		"ModelName": {
			"status": "EDS-510E"
		},
		"MAC Address": {
			"status": "0090E84BD034"
		},
		"Availability": {
			"status": "100.00%"
		},
		"sysDescr": {
			"status": "EDS-510E-3GTXSFP",
			"display": 2
		},
		"sysObjectId": {
			"status": "1.3.6.1.4.1.8691.7.84",
			"display": 2
		},
		"sysContact": {
			"status": "",
			"display": 2
		},
		"sysName": {
			"status": "",
			"display": 2
		},
		"sysLocation": {
			"status": "Switch Location",
			"display": 2
		},
		"ifNumber": {
			"status": "13",
			"display": 2
		},
		"macAddress": {
			"status": "0090E84BD034",
			"display": 2
		},
		"ifDescr": [{
			"status": "Ethernet Port 1",
			"display": 3,
			"index": "1"
		},
		{
			"status": "Ethernet Port 2",
			"display": 3,
			"index": "2"
		},
		{
			"status": "Ethernet Port 3",
			"display": 3,
			"index": "3"
		},
		{
			"status": "Ethernet Port 4",
			"display": 3,
			"index": "4"
		},
		{
			"status": "Ethernet Port 7",
			"display": 3,
			"index": "7"
		},
		{
			"status": "Ethernet Port G1",
			"display": 3,
			"index": "8"
		},
		{
			"status": "Ethernet Port G2",
			"display": 3,
			"index": "9"
		},
		{
			"status": "Ethernet Port G3",
			"display": 3,
			"index": "10"
		},
		{
			"status": "Ethernet Trunking Port 1 of EtherDevice Switch",
			"display": 3,
			"index": "11"
		},
		{
			"status": "Management Port of EtherDevice Server",
			"display": 3,
			"index": "129"
		}]
	}
}]
```

the *id* is the unique identifier of a device. The *severity* indicates whether the device is (1: normal, 2: warning, 3: critical). Basic device properties such as Mac address (*mac*) and (*sysobjectid*) are located in the root level of a device data. Other properties polled through SNMP are located in *device_components* object. The properties in it could either be an object (e.g. *sysDescr*) or an array (e.g. *ifDescr*). if it is an array, the *index* indicates the index of a property polled through SNMP.

## Get link information

To get links detected and monitored in MXviewOne, you can query

```
https://myServerAddress/api/links/site/{site_id}
```

The result would be an array as follows,

```js
[
  {
    "to_port_label": "",
    "id": 1,
    "status": 1,
    "from_device": 3,
    "type": 0,
    "from_port": 3,
    "y1": 0,
    "to_device": 9,
    "to_port": 20,
    "severity": 1,
    "use_custom_label": false,
    "redundancy": 0,
    "y4": -1,
    "group": 0,
    "local_vpn_connection_name": "",
    "speed": "1000000000",
    "y2": 0,
    "x3": -1,
    "x6": 0,
    "x1": 0,
    "x2": 0,
    "x4": -1,
    "x5": 0,
    "y3": -1,
    "duplex": "Half",
    "y5": 0,
    "y6": 0,
    "remote_vpn_connection_name": "",
    "vpn_from_ip": "0.0.0.0",
    "vpn_to_ip": "0.0.0.0",
    "count": 0,
    "from_port_label": "",
    "site_id": "e88dfb9cd7a3e8e6"
  },
]
```

The *id* is the unique identifier of a link. The *severity* indicates whether the link is (1: normal, 2: warning, 3: critical). The other important info of a link are *from_device*, *from_port*, *to_device*, *to_port*. the *from_device* and *to_device* mean the link is between which devices by their *id*; the *from_port* and *to_port* mean the link is in which port of either device.

## A simple example to infer network topology

From device and link information, a user can infer the physical network topology. The following is a simplified example of how you infer it.

Assume there are two devices and one links (we omit most properties to keep it simple)

```js
[
    {
        id: 1,
        ip: "192.168.127.1"
    },
    {
        id: 2,
        ip: "192.168.127.2"
    }
]
```

```js
[
    {
        id: 1,
        from_device: 1,
        to_device: 2,
        from_port: 5,
        to_port: 6
    }
]
```

You can infer that there is a physical link between the port 5 of *192.168.127.1* and the port 6 of *192.168.127.2*.

## Realtime Messages
Users can connect to MXviewOne through Socket.io(4.4.x) to receive realtime messages by connecting to https://127.0.0.1/ with a token.
```
var socket = sioClient(`https://localhost/?token=Bearer ${token}`);
```
The realtime messages are formatted as JSON objects, each message would be enclosed in a `trigger_detail` tag. The value of a `trigger_detail` would contain one or more different types of messages, each type of message may contain one or more objects. The value of an object would be identical to the value you access the data through RESTful API.

The following is an example realtime message.
```js
{
	"trigger_detail":{
		"events":[
			{
				"id":30989,
				"device_id":0,
				"source_ip":"127.0.0.1",
				"type":268435477,
				"severity":4,
				"port":0,
				"value":0,
				"threshold":0,
				"event_time":1559532049,
				"user":"admin",
				"detector":1,
				"security_event":false,
				"vpn_connection_name":"",
				"acked":false,
				"site_id":"aca4ab0ae4612c1b"
			}
		]
	}
}

```
### An example (in NodeJS)
```js
// get API TOKEN from MXviewOne
const TOKEN = `eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJ1c2VybmFtZSI6ImFkbWluIiwiaWF0IjoxNjc3MjI3NTMxLCJqdGkiOiI1MDMyODhhOGFkMDE4MzliOWI0NGZmMzg2OTZhZGQ4NDlhOTcxNzEyIn0.UeukWw9P2sUCcTIzyuwTqfYz67tL2YzpGNa7_80kIIU`;
process.env.NODE_TLS_REJECT_UNAUTHORIZED = '0';
const io = require("socket.io-client");
const socket = io("wss://127.0.0.1/?token=" + TOKEN
  , {
    reconnectionDelay: 10000,
    reconnection: true,
    transports: ['websocket'],
    agent: false,
    upgrade: false,
    rejectUnauthorized: false
  });

socket.on("connect", () => {
  console.log('connect');
});

// handle MXviewOne TRIGGER Event
socket.on("TRIGGER", data => {
  console.log('TRIGGER', data);
  if (typeof data.trigger_detail.events !== 'undefined') {
    let events = data.trigger_detail.events;
    // console.log(events);
  }

  else if (typeof data.trigger_detail.site_changed !== 'undefined') {
    let site_changed = data.trigger_detail.site_changed;
    // console.log(site_changed);
  }

});

socket.on('connect_error', err => {
  console.log('connect_error', err);
})

socket.on('disconnect', () => {
  console.log('disconnected');
});

process.on('uncaughtException', (err) => {
  console.error(err);
  try {
      let killTimer = setTimeout(() => {
          process.exit(1);
      }, 1000);
      killTimer.unref();
  } catch (e) {
      logger.error('error when exit', e.stack);
  }
});
```

## Parsing Events

Parsing an event can be tricky because each type of event uses different parameters.
Some parameters are used in all event types, the following code give the description of them.
```js
[
	{
		"id":30989, // event ID
		"device_id":0, // the device related to this event, 0 if it is not a device event
		"source_ip":"127.0.0.1", //the IP of the device related to this event. Empty if MXviewOne server cannot distinguish it.
		"type":268435477, // event type
		"severity":4, // the severity of the event
		"event_time":1559532049, // the unix time this event is detected by MXviewOne
		"detector":1, // indicates whether it is an MXviewOne detected device or a trap
		"acked":false, // indicates whether this event is already acked
		"site_id":"aca4ab0ae4612c1b" // site id
	}
]

```
For event-specific parameters for each event type, please see [Event Types](#event-types).

## Event Types

The following table explains all MXviewOne events and their parameters

### System
|event type|severity|description (Chinese)|description (English)|detail|JSON (RESTful API)|
|----------|--------|-----------|-----------|------|------------------|
|0x10000001||MXviewOne related events| you can just ignore them||
|0x10000002||MXviewOne停止|MXviewOne Server has stopped||
|0x10000005||MXviewOne OPC Server啟動|MXviewOne OPC Server is started||
|0x10000006||MXviewOne OPC Server停止|MXviewOne OPC Server has stopped||
|0x10000007||MXviewOne Syslog server啟動|MXviewOne Syslog Server is started||
|0x10000008||MXviewOne Syslog Server停止|MXviewOne Syslog Server has stopped||
|0x1000000B||MXviewOne系統事件被清除|All MXview One system events are cleared||
|0x1000000C|||All MXview One network and device events are cleared||
|0x10000011||Auto topology完成|Auto topology is finished||
|0x10000012||Auto topology開始|Auto topology is started||
|0x10000013||MXviewOne啟動|MXviewOne is started||
|0x10000014||MXviewOne DB 備份完成|MXviewOne DB has been backed up||
|0x10000015||使用者登入成功: {{param}}|MXviewUserLoginSuccessful|{{param}}: username|
|0x10000016||使用者登入失敗: {{param}}|MXviewUserLoginFailed|{{param}}: username|
|0x10000017||使用者登入失敗: {{param}}|MXviewUserBadLogin|{{param}}: username|
|0x10000018|||MXviewUserLogout|{{param}}: username|
|0x1000001C|||Less than 20% memory available.||
|0x1000001D|||CPU loading has exceeded 85% for 10 minutes consecutively.||
|0x1000001E|||Less than 5 GB of disk space available.||
|0x1000001F|||MXviewServerMQCommunicationLatency||
|0x10000020|||Maximum license limit reached.||
|0x10000021|||A firmware update is available. Please check the Firmware Management page for details.||
|0x10000022|||Job Scheduler Start||
|0x10000023|||Maximum license limit reached.||
|0x10000024|||Password automation has reached the scheduled time and will start executing.||
|0x10000025|||CLI Execute: x||
|0x10000026|||The password for the device was successfully automatically changed.(The password for the device failed to automatically change.)||
|0x10000027|||The temporary account at {{ip}} has been successfully activated(The temporary account with {{ip}} has been successfully deactivated.)||
|0x10000028|Warning||Account auditing has been completed, and the results do not match the baseline.||
|0x10000029|Warning||Account audit failed due to unable to retrieve data from devices at {{ip}}.||
|0x1000002A|Info||Event Message: Consumed {{consume}} points today. Available points: {{available}}.||
|0x1000002B|Info||The available points of IPS licenses is {{available}}, which will be run out in {{days}} days.||
|0x10000038|Info||Account auditing has been completed, and the results match the baseline.(Account audit failed due to unable to retrieve data from all devices.)||
|0x1000003A|||IP conflict detection cannot run because WinPcap/Libpcap is not found||
|0x1000003B|||The device IP found is {{ip}}.||

### Network and Device
|event type|severity|description (Chinese)|description (English)|detail|JSON (RESTful API)|
|----------|--------|-----------|-----------|------|------------------|
|0x00010001|Critical|設備無法存取|Device ICMP unreachable|||
|0x00010002|Warning|設備封包遺失率到達 y%, 超過閥值z% (基於ICMP)|The packet loss rate is y%, over threshold z%|*y%*: packet dropping rate, *z%*: threshold|`{value: y,threshold: z}`|
|0x00010003|Warning|設備封包遺失率到達 y% (超過要緊閥值 z%)|The packet loss rate is y%, over threshold z%|||
|0x00010006|Warning|電源 x 斷電|Power x is down|*x*: The nth power input (1 or 2)|`{trap_detail: x}`|
|0x00010007|Warning|設備無法以SNMP存取|Device SNMP unreachable|||
|0x0001000D|By user preference(default) Warning|設備可用性低於閥值 / 目前設備可用性為 y%, 低於閥值z%|The availability of device is y%, below threshold z%|*y%*: device availability, *z%*: threshold|`{value: y,threshold: z}`|
|0x00010010||設備受到DDOS攻擊|(EDR) The device is under DDoS attack|||
|0x00010011||設備Firewall規則被違反|(EDR) The firewall rules are violated|||
|0x00010012||設備Trusted Access規則被違反|(EDR) The trusted access rules are violated|||
|0x00010013||設備login失敗超出threshold|(EDR) User login failed too many time|||
|0x00010015||GOOSE狀態：超時|GooseTimeout|||
|0x00010016||GOOSE狀態：被篡改|GooseTampered|||
|0x00010017||GOOSE狀態：繼續執行|GooseResume|||
|0x00010018|||Port looping on port x has been detected.|x: Port number|{port: x}|
|0x80010018|Normal (Information)||Port looping on port x has been resolved.|x: Port number|`{port: x}`|
|0x00010019||||L3 firewall policy violation.||
|0x80010019||||Recovered from L3 firewall policy violation.||
|0x0001001A||||Device lock down violation.||
|0x8001001A||||Recovered from device lock down violation.||
|0x0001001B||||IP conflict detected: - MAC: x, NIC Vendor: y|`{mac: x, nic: y}`|
|0x8001001B||||IP conflict recovered||
|0x00010100~0x00020000||自訂事件|A custom event is triggered. x. Threshold=y, value=z. w|*x*: the name of the device property of this custom event<br>*y*: the threshold<br>*z*: the value<br>*w*: detail description of this event|`{eventName:x, threshold: y, value: z, eventDescription: w }`|
|0x00020002|Normal (Informational)<br> * By user preference|埠 x 連線|Port x link up|*x*: Port number|`{port: x}`|
|0x00020001|Critical<br> * By user preference|埠 x 斷線|Port x link down|*x*: Port number|`{port: x}`|
|0x80020001|Normal (Information)|埠 x 恢復連接|Port x link up|*x*: Port number|`{port: x}`|
|0x80020002|Critical|埠 x 斷線|Port x link down|*x*: Port number|`{port: x}`|
|0x00020005|Warning<br> * By user preference|埠 x 輸入頻寬利用率到達 y% (超過閥值 z%)|The input traffic utilization of port x is y%, over threshold z%|*x*: Port number, *y%*: link usage, *z%*: link usage threshold|`{port: x, value: y, threshold: z}`|
|0x00020006|Warning<br> * By user preference|埠 x 輸入頻寬利用率下降到 y% (低於閥值 z%)|The input traffic utilization of port x is y%, below threshold z%|*x*: Port number,  *y%*: link usage, *z%*: link usage threshold|`{port: x, value: y, threshold: z}`|
|0x80020006|Normal (Information)|埠 x 輸入頻寬利用率恢復到 y% (超過閥值 z%)|The input traffic utilization of port x is y%, no longer below threshold z%|*x*: Port number,  *y%*: link usage, *z%*: link usage threshold|`{port: x, value: y, threshold: z}`|
|0x00020007|Warning<br> * By user preference|埠 x 輸出頻寬利用率到達 y% (超過閥值 z%)|The output traffic utilization of port x is y%, over threshold z%|*x*: Port number,  *y%*: link usage, *z%*: link usage threshold|`{port: x, value: y, threshold: z}`|
|0x80020007|Normal (Information)|埠 x 輸出頻寬利用率恢復到 y% (低於閥值 z%)|The output traffic utilization of port x is y%, no longer over threshold z%|*x*: Port number,  *y%*: link usage, *z%*: link usage threshold|`{port: x, value: y, threshold: z}`|
|0x00020008|Warning<br> * By user preference|埠 x 輸出頻寬利用率下降到 y% (低於閥值 z%)|The output traffic utilization of port x is y%, below threshold z%|*x*: Port number,  *y%*: link usage, *z%*: link usage threshold||
|0x80020008|Normal (Information)|埠 x 輸出頻寬利用率恢復到 y% (超過閥值 z%)|The output traffic utilization of port x is y%, no longer below threshold z%|*x*: Port number,  *y%*: link usage, *z%*: link usage threshold||
|0x00020009|Warning<br> * By user preference|埠 x 輸入資料封包錯誤率到達 y% (超過閥值z%)|The input packet error rate of port x is y%, no longer over threshold z%|*x*: Port number,  *y%*: packet error rate, *z%*: packet error rate threshold||
|0x80020009|Normal (Information)|埠 x 輸入資料封包錯誤率恢復到 y% (低於閥值z%)|The input packet error rate of port x is y%, below threshold z%|*x*: Port number,  *y%*: packet error rate, *z%*: packet error rate threshold||
|0x0002000A|Warning<br> * By user preference|埠 x 輸出資料封包錯誤率到達 y% (超過閥值 z%)|The output packet error rate of port x is y%, no longer over threshold z%|*x*: Port number,  y%: packet error rate, *z%*: packet error rate threshold||
|0x8002000A|Normal (Information)|埠 x 輸出資料封包錯誤率恢復到 y% (低於閥值 z%)|The output packet error rate of port x is y%, below threshold z%|*x*: Port number,  *y%*: packet error rate, *z%*: packet error rate threshold||
|0x00020010 / 0x80020010||SFP 埠 x Tx {{currentdB}}dBm 低於閾值 {{thresholddB}}dBm / SFPTx {{recoverydB}}dBm 已恢復正常|SfpTxBelow / SfpTxBelowRecovery|x: Port number||
|0x00020011 / 0x80020011||SFP 埠 x Rx {{currentdB}}dBm 低於閾值 {{thresholddB}}dBm / SFP埠 x Rx {{recoverydB}}dBm 已恢復正常|SfpRxBelow / SfpRxBelowRecovery|x: Port number|
|0x00020012 / 0x80020012||SFP 埠 x 溫度 {{currentTemp}}ºc高於閾值 {{thresholdTemp}}ºc /SFP 埠 x 溫度 {{recoveryTemp}}ºc已恢復正常|SfpTempOver / SfpTempOverRecovery|x: Port number|
|0x00020013 / 0x80020013||SFP 埠 x 電壓 {{currentVoltage}}v高於閾值{{thresholdVoltage}}v/SFP 埠 x 電壓 {{recoveryVoltage}}v 已恢復正常|SfpVoltageOver / SfpVoltageOverRecovery|x: Port number|
|0x00020014 / 0x80020014||SFP 埠 x 電壓 {{currentVoltage}}v低於閾值 {{thresholdVoltage}}v /SFP 埠 x 電壓{{recoveryVoltage}}v 已恢復正常|SfpVoltageBelow / SfpVoltageBelowRecovery|x: Port number|
|0x00030001||VPN連線中斷|(EDR) VPN connection is disconnected||
|0x00030002||VPN連線恢復|(EDR) VPN connection is connected||
|0x00030003||x 光纖警告 ( {{warningType}} ) |Fiber warning (PRP/HSR)|||
|0x00F10009||設備由 PoE 供電|PoEOn|
|0x00F1000A||設備未通過 PoE 供電|PoEOff|
|0x10010008||設備冷啟動|(Trap)Device cold start|||
|0x10010009||設備熱啟動|(Trap)Device warm start|||
|0x1001000A||使用者登入設備認證失敗|(Trap)User login failed|||
|0x1001000B||DI x 開啟|(Trap) DI x is up|*x*:  index of DI|`{trap_detail: x}`|
|0x1001000C||DI x 關閉|(Trap) DI x is down|*x*:  index of DI|`{trap_detail: x}`|
|0x1001000E||電源 x 以直流電(DC)供電中|(Trap) The input of Power x is DC|x: The nth power input (1 or 2)||
|0x10710001|Normal (Information)|Turbo Ring 拓撲改變|(Trap)Turbo Ring Topology is changed|||
|0x10710002|Normal (Information)|Turbo Ring Coupling 埠改變|(Trap)Turbo Ring Coupling port is changed|||
|0x10710003|Normal (Information)|Turbo Ring Master 改變|(Trap)Turbo Ring Master is changed|||
|0x10710004|Normal (Information)|設備配置改變|(Trap)Device topology is changed|||
|0x10710005|Normal (Information)|設備韌體更新|(Trap)Device firmware is upgraded|||
|0x10710006|Normal (Information)|收到設備 trap|(Trap)A trap is received|*x*: trap oid|`{trapoid: x}`|
|0x10710007|Normal (Information)|收到設備 Inform|(Trap)An inform is received|*x*: trap oid|`{trapoid: x}`|
|0x10710008|Normal (Information)|埠 x 頻寬利用率到達 y%|(Trap) The port traffic utilization of port x is over y%|*x*: Port Number,  *y%*: traffic utilization|`{port: x,value: y}`|
|0x10710009||設備LLDP Table changed|(Trap) The LLDP table of a device is changed|||
|0x1071000A||PoE device電流超過閥值告警|(Trap) PoE device current over threshold|||
|0x1071000B||PoE device檢測失敗告警|(Trap) PoE device check failed warning|||
|0x1071000C||PoE device上電告警|(Trap) A PoE device is on|||
|0x1071000D||PoE device斷電告警|(Trap) A PoE device is off|||
|0x1071000E||PSE(switch) over threshold告警|(Trap) The PSE is over threshold|||
|0x1071000F||PSE(switch) MOSFet失效告警|(Trap) MOSFet on a PSE is failed|||
|0x10710010||PSE(switch)溫度超標告警|(Trap) The temperature of the PSE is over threshold|||
|0x10710011||PSE(switch) 輸入電壓過低告警|(Trap) The input voltage of the PSE is too low|||
|0x10710012||ABC02告警|(Trap) ABC02 warning|||
|0x10710013||TurboRing Master與設定不一致告警|(Trap) Turboring master mismatch|||
|0x10710014||使用者登入設備成功告警|(Trap) A user is logged in the device|||
|0x10710015||使用這登入設備失敗告警|(Trap) A user failed to logged in the device|||
|0x10710016||設備超出登入人數上限告警|(Trap) Too many users have logged in the device|||
|0x10710017||設備使用者設定改變告警|(Trap) The user settings of the device is changed|||
|0x10710018||設備匯入import告警|(Trap) A new config is imported to the device|||
|0x10710019||PHR功能失效告警|(Trap) The PHR function of the device is failed|||
|0x1071001A||設備Firewall規則被違反告警|(Trap)(EDR) The firewall rules are violated|||
|0x1071001B||mac sticky違反告警|(Trap) The MAC sticky rules are violated|||
|0x1071001C||設備遠端登入成功(TACACS, RADIUS)告警|(Trap) A remote login (through TACACS+ or RADIUS) is successful|||
|0x1071001D||設備遠端登入失敗(TACACS, RADIUS)告警|(Trap) A remote login (through TACACS+ or RADIUS) is failed|||
|0x1071001F|||MXnosNotifyLoginLockoutTrapEvent|||
|0x10710021|||MXnosNotifyCertChangedTrapEvent|||
|0x10710022|||MXnosNotifyPasswordChangedTrapEvent|||
|0x10710023|||MXnosNotifyConfigChangedTrapEvent|||
|0x10710024|||MXnosNotifyConfigImportedTrapEvent|||
|0x10710028|||MXnosDeviceioNotifyDiOpenTrapEvent|||
|0x10710029|||MXnosDeviceioNotifyDiShortTrapEvent|||
|0x1071002A|||RlpsNotifyRateLimitOnTrapEvent|||
|0x1071002B|||RlpsNotifyRateLimitOffTrapEvent|||
|0x1071002C|||PsmsNotifyViolationPortTrapEvent|||
|0x1071002D|||SweNotifyTopologyChangedTrapEvent|||
|0x1071002F|||Trv2NotifyMasterChangedTrapEvent|||
|0x10710030|||Trv2NotifyMasterMismatchTrapEvent|||
|0x10710031|||Ieee8021SpanningTreeTopologyChangeTrapEvent|||
|0x10710032|||Ieee8021SpanningTreeNewRootTrapEvent|||
|0x10710033|||RstpNotifyProtocolMigrationTrapEvent|||
|0x10710034|||RstpNotifyInvalidBpduTrapEvent|||
|0x10710035|||RstpNotifyNewPortRoleTrapEvent|||
|0x10710036|||LhcStatusFailTrapEvent|||
|0x10710037|||DualhomingNotifyPathChangedTrapEvent|||
|0x10710038|||Dot1xNotifyAuthFailTrapEvent|||
|0x1071003A|||PoeNotifyPdPowerOnTrapEvent|||
|0x1071003B|||PoeNotifyPdPowerOffTrapEvent|||
|0x1071003C|||PoeNotifyLowInputVoltageTrapEvent|||
|0x1071003D|||PoeNotifyPdOverCurrentTrapEvent|||
|0x1071003E|||PoeNotifyPdNoResponseTrapEvent|||
|0x1071003F|||PoeNotifyOverBudgetLimitTrapEvent|||
|0x10710040|||PoeNotifyPdDetectionFailureTrapEvent|||
|0x10710041|||RisingAlarmTrapEvent|||
|0x10710042|||FallingAlarmTrapEvent|||
|0x10710043|||VarTrackingStatusChangedTrapEvent|||
|0x10710044|||VarTrackingVRRPChangedTrapEvent|||
|0x10710045|||VarTrackingStaticRouteChangedTrapEvent|||
|0x10710046|||VarTrackingPortEnableChangedTrapEvent|||
|0x10710047|||VarEPSChangeToOnTrapEvent|||
|0x10710048|||VarEPSChangeToOffTrapEvent|||
|0x10710049|||VarDyingGaspTrapEvent|||
|0x10710050|||VarPtpGrandmasterChangedTrapEvent|||
|0x10710051|||VarPtpSyncStatusChangedTrapEvent|||
|0x10720001|||Port traffic overload 告警|(Trap) Port traffic overload|||
|0x10720002||偵測到封包looping告警|(Trap) A loop is detected|||
|0x10720003|||Fiber warning|(Trap) Fiber warning|||
|0x10A10001||Nport設備 Syslog超過threshold告警|(Trap)(Nport) Too many syslog records|||
|0x10F10006||用戶端 加入: x|(Trap) A wireless client (MAC address x) is joined|*x*: MAC address of the joined client||
|0x10F10007||用戶端 離開: x|(Trap) A wireless client (MAC address x) is left|*x*: MAC address of the client left||
|0x80010001|Normal (Information)|設備恢復存取|Device ICMP reachable|||
|0x80010002|Normal (Information)|設備封包遺失率恢復到 y%, 低於閥值z% (基於ICMP)|The packet loss rate is y%, no longer over threshold z%|||
|0x80010003|Normal (Information)||The packet loss rate is y%, no longer over threshold z%|||
|0x80010006|Normal (Information)|電源 x 通電|Power x is up|*x*: The nth power input (1 or 2)|`{trap_detail: x}`|
|0x80010007|Normal (Information)|設備恢復以SNMP存取|Device SNMP reachable|||
|0x8001000D|Normal (Information)|設備可用性高於閥值 /目前設備可用性為 y%, 高於或等於閥值 z%|The availability of device is y%, no longer below threshold z%|||
|0x8001000E||電源 x 以交流電(AC)供電中|(Trap) The input of Power x is AC|x: The nth power input (1 or 2)|||
|0x80010100~0x80020000||自訂事件回復|A custom event is recovered. x. Threshold=y, value=z. w|*x*: the name of the device property of this custom event<br>*y*: the threshold<br>*z*: the value<br>*w*: detail description of this event|`{eventName:x, threshold: y, value: z, eventDescription: w }`|
|0x268435489|Normal (Information)||A firmware update is available. Please check the Firmware Management page for details|||
|0x10710050|||PTP Grandmaster Changed|||
|0x10710051|||PTP Sync Status Changed|||
|0x10718852|Warning||The fan module is behaving abnormally|||
|0x10710053|Warning||The power module fan is behaving abnormally|||
|0x10710054|Warning||The system temperature has exceeded the set alarm threshold|||
|0x10710055|Warning||One of the component thermal sensors registered overheating|||
|0x10710056|Warning||The system has entered the Power Cut-off state|||
|0x10710057|Warning||The power module x has entered the over-temperature protection state|x: The nth power input (1 or 2)||
|0x10710058|Normal (Information)||VRRP Master Changed|||
|0x10710059|Normal (Information)||Firmware upgraded|||
|0x10710069|Normal (Information)||MSTP topology is changed|||
|0x1071006A|Normal (Information)||A new root bridge is selected in the topology|||
|0x1071006B|Normal (Information)||A new role is selected for the port|||
|0x1071006C|Normal (Information)||The designated router changed|||
|0x1071006D|Normal (Information)||The interface as designated router and adjacency changed|||
|0x1071006E|Normal (Information)||The interface become designated router|||
|0x1071006F|Normal (Information)||PHR AB Port Timediff.|||
|0x10710070|Normal (Information)||PHR AB Port Wrong LAN.|||
|0x10710071|Normal (Information)||MRP multiple managers event occurred.|||
|0x10710072|Normal (Information)||MRP ring open event occurred.|||
|0x1071004A|Normal (Information)||Port x Non-PD or PD short circuited.|x: Port number||
|0x1071004B|Normal (Information)||Multiple Network Coupling changed.|||
|0x10B10001|Information||PWR On|pwrOn||
|0x10B10002|Information||PWR Off|pwrOff||
|0x10B10003|Information||PWR Fail|pwrFail||
|0x10B10004|Information||CPU Usage Over Threshold|cpuUsageOverThreshold||
|0x10B10005|Information||CPU Usage Under Threshold|cpuUsageUnderThreshold||
|0x10B10006|Information||RAM Usage Over Threshold|ramUsageOverThreshold||
|0x10B10007|Information||RAM Usage Under Threshold|ramUsageUnderThreshold||
|0x10B10008|Information||Disk Usage Over Threshold|diskUsageOverThreshold||
|0x10B10009|Information||Disk Usage Under Threshold|diskUsageUnderThreshold||
|0x10B1000A|Information||Module Plug|modulePlug||
|0x10B1000B|Information||Module Unplug|moduleRemove||
|0x10B1000C|Information||SD Card Insert|sdCardInsert||
|0x10B1000D|Information||SD Card Remove|sdCardRemove||
|0x10B1000E|Information||Mode Switch Set to Run|modeSwitchSetToRun||
|0x10B1000F|Information||Mode Switch Set to Stop|modeSwitchSetToStop||
|0x10B10010|Information||Mode Switch Set to Remote|modeSwitchSetToRemove||
|0x10B10011|Information||Reset Button Push|resetButtonPush||
|0x10B10012|Information||Configuration Change Success|configChangeSuccess||
|0x10B10013|Information||Configuration Change Failed|configChangeFailed||
|0x10B10014|Information||Time Change|timeChange||
|0x10B10015|Information||Time Sync by NTP Server Success|timeSyncByNTPServerSuccess||
|0x10B10016|Information||Time Sync by NTP Server Failed|timeSyncByNTPServerFailed||
|0x10B10017|Information||Configuration Integrity Error|configIntegrityError||
|0x10B10018|Information||Configuration Import Success|configImportSucess||
|0x10B10019|Information||Configuration Import Failed|configImportFailed||
|0x10B1001A|Information||Configuration Export Success|configExportSuccess||
|0x10B1001B|Information||Configuration Export Failed|configExportFailed||
|0x10B1001C|Information||Configuration Reset Origin Device Success|configResetOriginDeviceSucess||
|0x10B1001D|Information||Configuration Reset to Factory Default Failed|configResetFactoryDefaultFailed||
|0x10B1001E|Information||Time Sync by DNP3 Server Success|timeSyncByDNP3ServerSuccess||
|0x10B1001F|Information||Time Sync by DNP3 Server Failed|timeSyncByDNP3ServerFailed||
|0x10B10020|Information||FWR Upgrade Success|fwrUpgradeSuccess||
|0x10B10021|Information||FWR Upgrade Failed|fwrUpgradeFailed||
|0x10B10022|Information||Application Run Success|applicationRunSuccess||
|0x10B10023|Information||Application Run Failed|applicationRunFailed||
|0x10B10024|Information||Application Stop Success|applicationStopSuccess||
|0x10B10025|Information||Application Stop Failed|applicationStopFailed||
|0x10B10026|Information||Application Download Success|applicationDownloadSuccess||
|0x10B10027|Information||Application Download Failed|applicationDownloadFailed||
|0x10B10028|Information||Modules Topology Mismatch|moduleTopologyMismatach||
|0x10B10029|Information||I/O Module Field Circuit Supply Loss|ioModuleFieldCircuitSupplyLoss||
|0x10B1002A|Information||I/O Module Loop Power Supply 1 Loss|ioModuleLoopPower1Loss||
|0x10B1002B|Information||I/O Module Loop Power Supply 2 Loss|ioModuleLoopPower2Loss||
|0x10B1002C|Information||I/O Module DO Common Power Loss|ioModuleDOCommonPowerLoss||
|0x10B1002D|Information||I/O Module System Power Fail|ioModuleSystemPowerFailed||
|0x10B1002E|Information||I/O Module Field Power Fail|ioModuleFieldPowerFailed||
|0x10B1002F|Information||I/O Channel Wirebreak|ioChannelWirebreak||
|0x10B10030|Information||I/O Channel Underflow|ioChannelUnderflow||
|0x10B10031|Information||I/O Channel Under Range|ioChannelUnderRange||
|0x10B10032|Information||I/O Channel Over Range|ioChannelOverRange||
|0x10B10033|Information||I/O Channel Overflow|ioChannelOverflow||
|0x10B10034|Information||I/O Channel Short Circuit|ioChannelShortCircuit||
|0x10B10035|Information||DNP3 Outstation Connect|dnp3OutStationConnect||
|0x10B10036|Information||DNP3 Outstation Disconnect|dnp3OutStationDisconnect||
|0x10B10037|Information||DNP3 Outstation Run|dnp3OutStationRun||
|0x10B10038|Information||DNP3 Outstation Stop|dnp3OutStationStop||
|0x10B10039|Information||DNP3 Outstation Config Change Success|dnp3OutStationConfigChangeSuccess||
|0x10B1003A|Information||DNP3 Outstation Config Change Failed|dnp3OutStationConfigChangeFailed||
|0x10B1003B|Information||DNP3 Outstation Disable|dnp3OutStationDisable||
|0x10B1003C|Information||DNP3 Outstation Error|dnp3OutStationError||
|0x10B1003D|Information||OPCUA Connect|opcuaConnect||
|0x10B1003E|Information||OPCUA Disconnect|opcuaDisconnect||
|0x10B1003F|Information||OPCUA Run|opcuaRun||
|0x10B10040|Information||OPCUA Stop|opcuaStop||
|0x10B10041|Information||OPCUA Config Change Success|opcuaConfigChangeSuccess||
|0x10B10042|Information||OPCUA Config Change Failed|opcuaConfigChangeFailed||
|0x10B10043|Information||OPCUA Disable|opcuaDisable||
|0x10B10044|Information||OPCUA Error|opcuaError||
|0x10B10045|Information||Modbus Master Command Recover|modbusMasterCommandRecovered||
|0x10B10046|Information||Modbus Master Command Fail|modbusMasterCommandFailed||
|0x10B10047|Information||Modbus Master Device Connected|modbusMasterDeviceConnected||
|0x10B10048|Information||Modbus Master Device Disconnected|modbusMasterDeviceDisconnected||
|0x10B10049|Information||Modbus Master Config Update Success|modbusMasterConfigUpdateSuccess||
|0x10B1004A|Information||Modbus Master Config Update Failed|modbusMasterConfigUpdateFailed||
|0x10B1004B|Information||Modbus Slave Invalid Function|modbusSlaveInvalidFunction||
|0x10B1004C|Information||Modbus Slave Device Connected|modbusSlaveDeviceConnected||
|0x10B1004D|Information||Modbus Slave Device Disconnected|modbusSlaveDeviceDisconnected||
|0x10B1004E|Information||Modbus Slave Config Update Success|modbusSlaveConfigUpdateSuccess||
|0x10B1004F|Information||Modbus Slave Config Update Failed|modbusSlaveConfigUpdateFailed||
|0x10B10050|Information||Modbus Run|modbusRun||
|0x10B10051|Information||Modbus Stop|modbusStop||
|0x10B10052|Information||Modbus Disable|modbusDisable||
|0x10B10053|Information||Modbus Error|modbusError||
|0x10B10054|Information||Login Success|loginSuccess||
|0x10B10055|Information||Login Failed|loginFailed||
|0x10B10056|Information||Logout Success|logoutSuccess||
|0x10B10057|Information||Logout Failed|logoutFailed||
|0x10B10058|Information||Remote RADIUS Server Connection Success|remoteRADIUSServerConnectionSuccess||
|0x10B10059|Information||Remote RADIUS Server Connection Failed|remoteRADIUSServerConnectionFailed||
|0x10B1005A|Information||Remote TACACS+ Server Connection Success|remoteTACACSServerConnectionSuccess||
|0x10B1005B|Information||Remote TACACS+ Server Connection Failed|remoteTACACSServerConnectionFailed||
|0x10B1005C|Information||Remote LDAP Server Connection Success|remoteLDAPServerConnectionSuccess||
|0x10B1005D|Information||Remote LDAP Server Connection Failed|remoteLDAPServerConnectionFailed||
|0x10B1005E|Information||Lock Success|lockSuccess||
|0x10B1005F|Information||Lock Failed|lockFailed||
|0x10B10060|Information||Account Create Success|accountCreateSuccess||
|0x10B10061|Information||Account Create Failed|accountCreateFailed||
|0x10B10062|Information||Account Delete Success|accountDeleteSuccess||
|0x10B10063|Information||Account Delete Failed|accountDeleteFailed||
|0x10B10064|Information||Account Activate Success|accountActivateSuccess||
|0x10B10065|Information||Account Activate Failed|accountActivateFailed||
|0x10B10066|Information||Account Deactivate Success|accountDeactivateSuccess||
|0x10B10067|Information||Account Deactivate Failed|accountDeactivateFailed||
|0x10B10068|Information||Account Setting Modify Success|accountSettingModificationSuccess||
|0x10B10069|Information||Account Setting Modify Failed|accountSettingModificationFailed||
|0x10B1006A|Information||Account Password Expiration Warning|accountPasswordExpirationWarning||
|0x10B1006B|Information||Role Create Success|roleCreateSuccess||
|0x10B1006C|Information||Role Create Failed|roleCreateFailed||
|0x10B1006D|Information||Role Delete Success|roleDeleteSuccess||
|0x10B1006E|Information||Role Delete Failed|roleDeleteFailed||
|0x10B1006F|Information||Privilege of a Role Modify Success|privilegeOfARoleModificationSuccess||
|0x10B10070|Information||Privilege of a Role Modify Failed|privilegeOfARoleModificationFailed||
|0x10B10071|Information||Access Violation(Permission Denied)|accessViolation||
|0x10B10072|Information||Event Log Clear Success|logClearSuccess||
|0x10B10073|Information||Event Log Clear Failed|logClearFailed||
|0x10B10074|Information||Event Log Over the Threshold of Storage Capacity|logOverThreshold||
|0x10B10075|Information||Certificate & Private Key Generate Success|certificatePrivateKeyGenerateSuccess||
|0x10B10076|Information||Certificate & Private Key Generate Failed|certificatePrivateKeyGenerateFailed||
|0x10B10077|Information||Certificate & Private Key Import Success|certificatePrivateKeyImportSuccess||
|0x10B10078|Information||Certificate & Private Key Import Failed|certificatePrivateKeyImportFailed||
|0x10B10079|Information||Certificate Expire in N Days|certificateExpireinNDays||
|0x10B1007A|Information||Certificate Revoked|certificateRevoked||
|0x10B1007B|Information||CA Certificate Import Success|caCertificateImportSuccess||
|0x10B1007C|Information||CA Certificate Import Failed|caCertificateImportFailed||
|0x10B1007D|Information||HTTPS Certificate & Key Set Success|httpsCertificateKeySetSuccess||
|0x10B1007E|Information||HTTPS Certificate & Key Set Failed|httpsCertificateKeySetFailed||
|0x10B1007F|Information||HTTPS Certificate & Key Clear Success|httpsCertificateKeyClearSuccess||
|0x10B10080|Information||HTTPS Certificate & Key Clear Failed|httpsCertificateKeyClearFailed||
|0x10B10081|Information||IDE comm Certificate & Key Set Success|ideCertificateKeySetSuccess||
|0x10B10082|Information||IDE comm Certificate & Key Set Failed|ideCertificateKeySetFailed||
|0x10B10083|Information||IDE comm Certificate & Key Clear Success|ideCertificateKeyClearSuccess||
|0x10B10084|Information||IDE comm Certificate & Key Clear Failed|ideCertificateKeyClearFailed||
|0x10B10085|Information||Syslog Over TLS Certificate & Key Set Success|syslogTLSCertificateKeySetSuccess||
|0x10B10086|Information||Syslog Over TLS Certificate & Key Set Failed|syslogTLSCertificateKeySetFailed||
|0x10B10087|Information||Syslog Over TLS Certificate & Key Clear Success|syslogTLSCertificateKeyClearSuccess||
|0x10B10088|Information||Syslog Over TLS Certificate & Key Clear Failed|syslogTLSCertificateKeyClearFailed||
|0x10B10089|Information||RadSec Certificate & Key Set Success|radSecCertificateKeySetSuccess||
|0x10B1008A|Information||RadSec Certificate & Key Set Failed|radSecCertificateKeySetFailed||
|0x10B1008B|Information||RadSec Certificate & Key Clear Success|radSecCertificateKeyClearSuccess||
|0x10B1008C|Information||RadSec Certificate & Key Clear Failed|radSecCertificateKeyClearFailed||
|0x10B1008D|Information||LDAPS Certificate & Key Set Success|ldapsCertificateKeySetSuccess||
|0x10B1008E|Information||LDAPS Certificate & Key Set Failed|ldapsCertificateKeySetFailed||
|0x10B1008F|Information||LDAPS Certificate & Key Clear Success|ldapsCertificateKeyClearSuccess||
|0x10B10090|Information||LDAPS Certificate & Key Clear Failed|ldapsCertificateKeyClearFailed||
|0x10B10091|Information||OpenVPN Client Certificate & Key Set Success|openvpnCertificateKeySetSuccess||
|0x10B10092|Information||OpenVPN Client Certificate & Key Set Failed|openvpnCertificateKeySetFailed||
|0x10B10093|Information||OpenVPN Client Certificate & Key Clear Success|openvpnCertificateKeyClearSuccess||
|0x10B10094|Information||OpenVPN Client Certificate & Key Clear Failed|openvpnCertificateKeyClearFailed||
|0x10B10095|Information||MQTT Certificate & Key Set Success|mqttCertificateKeySetSuccess||
|0x10B10096|Information||MQTT Certificate & Key Set Failed|mqttCertificateKeySetFailed||
|0x10B10097|Information||MQTT Certificate & Key Clear Success|mqttCertificateKeyClearSuccess||
|0x10B10098|Information||MQTT Certificate & Key Clear Failed|mqttCertificateKeyClearFailed||
|0x10B10099|Information||OPCUA Certificate & Key Set Success|opcuaCertificateKeySetSuccess||
|0x10B1009A|Information||OPCUA Certificate & Key Set Failed|opcuaCertificateKeySetFailed||
|0x10B1009B|Information||OPCUA Certificate & Key Clear Success|opcuaCertificateKeyClearSuccess||
|0x10B1009C|Information||OPCUA Certificate & Key Clear Failed|opcuaCertificateKeyClearFailed||
|0x10B1009D|Information||EST CA Certificate Enroll Success|estCaCertificateKeyEnrollSuccess||
|0x10B1009E|Information||EST CA Certificate Enroll Failed|estCaCertificateKeyEnrollFailed||
|0x10B1009F|Information||EST Certificate & Key Enroll Success|estCertificateKeyEnrollSuccess||
|0x10B100A0|Information||EST Certificate & Key Enroll Failed|estCertificateKeyEnrollFailed||
|0x10B100A1|Information||SSH Key re-Generate Success|sshKeyRegenerateSuccess||
|0x10B100A2|Information||SSH Key re-Generate Failed|sshKeyRegenerateFailed||
|0x10B100A3|Information||SNMP v3 Key Modify Success|snmpv3KeyModificationSuccess||
|0x10B100A4|Information||SNMP v3 Key Modify Failed|snmpv3KeyModificationFailed||
|0x10B100A5|Information||TACACS+ Key Modify Success|tacacsKeyModificationSuccess||
|0x10B100A6|Information||TACACS+ Key Modify Failed|tacacsKeyModificationFailed||
|0x10B100A7|Information||RADIUS Key Modify Success|radiusKeyModificationSuccess||
|0x10B100A8|Information||RADIUS Key Modify Failed|radiusKeyModificationFailed||
|0x10B100A9|Information||LDAP Key Modify Success|ldapKeyModificationSuccess||
|0x10B100AA|Information||LDAPKey Modify Failed|ldapKeyModificationFailed||
|0x10B100AB|Information||OpenVPN Enable Success|openvpnEnableSuccess||
|0x10B100AC|Information||OpenVPN Enable Failed|openvpnEnableFailed||
|0x10B100AD|Information||OpenVPN Disable Success|openvpnDisableSuccess||
|0x10B100AE|Information||OpenVPN Disable Failed|openvpnDisableFailed||
|0x10B100AF|Information||OpenVPN Activate Success|openvpnActivateSuccess||
|0x10B100B0|Information||OpenVPN Activate Failed|openvpnActivateFailed||
|0x10B100B1|Information||OpenVPN Deactivate Success|openvpnDeactivateSuccess||
|0x10B100B2|Information||OpenVPN Deactivate Failed|openvpnDeactivateFailed||
|0x10B100B3|Information||OpenVPN Configuration Import Success|openvpnConfigImportSuccess||
|0x10B100B4|Information||OpenVPN Configuration Import Failed|openvpnConfigImportFailed||
|0x10B100B5|Information||OpenVPN Connect Success|openvpnConnectSuccess||
|0x10B100B6|Information||OpenVPN Connect Failed|openvpnConnectFailed||
|0x10B100B7|Information||OpenVPN Disconnect Success|openvpnDisconnectSuccess||
|0x10B100B8|Information||OpenVPN Disconnect Failed|openvpnDisconnectFailed||
|||||||