CREATE TYPE node_type AS enum (
    'group',
    'device',
    'virtual'
);

CREATE TYPE edge_type AS enum (
    'link',
    'virtual'
);

CREATE TABLE IF NOT EXISTS edges (
    id serial PRIMARY KEY,
    TYPE edge_type NOT NULL
);

CREATE TABLE IF NOT EXISTS nodes (
    id serial PRIMARY KEY,
    TYPE node_type NOT NULL,
    x int NOT NULL DEFAULT 0,
    y int NOT NULL DEFAULT 0
);

CREATE TABLE IF NOT EXISTS access_settings (
    id serial PRIMARY KEY,
    snmp_version smallint DEFAULT 1,
    snmp_security_name varchar(63) DEFAULT 'admin',
    snmp_password varchar(63),
    snmp_security_level smallint DEFAULT 1,
    snmp_encrypt_key varchar(63),
    snmp_auth_proto smallint DEFAULT 1,
    snmp_priv_proto smallint DEFAULT 1,
    snmp_get_community varchar(63) DEFAULT 'public',
    snmp_set_community varchar(63) DEFAULT 'private',
    snmp_port int DEFAULT 161
);

CREATE TABLE IF NOT EXISTS groups (
    id int UNIQUE REFERENCES nodes (id) NOT NULL DEFAULT nextval(pg_get_serial_sequence('nodes', 'id')),
    name varchar(63),
    description text,
    x int NOT NULL DEFAULT 0,
    y int NOT NULL DEFAULT 0,
    parent_id int REFERENCES groups (id) NOT NULL,
    deleted boolean NOT NULL DEFAULT FALSE
);

CREATE TABLE IF NOT EXISTS devices (
    id int UNIQUE REFERENCES nodes (id) NOT NULL DEFAULT nextval(pg_get_serial_sequence('nodes', 'id')),
    ip inet NOT NULL,
    group_id int REFERENCES groups (id) NOT NULL,
    mac text[][],
    model varchar(63),
    alias varchar(63),
    icon_url varchar(63),
    x int,
    y int,
    access_settings_id int REFERENCES access_settings (id),
    redundancy int NOT NULL,
    deleted boolean NOT NULL DEFAULT FALSE
);

CREATE TABLE IF NOT EXISTS events (
    id bigserial PRIMARY KEY,
    event_type int NOT NULL,
    detector smallint NOT NULL,
    device_id int REFERENCES devices (id),
    /* is this nullable? */
    event_severity smallint NOT NULL,
    event_time timestamptz NOT NULL DEFAULT NOW(),
    source_ip inet,
    event_info jsonb
);

CREATE TABLE IF NOT EXISTS links (
    id int UNIQUE REFERENCES edges (id) NOT NULL DEFAULT nextval(pg_get_serial_sequence('edges', 'id')),
    from_device int NOT NULL REFERENCES devices (id),
    to_device int NOT NULL REFERENCES devices (id),
    from_port smallint NOT NULL,
    to_port smallint NOT NULL,
    is_wireless bool NOT NULL,
    deleted boolean NOT NULL DEFAULT FALSE
);

CREATE TABLE IF NOT EXISTS settings (
    id smallserial PRIMARY KEY,
    system_mail_username varchar(63),
    system_mail_password varchar(63),
    system_mail_server varchar(63),
    system_mail_port int,
    system_mail_encrypt int,
    system_mail_sender varchar(63),
    access_settings_id int NOT NULL REFERENCES access_settings (id)
);

CREATE TABLE IF NOT EXISTS site_data (
    id varchar(63) PRIMARY KEY,
    name varchar(63)
);

CREATE TABLE IF NOT EXISTS notifications (
    id smallserial PRIMARY KEY,
    name varchar(63),
    type int,
    device_ip inet,
    actions integer[],
    component varchar(63),
    event_digest varchar(20),
    email_content varchar(2000)
);

CREATE TABLE IF NOT EXISTS actions (
    id smallserial PRIMARY KEY,
    action_name varchar(63),
    action_type int,
    action_content varchar(63)
);

CREATE TABLE IF NOT EXISTS scan_ranges (
    name varchar(63) PRIMARY KEY,
    active boolean,
    start_ip inet NOT NULL,
    end_ip inet NOT NULL,
    group_id int REFERENCES groups (id)
);

CREATE TABLE IF NOT EXISTS custom_event_definitions (
    id smallserial PRIMARY KEY,
    field_name varchar(20) NOT NULL,
    field_index int,
    threshold_type smallint NOT NULL,
    threshold varchar(64) NOT NULL,
    active bool NOT NULL,
    severity smallint NOT NULL,
    description varchar(64) NOT NULL,
    recovery_description varchar(64) NOT NULL,
    devices integer[]
);

CREATE TABLE IF NOT EXISTS traffic_records (
    id serial PRIMARY KEY,
    device_id int REFERENCES devices (id),
    port_index int NOT NULL,
    input_bandwidth_utilization int,
    input_packet_error_rate int,
    output_bandwidth_utilization int,
    output_packet_error_rate int,
    record_time timestamptz NOT NULL DEFAULT NOW()
);

CREATE TABLE IF NOT EXISTS wireless_roles (
    device_id int PRIMARY KEY REFERENCES devices (id) NOT NULL,
    role smallint NOT NULL,
    wireless_interface int[]
);

CREATE TABLE IF NOT EXISTS wireless_histories (
    id bigserial PRIMARY KEY,
    linked_ap_id int REFERENCES devices (id),
    client_id int REFERENCES devices (id) NOT NULL,
    rssi smallint NOT NULL,
    -- noise smallint,
    -- snr smallint,

    created_time timestamptz NOT NULL DEFAULT NOW()
);

CREATE TABLE IF NOT EXISTS background_infos (
    id serial PRIMARY KEY,
    group_id int REFERENCES groups (id) NOT NULL,
    x int NOT NULL,
    y int NOT NULL,
    width int,
    height int,
    alpha int,
    saturation int,
    scale int,
    rotate int
);

