#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <tchar.h>
#include "registry.h"

/* Obtain a registry value
	Inputs:
		<path> the path of the registry
		<key> the key under the path
		<size> the length of the value buffer
	Outputs:
		<val> the value buffer
	Returns:
		0 for success; others for failure
*/
int
registry_machine_get(TCHAR *path, TCHAR *key, TCHAR *val, DWORD size)
{
	HKEY	hKey;
	DWORD	dwType;
	ULONG	ulRet;

	/* Note: dedicated HKEY */
	ulRet = RegOpenKeyEx( HKEY_LOCAL_MACHINE, path, 0, KEY_QUERY_VALUE, &hKey);
	if ( ulRet == ERROR_SUCCESS )
	{
		memset( val, 0, size );
		/* Note: type of register value not returned to caller */
		ulRet = RegQueryValueEx( hKey, key, NULL, &dwType, (LPBYTE)val, (LPDWORD)&size );
		/* buffer not enough returns ERROR_MORE_DATA */
		RegCloseKey( hKey );
		if ( ulRet == ERROR_SUCCESS )
			return size;
	}
	return -1;
}

/* Update/Insert a registry value
	Inputs:
		<path> the path of the registry
		<key> the key under the path
		<szName> the value
		<type> the type of the registry
	Returns:
		0 for success; others for failure
*/
int
registry_machine_update(TCHAR *path, TCHAR *key, TCHAR *val, DWORD type, int sz)
{
	HKEY	hKey;
	ULONG	ulRet;

	/* Note: dedicated HKEY */
	ulRet = RegOpenKeyEx( HKEY_LOCAL_MACHINE, path, 0, KEY_SET_VALUE, &hKey);
	if ( ulRet == ERROR_SUCCESS )
	{
		DWORD size;

		if (type == REG_DWORD)
			size = sizeof(DWORD);
		else if (type == REG_SZ)
			size = (DWORD) _tcslen(val)*sizeof(TCHAR)+1;
		else if (type == REG_MULTI_SZ)
			size = sz;
		else /* REG_EXPAND_SZ, REG_BINARY, or ... */
			size = MAX_PATH;
		ulRet = RegSetValueEx( hKey, key, 0, type, (LPBYTE) val, size);
		RegFlushKey( hKey );
		RegCloseKey( hKey );
		if ( ulRet == ERROR_SUCCESS )
			return 0;
	}
	return -1;
}


/* make a function call upon each registry value under a path
	Inputs:
		<path>	the path
		<cbfun>	the callback function
		<arg>	the argument passed to the callback function
	Returns:
		0 for success; others for failure
*/
int
registry_machine_foreach(TCHAR *path, registry_t cbfun, void *arg, BOOL clean)
{
	HKEY	hKey;
	LONG	ulRet;
	int		nEnumIndex = 0;
	DWORD	dwLenVal, dwLenDat, dwType;
	TCHAR	tval[256], tdat[256];
	REGSAM	samDesired;

	if (clean == TRUE)
#ifdef WINCE
		samDesired = KEY_SET_VALUE;
#else
		/* need KEY_SET_VALUE and KEY_QUERY_VALUE */
		samDesired = KEY_ALL_ACCESS;
#endif
	else
		samDesired = KEY_READ;
	/* Note: dedicated HKEY */
	if ( RegOpenKeyEx( HKEY_LOCAL_MACHINE, path, 0, samDesired, &hKey) != ERROR_SUCCESS )
		return -1;

	do {
		dwLenVal = sizeof(tval);
		memset( tval, 0, 256*sizeof(TCHAR) );
		dwLenDat = sizeof(tdat);
		memset( tdat, 0, 256*sizeof(TCHAR) );
		if (clean == TRUE)
		{
			ulRet = RegEnumValue(hKey, 0, tval, &dwLenVal, NULL, &dwType, (LPBYTE)tdat, &dwLenDat);
			if ( ulRet == ERROR_SUCCESS )
			{
				LONG	ret;
				ret = RegDeleteValue(hKey, tval); /* no check return */
#if 0
				if ( ret == ERROR_SUCCESS )
					printf("deleted\n");
				else
					printf("delete failed\n");
#endif
			}
#if 0
			else
			{
				printf("RegEnumValue() failed %d\n", GetLastError());
				//ErrorMsg(TEXT("RegEnumValue"));
				/* when no value at first or no more value */
				if (ulRet == ERROR_NO_MORE_ITEMS) printf("empty\n");
				else if (ulRet == ERROR_MORE_DATA) printf("buffer too smaller\n");
				/* when error also break loop */
			}
#endif
		}
		else
		{
			ulRet = RegEnumValue(hKey, nEnumIndex, tval, &dwLenVal, NULL, &dwType, (LPBYTE)tdat, &dwLenDat);
			if ( ulRet != ERROR_SUCCESS ) break;
			if (cbfun != NULL) {
				if (dwType == REG_SZ || dwType == REG_MULTI_SZ ||
					dwType == REG_EXPAND_SZ)
					/* REG_DWORD, REG_BINARY not handled */
					cbfun(tval, dwLenVal, tdat, dwLenDat, arg);
			}
			nEnumIndex++;
		}
	} while ( ulRet == ERROR_SUCCESS );

	RegCloseKey( hKey );

	return 0;
}

#ifndef WINCE
static int
check_path(TCHAR *path, TCHAR *achKey, TCHAR *AdapterName, TCHAR *subpath, TCHAR *Name, TCHAR *NetName)
{
	HKEY	hKey;
	TCHAR	newpath[MAX_PATH];
	DWORD	dwType;
	ULONG	ulRet;
	DWORD	size = MAX_PATH;

	_stprintf(newpath, L"%s\\%s\\%s\\%s", path, achKey, AdapterName, subpath);
#if 0
	_tprintf("%S,%S,%S,%S]\n", path, achKey, AdapterName, subpath);
	_tprintf("%S]\n", newpath);
#endif

	/* Note: dedicated HKEY */
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, newpath, 0, KEY_READ, &hKey) != ERROR_SUCCESS)
		return -1;

	memset( NetName, 0, size );
	/* Note: type of register value not returned to caller */
	ulRet = RegQueryValueEx( hKey, Name, NULL, &dwType, (LPBYTE)NetName, (LPDWORD)&size );
#if 0
	printf("size=%d\n", size); /* 2 more bytes for null wide char ? */
	_tprintf(L"%s]\n", NetName);
#endif
	/* buffer not enough returns ERROR_MORE_DATA */
	RegCloseKey( hKey );
	if ( ulRet == ERROR_SUCCESS )
		return size;

	return -2;
}

#define MAX_KEY_LENGTH 255
#define MAX_VALUE_NAME 16383

int
registry_machine_find_data(TCHAR *path, TCHAR *AdapterName, TCHAR *subpath, TCHAR *Name, TCHAR *NetName)
{
	HKEY	hKey;
    TCHAR    achKey[MAX_KEY_LENGTH];   // buffer for subkey name
    DWORD    cbName;                   // size of name string 
    TCHAR    achClass[MAX_PATH] = TEXT("");  // buffer for class name 
    DWORD    cchClassName = MAX_PATH;  // size of class string 
    DWORD    cSubKeys=0;               // number of subkeys 
    DWORD    cbMaxSubKey;              // longest subkey size 
    DWORD    cchMaxClass;              // longest class string 
    DWORD    cValues;              // number of values for key 
    DWORD    cchMaxValue;          // longest value name 
    DWORD    cbMaxValueData;       // longest value data 
    DWORD    cbSecurityDescriptor; // size of security descriptor 
    FILETIME ftLastWriteTime;      // last write time 
    DWORD i, retCode; 
	int		len;
 
	/* Note: dedicated HKEY */
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, path, 0, KEY_READ, &hKey) != ERROR_SUCCESS)
		return -1;

    // Get the class name and the value count. 
    retCode = RegQueryInfoKey(
        hKey,                    // key handle 
        achClass,                // buffer for class name 
        &cchClassName,           // size of class string 
        NULL,                    // reserved 
        &cSubKeys,               // number of subkeys 
        &cbMaxSubKey,            // longest subkey size 
        &cchMaxClass,            // longest class string 
        &cValues,                // number of values for this key 
        &cchMaxValue,            // longest value name 
        &cbMaxValueData,         // longest value data 
        &cbSecurityDescriptor,   // security descriptor 
        &ftLastWriteTime);       // last write time 
 
    // Enumerate the subkeys, until RegEnumKeyEx fails.
    
    if (!cSubKeys)
	{
		RegCloseKey( hKey );
		return -1;
	}
#if 0
	printf( "\nNumber of subkeys: %d\n", cSubKeys);
#endif
	for (i=0; i<cSubKeys; i++) 
	{ 
		cbName = MAX_KEY_LENGTH;
		retCode = RegEnumKeyEx(hKey, i,
					achKey,
					&cbName,
					NULL,
					NULL,
					NULL,
					&ftLastWriteTime);
		if (retCode == ERROR_SUCCESS)
		{
#if 0
			_tprintf(TEXT("(%d) %s\n"), i+1, achKey);
#endif
			len = check_path(path, achKey, AdapterName, subpath, Name, NetName);
			if (len > 0) break;
		}
	}
	if (i >= cSubKeys)
	{
		RegCloseKey( hKey );
		return -2;
	}

	RegCloseKey( hKey );

	return len;
}
#endif

int
registry_machine_find_name(TCHAR *path, TCHAR *data, TCHAR *name, int sz)
{
	HKEY	hKey;
	LONG	ulRet;
	int		nEnumIndex = 0;
	DWORD	dwLenVal, dwLenDat, dwType;
	TCHAR	tval[256], tdat[256];
	int		rc = 0; /* not found */

	/* Note: dedicated HKEY */
	if ( RegOpenKeyEx( HKEY_LOCAL_MACHINE, path, 0, KEY_READ, &hKey) != ERROR_SUCCESS )
		return -1;

	do {
		dwLenVal = sizeof(tval);
		memset( tval, 0, 256 );
		dwLenDat = sizeof(tdat);
		memset( tdat, 0, 256 );
		ulRet = RegEnumValue(hKey, nEnumIndex, tval, &dwLenVal, NULL, &dwType, (LPBYTE)tdat, &dwLenDat);
		if ( ulRet != ERROR_SUCCESS )
		{
			rc = -2;
			break;
		}
		if (dwType != REG_SZ) continue;
		if (!_tcsnicmp(tdat, data, dwLenDat))
		{
			if (dwLenVal < (DWORD) sz)
			{
				_tcsncpy(name, tval, dwLenVal);
				rc = dwLenVal;
			}
			else
			{
				_tcsncpy(name, tval, sz);
				rc = sz;
			}
			break;
		}
		nEnumIndex++;
	} while ( ulRet == ERROR_SUCCESS );

	RegCloseKey( hKey );

	return rc;
}
