/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    lcm.c

    Routines to operate the LCM panel on an embedded computer.

    2008-07-01	CF Lin
*/

#include <malloc.h>
#include <stdio.h>
#include "mxlcm.h"

#ifdef CTL_CODE
#undef CTL_CODE
#endif

#define CTL_CODE(DeviceType, Function, Method, Access) ( \
  ((DeviceType) << 16) | ((Access) << 14) | ((Function) << 2) | (Method) \
)

#define IOCTL_LCM_CLEAR_SCREEN      CTL_CODE( 0, 1, 0, 0)
#define IOCTL_LCM_GET_X_RANGE       CTL_CODE( 0, 2, 0, 0)
#define IOCTL_LCM_GET_Y_RANGE       CTL_CODE( 0, 3, 0, 0)
#define IOCTL_LCM_GET_X		        CTL_CODE( 0, 4, 0, 0)
#define IOCTL_LCM_GET_Y				CTL_CODE( 0, 5, 0, 0)
#define IOCTL_LCM_BACKLIGHT_ON		CTL_CODE( 0, 6, 0, 0)
#define IOCTL_LCM_BACKLIGHT_OFF		CTL_CODE( 0, 7, 0, 0)
#define IOCTL_LCM_CLEAR_LINE		CTL_CODE( 0, 8, 0, 0)
#ifdef UC74XX
#define IOCTL_LCM_GOTO_X			CTL_CODE( 0, 9, 0, 0)
#define IOCTL_LCM_GOTO_Y			CTL_CODE( 0, 10, 0, 0)
#else /* DA-66X */
#undef IOCTL_LCM_GOTO_XY
#undef IOCTL_LCM_GET_XY
#define IOCTL_LCM_GOTO_XY			CTL_CODE( 0, 45, 0, 0)
#define IOCTL_LCM_GET_XY			CTL_CODE( 0, 46, 0, 0)
#endif

/* mapping from Linux element 0 to WinCE element 1 */
static int ioctl_mappings[][2] = 
{
	{IOCTL_LCM_CLEAN_SCREEN,	IOCTL_LCM_CLEAR_SCREEN}, 
	{IOCTL_LCM_CLEAN_LINE,		IOCTL_LCM_CLEAR_LINE}, 
	{IOCTL_LCM_BACK_LIGHT_ON,	IOCTL_LCM_BACKLIGHT_ON}, 
	{IOCTL_LCM_BACK_LIGHT_OFF,	IOCTL_LCM_BACKLIGHT_OFF}, 
};

void
mxlcm_close(HANDLE fd)
{
	CloseHandle(fd);
}

/* control the LCM screen */
int
mxlcm_control(HANDLE fd, int code)
{
	DWORD  dwBack, size;
	DWORD  ret;
	int i;
	
	for (i=0; i < sizeof(ioctl_mappings)/sizeof(ioctl_mappings[0]);i++)
	if (code==ioctl_mappings[i][0])
	{
		ret = DeviceIoControl(fd, ioctl_mappings[i][1], NULL, 0, (LPVOID)&dwBack, sizeof(DWORD), &size, NULL);
		return ret? 0:-1;
	}
	return 1;
}

#ifndef UC74XX /* DA-66X */
typedef struct _LCM_POS
{
	UCHAR		x;
	UCHAR		y;
} LCM_POS;
#endif

int
mxlcm_get_cursor(HANDLE fd, int *x, int *y)
{
	DWORD  size, ret;
#ifdef UC74XX /* UC-74xx */
	BYTE   xb, yb;

	ret = DeviceIoControl(fd, IOCTL_LCM_GET_X, NULL, 0, (LPVOID)&xb, sizeof(BYTE), &size, NULL);
	if (ret == FALSE)
		return -1;
	ret = DeviceIoControl(fd, IOCTL_LCM_GET_Y, NULL, 0, (LPVOID)&yb, sizeof(BYTE), &size, NULL);
	if (ret == FALSE)
		return -2;
	*x = xb;
	*y = yb;
#else /* DA-66X */
	LCM_POS		pos;

	ret = DeviceIoControl(fd, IOCTL_LCM_GET_XY, NULL, 0, (LPVOID)(&pos), sizeof(LCM_POS), &size, NULL);
	if (ret == 0)
		return -2;
	
	*x = (unsigned int)pos.x;
	*y = (unsigned int)pos.y;
#endif
	return 0;
}

int
mxlcm_set_cursor(HANDLE fd, int x, int y)
{
	DWORD  size, ret;
#ifdef UC74XX /* UC-74xx */
	BYTE   xb, yb;

	xb = x;
	yb = y;
	ret = DeviceIoControl(fd, IOCTL_LCM_GOTO_X, (LPVOID)(&xb), sizeof(BYTE), NULL, 0, &size, NULL);
	if (ret == FALSE)
		return -1;
	ret = DeviceIoControl(fd, IOCTL_LCM_GOTO_Y, (LPVOID)(&yb), sizeof(BYTE), NULL, 0, &size, NULL);
#else /* DA-66X */
	LCM_POS		pos;

	pos.x = (UCHAR)x;
	pos.y = (UCHAR)y;
	ret = DeviceIoControl(fd, IOCTL_LCM_GOTO_XY, (LPVOID)(&pos), sizeof(LCM_POS), NULL, 0, &size, NULL);
#endif
	if (ret == FALSE)
		return -2;
	return 0;
}

/* open a handle to the LCM */
HANDLE
mxlcm_open(void)
{
	HANDLE fd;

	fd = CreateFile ( L"LCM1:", GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
	if (fd==INVALID_HANDLE_VALUE)
		return NULL;
	else
		return fd;
}

int
mxlcm_write(HANDLE fd, int x, int y, char *str, int len)
{
	DWORD n;

	if (mxlcm_set_cursor(fd, x, y)==0)
	{
	    if (!WriteFile( fd, str, len, &n, 0))
		    return 0;
	    return n;
	}
	else
		return -1;
}

int
mxlcm_write_screen(HANDLE fd, char text[MAX_LCM_ROWS][MAX_LCM_COLS])
{
	int y;

	mxlcm_clear_screen(fd);
#ifdef UC74XX /* UC-74xx */
	for(y=0; y < MAX_LCM_ROWS-1; y++)
#else /* DA-66X */
	for(y=0; y < MAX_LCM_ROWS; y++)
#endif
	{
		mxlcm_write(fd, 0, y, text[y], MAX_LCM_COLS);
	}
	return 0;
}
