
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "modbus.h"

#define ASCII_CR		0x0d
#define ASCII_LF		0x0a
#define ASCII_COLON		':'

static char hex[] = "0123456789ABCDEF";

static void
expand_hex (char *v, int vsize)
{
    int     i,j;

    for (i = vsize - 1; i >= 0; i--)
    {
		j = i << 1;
		v[j + 1] = hex[v[i] & 0xf];
		v[j] = hex[(v[i] >> 4) & 0xf];
    }
}

static void
shrink_hex (char *v, int vsize)
{
    int     i;

    for (i = 0 ; i < vsize; i++)
	{
		if ( v[i] <= '9' && v[i] >= '0' ) v[i] = v[i]-'0';
		else if (v[i] >='A' && v[i] <= 'F') v[i] = v[i]-'A'+10;
		else if (v[i] >='a' && v[i] <= 'f') v[i] = v[i]-'a'+10;
	}

    for (i = 0; i < vsize; i+=2)
    {
		v[i>>1] = v[i] << 4 | v[i+1];
    }
}

static unsigned char
do_LRC(unsigned char *buf, int len)
{
	int		i;
	unsigned char	lrc=0;

	for ( i=0; i<len; i++ )
		lrc += *buf++;
	return ((unsigned char)(-((char)lrc)));
}

/* 	digest a ASCII ADU.
		<ASCII_COLON><hex_data><hex_LRC><ASCII_CR><ASCII_LF>
	Inputs:
		<frame> data packet
		<plen> the length of data packet
	Outputs:
		<data> PDU data <bin_data>
		<plen> the length of data that is consumed
	Returns:
		the # of bytes in a packet excluding crc, 0 meaning incomplete/wrong crc packet
*/
unsigned int
mbasc_packet_digest(unsigned char *frame, unsigned int *plen, unsigned char *data)
{
	unsigned int size, dlen, consumed, data_size;
	unsigned char *pkt, *end;

	dlen = *plen;
	consumed = dlen;
	size = data_size = 0;
	pkt = frame;
	end = frame+dlen;
	/* search for ASCII_COLON */
	while(pkt != end && *pkt != ASCII_COLON) pkt++;
	if (pkt==end) /* no ASCII_COLON code, consume all */
	{
		*plen = consumed;
		return 0;
	}
	consumed = pkt-frame; /* consume possible data in front of <ASCII_COLON> */
	/* search for ASCII_CR/ASCII_LF */
	frame = pkt++;
	while(pkt != end && *pkt != ASCII_CR) 
		data[size++] = *pkt++; /* copy <hex_data><hex_LRC:2> */
	if (pkt==end || ++pkt==end) /* no ASCII_CR and ASCII_LF, or no ASCII_LF */
	{
		*plen = consumed;
		return 0;
	}
	consumed++; /* ASCII_COLON */
	consumed += size+2;
	if (*pkt == ASCII_LF) /* consume all */
	{
		shrink_hex(data, size);
		size = (size>>1)-1;
		if (data[size] == do_LRC(data, size))
			data_size = size;
	}
	*plen = consumed;
	return data_size;
}

/* 	format an ASCII ADU.
	Inputs:
		<data> PDU data data>
		<dlen> the length of PDU data
	Outputs:
		<frame> ADU frame <ASCII_COLON><hex_data><hex_LRC><ASCII_CR><ASCII_LF>
	Returns:
		the length of a ADU frame
*/
unsigned int
mbasc_packet_format(unsigned char *data, unsigned int dlen, unsigned char *frame)
{
	unsigned char *pkt = frame +1;
	unsigned int size = dlen;

	frame[0] = ASCII_COLON;
	memcpy(pkt, data, size); /* copy */
	pkt[size] = do_LRC(data, size);
	expand_hex(pkt, ++size);
	size <<= 1;
	pkt[size++] = ASCII_CR;
	pkt[size++] = ASCII_LF;
	return 1+size;
}
