/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    netiface_wce.c

    Routines of processing network interfaces.

    2009-01-15	CF Lin
		new release
*/
#include <windows.h>
#include <stdio.h>
#include <Iphlpapi.h>
#include "netiface.h"
#include "registry.h"

//-------------------------------------------------------------
//	macros and constants
//-------------------------------------------------------------
#define FILE_DEVICE_PHYSICAL_NETCARD    0x00000017
#define FILE_ANY_ACCESS                 0
#define WINCE_IOCTL_START				8
#define METHOD_OUT_DIRECT               2

#define CTL_CODE( DeviceType, Function, Method, Access ) \
( \
    ((DeviceType) << 16) | ((Access) << 14) | ((Function) << 2) | (Method) \
)

#define _NDIS_CONTROL_CODE(request,method) \
            CTL_CODE(FILE_DEVICE_PHYSICAL_NETCARD, request, method, FILE_ANY_ACCESS)

#define	IOCTL_NDIS_GET_ADAPTER_NAMES	_NDIS_CONTROL_CODE( WINCE_IOCTL_START+6, METHOD_OUT_DIRECT )
#define	IOCTL_NDIS_GET_ADAPTER_BINDINGS	_NDIS_CONTROL_CODE( WINCE_IOCTL_START+8, METHOD_OUT_DIRECT )
#define	IOCTL_NDIS_REBIND_ADAPTER		_NDIS_CONTROL_CODE( WINCE_IOCTL_START+3, METHOD_OUT_DIRECT )

#define DD_NDIS_DEVICE_NAME TEXT("NDS0:")

/* DA660 */
static WCHAR *szInterfaceIndex[] = { L"IXP4XXETHNPE1", L"IXP4XXETHNPE2", L"PCI\\RTL81391", L"PCI\\RTL81392"};
static int szInterfaceSize = sizeof(szInterfaceIndex)/sizeof(szInterfaceIndex[0]);

/* io control for netwotk interface */
static BOOL 
DoNdisIOControl(DWORD dwCommand, LPVOID pInBuffer, DWORD cbInBuffer, 
                LPVOID pOutBuffer, DWORD * pcbOutBuffer)
{
   HANDLE hNdis;
   BOOL	fResult = FALSE;

   hNdis = CreateFile(DD_NDIS_DEVICE_NAME, GENERIC_READ | GENERIC_WRITE,
			FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, 0, NULL);
   if ( hNdis != INVALID_HANDLE_VALUE ) 
   {
      fResult = DeviceIoControl(hNdis, dwCommand, pInBuffer, cbInBuffer,
			pOutBuffer, (pcbOutBuffer ? *pcbOutBuffer : 0), pcbOutBuffer, NULL);
      CloseHandle(hNdis);
   }
   return fResult;
}

/* Get the interface configuration from the kernel. 
    Inputs:
        <ifname> the name of the interface, NULL for all interfaces
		<ifaces> an empty buffer 
		<max> the maximum interfaces
    Outputs:
        <ifaces> the buffer containing interface information
    Returns:
        the number of interfaces
*/
int
mxiface_get_info(char *ifname, MXIFACE *ifaces, int max)
{
	PIP_ADAPTER_INFO list;
    IP_ADAPTER_INFO pAdapterInfo[MAX_IFACE];
    ULONG ulSizeAdapterInfo;
	MXIFACE *ifr;
	int i = 0;

	memset(ifaces, 0, max*sizeof(MXIFACE));
    /* get all adapters */
	ulSizeAdapterInfo = MAX_IFACE * sizeof(IP_ADAPTER_INFO);
    if (GetAdaptersInfo( pAdapterInfo, &ulSizeAdapterInfo) == NO_ERROR)
	for (list = pAdapterInfo;list && i < max; list = list->Next) 
	{
		ifr = &ifaces[i];
		/* go through adapters */
		strcpy( ifr->ifname,  list->AdapterName );
	//printf("%d,%s]\n", i, ifr->ifname);
		strcpy( ifr->ipaddr,  list->IpAddressList.IpAddress.String );
		strcpy( ifr->netmask, list->IpAddressList.IpMask.String );
		strcpy( ifr->gateway, list->GatewayList.IpAddress.String );
		ifr->enable_dhcp = list->DhcpEnabled;
		if (list->DhcpEnabled)
			strcpy( ifr->dhcpsrv, list->DhcpServer.IpAddress.String );
		else
			ifr->dhcpsrv[0] = 0;
		/* MAC address */
		memcpy(ifr->macaddr, list->Address, 6);
		if (!ifname)
		{
			i++;
			continue;
		}
		if (strcmp(list->AdapterName, ifname)==0)
		{
			i = 1;
			break;
		}
	}
	return i;
}

/* Get the interface configuration from the registry. 
    Inputs:
        <ifname> the name of the interface, NULL for all interfaces
		<ifaces> an empty buffer 
		<max> the maximum interfaces
    Outputs:
        <ifaces> the buffer containing interface information
    Returns:
        the number of interfaces
*/
int
mxiface_get_info_static(char *ifname, MXIFACE *ifaces, int max) 
{
	int i;
	char tmp[MAX_PATH];
	WCHAR keypath[MAX_PATH], temp[MAX_PATH];
	MXIFACE *ifr;
	DWORD n;

	max = mxiface_get_info(ifname, ifaces, max);
	for (i=0; i < max; i++)
	{
		ifr = &ifaces[i];
		/* adapter_name */
		sprintf(tmp, "Comm\\%s\\Parms\\TcpIp", ifr->ifname);
		mbyte2wchar(tmp, keypath, MAX_PATH);
		//printf("%S\n", keypath);

		if (registry_machine_get(keypath, L"IpAddress", temp) < 0)
			ifr->ipaddr[0] = 0;
		else wchar2mbyte(temp, ifr->ipaddr, 16);

		if (registry_machine_get(keypath, L"Subnetmask", temp) < 0)
			ifr->netmask[0] = 0;
		else wchar2mbyte(temp, ifr->netmask, 16);

		if (registry_machine_get(keypath, L"DefaultGateway", temp) < 0)
			ifr->gateway[0] = 0;
		else wchar2mbyte(temp, ifr->gateway, 16);

		if (registry_machine_get(keypath, L"EnableDHCP", (WCHAR *)&n) < 0)
			ifr->enable_dhcp = 0;
		else ifr->enable_dhcp = n;
	}
	return max;
}

int
mxiface_get_mac_address(char *ifname, unsigned char *mac)
{
	MXIFACE ifaces, *ifr;

	ifr = &ifaces;
	if (mxiface_get_info(ifname, ifr, 1))
	{
		memcpy(mac, ifr->macaddr, 6);
		return 0;
	}
	else
		return -1;
}

/*  change the settings of an interface onto file
    Inputs:
        <ifr> the interface 
*/
int
mxiface_update_interface_file(MXIFACE *ifr)
{
	WCHAR keypath[MAX_PATH], temp[MAX_PATH];
	char tmp[MAX_PATH];
	DWORD n;

	if (ifr == NULL || ifr->ifname[0] == 0 ||
		ifr->ipaddr[0] == 0 || ifr->netmask[0] == 0) return -1;

	/* adapter_name */
	sprintf(tmp, "Comm\\%s\\Parms\\TcpIp", ifr->ifname);
	mbyte2wchar(tmp, keypath, MAX_PATH);
	//printf("%S\n", keypath);

	/* update the ip address */
	mbyte2wchar(ifr->ipaddr, temp, MAX_PATH);
	registry_machine_update(keypath, L"IpAddress", temp, REG_SZ);

	/* update the netmask */
	mbyte2wchar(ifr->netmask, temp, MAX_PATH);
	registry_machine_update(keypath, L"Subnetmask",	temp, REG_SZ);

	/* update the gateway */
	mbyte2wchar(ifr->gateway, temp, MAX_PATH);
	registry_machine_update(keypath, L"DefaultGateway", temp, REG_SZ);

	if (ifr->enable_dhcp) n = 1;
	else n = 0;
	registry_machine_update(keypath, L"EnableDHCP", (WCHAR *)&n, REG_DWORD);

	return 0;
}

/*  Get the list of DNS servers from a networking interface
    Inputs:
        <ifname> the name of an interface
		<dns_list> buffer for the list of DNS servers
		<size> size of buffer for the list of DNS servers
	Outputs:
		<dns_list> the list of DNS servers separated by a space
    Returns:
	    0 on success
*/
int
mxhost_get_dns(char *ifname, char *dns_list, int size)
{
	WCHAR keypath[MAX_PATH], temp[MAX_PATH];
	char tmp[MAX_PATH];
	char *p;
	int n, m;

	if (ifname == NULL || ifname[0] == 0) return -1;

	sprintf(tmp, "Comm\\%s\\Parms\\TcpIp", ifname);
	mbyte2wchar(tmp, keypath, MAX_PATH);
	//printf("%S\n", keypath);

	if ((n = registry_machine_get(keypath, L"DNS", temp)) < 0) return -2;
	m = WideCharToMultiByte(CP_ACP, 0, temp, n/2, dns_list, size, NULL, NULL);
	for (n = 0, p = dns_list; n < m-1; n++, p++)
	{
		if (*p == 0 && *(p+1) == 0) break;
		if (*p == 0) *p = ' ';
	}
	return 0;
}

/*  Update the list of DNS servers onto a networking interface
    Inputs:
        <ifname> the name of the updated interface
		<dns_list> the list of DNS servers separated by a space
    Returns:
	    0 on success
*/
int
mxhost_update_dns(char *ifname, char *dns_list)
{
	WCHAR keypath[MAX_PATH], temp[MAX_PATH];
	char tmp[MAX_PATH];
	char *p;
	int i, n;

	if (ifname == NULL || ifname[0] == 0) return -1;
	if (dns_list == NULL /* || dns_list[0] == 0*/) return -2;

	sprintf(tmp, "Comm\\%s\\Parms\\TcpIp", ifname);
	mbyte2wchar(tmp, keypath, MAX_PATH);

	n = strlen(dns_list);
	if (n+2 > MAX_PATH) return -3;
	strcpy(tmp, dns_list);
	for (i = 0, p = tmp; i < n; i++, p++)
		if (*p == ' ') *p = 0;
	*p = 0; n++; tmp[n] = 0;
	MultiByteToWideChar(CP_ACP, 0, tmp, n+1, temp, MAX_PATH);
	if (registry_machine_update(keypath, L"DNS", temp, REG_MULTI_SZ) < 0)
		return -4;
	return 0;
}

static int
split_wtoken (WCHAR **temp, int tempcount, WCHAR *pkt)
{
    int     i = 0, len;

    if (!pkt)
	return -1;

    while(i < tempcount)
    {
	len = wcslen(pkt);
	if (len==0)
		break;
	temp[i++] = pkt;
	pkt += (len+1);
    }
    return i;
}

static int
mxiface_reorder(WCHAR **temp, int max, WCHAR *pkt)
{
	int i, k, num;

	if (max < szInterfaceSize)
		return 0;
	max = szInterfaceSize;
	num = split_wtoken (temp, max, pkt);
	for (k=0; k < max; k++)
	for (i=k; i < num; i++)
	{
		if (wcscmp(temp[i], szInterfaceIndex[k])==0)
		{
			/* swap */
			pkt = temp[i]; /* found */
			temp[i] = temp[k];
			temp[k] = pkt;
			break;
		}
	}
	return num;
}

int
mxiface_rebind(char *ifname)
{
	WCHAR wszAdapterNames[512], *pInterface[MAX_IFACE];
    TCHAR *pCurrentBinding, szAdapterBindings[512] = {0};
    DWORD ifrlen, buflen;
	WCHAR wmszTmp[100];
    int i, num_ifaces;

	if (ifname)
	{
		mbyte2wchar(ifname, wszAdapterNames, 512);
		num_ifaces = 1;
		pInterface[0] = wszAdapterNames;
	}
	else
	{
    /* Get the array of adapters */
	buflen = sizeof(wszAdapterNames);
    if (!DoNdisIOControl(IOCTL_NDIS_GET_ADAPTER_NAMES, NULL, 0, wszAdapterNames, &buflen)) 
	{
		return -1;
    }
	num_ifaces = mxiface_reorder (pInterface, MAX_IFACE, wszAdapterNames);
	}
	for (i=0; i < num_ifaces; i++)
    {
		//wprintf(L"pCurrentBinding: %s\n", pInterface[i]);
		buflen = sizeof(szAdapterBindings);
		ifrlen = (wcslen(pInterface[i])+1)*sizeof(WCHAR);
        if(!DoNdisIOControl(IOCTL_NDIS_GET_ADAPTER_BINDINGS, pInterface[i], ifrlen, szAdapterBindings, &buflen)) 
		{
			return -1;
        }
        /* Loop through all bindings of this adapter for TCP/IP */
        pCurrentBinding = szAdapterBindings;
        while(*pCurrentBinding)
        {
			//wprintf(L"To rebind: %s\n", pCurrentBinding);
			if (!_tcsicmp(pCurrentBinding, _T("TCPIP"))     ||
                !_tcsicmp(pCurrentBinding, _T("80211SIMP")) ||
                !_tcsicmp(pCurrentBinding, _T("MBRIDGE"))   ||
                !_tcsicmp(pCurrentBinding, _T("MSTCP"))) 
            {
				wcscpy(wmszTmp, pInterface[i]);
				wmszTmp[wcslen(wmszTmp)+1] = 0x00;
				DoNdisIOControl(IOCTL_NDIS_REBIND_ADAPTER, wmszTmp, ifrlen+sizeof(WCHAR), NULL, NULL);
				/* Give NDIS some time before next rebind */
				Sleep(3000);
                break;
            }
            pCurrentBinding += _tcslen(pCurrentBinding)+1;
        }
    }
	return 0;
}

/*  restart the network driver
	Returns:
	    0 on success
*/
int
mxiface_restart(void)
{
	return mxiface_rebind(NULL);
}
