/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    usb_cf.c

    Routines to check USB/CF status and avaiable space.

    2008-07-15	CF Lin
		new release
*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/statfs.h>
#include <sys/stat.h>
#include <string.h>

#define BUFFER_SIZE		128

/*  check the avaiable space of the CF card
    Returns:
	space in bytes, otherwise 0, for no card there
*/
int	
cf_available_space(void)
{
    struct statfs s;

    if (statfs( "/mnt/hda", &s) != 0 ||
        s.f_blocks <= 0) 
	return 0;
    else
	return s.f_bavail*s.f_bsize;
}

/*	check the avaiable space of the USB storage
	Inputs:
		<num> 0 or 1, specify the USB
	Returns:
		space in bytes, otherwise 0, for no USB there
*/
int
usb_available_space(int num)
{
    struct statfs s;
    char mount[16];

    sprintf(mount, "/mnt/sda%d", num==0? 0:1);
    if (statfs( mount, &s) != 0 ||
	s.f_blocks <= 0) 
	return 0;
    else
	return s.f_bavail*s.f_bsize;
}

static int
file_is_there(char *fname)
{
    struct stat st;

    return (stat(fname, &st)? 0:1);
}

static int
device_is_mounted(char *fname, char *key)
{
	FILE *f;
	char buf[BUFFER_SIZE];
	int	 status = 0, len;

	len = strlen(key);
	/* mounted or not */
	f = fopen(fname, "r");
	if ( f )
	{
		while ( fgets(buf, BUFFER_SIZE, f) ) 
		if ( strncmp(buf, key,len) == 0 ) 
		{
			status = 1;
			break;
		}
		fclose(f);
	}
	return status;
}

/*   check if the CF card is in place
     Returns:
        1, the card is plugged
	2, the card is mounted
	otherwise 0 
*/
int
cf_is_in_place(void)
{
	int status = 0;

	/* in plug or not */
	if (file_is_there("/proc/ide/hda")) 
	{
		status = 1;
		if (device_is_mounted("/proc/mounts", "/dev/hda1"))
		status = 2;
	}
	return status;
}

/*	check if the USB is in place
	Inputs:
		<num> 0 or 1, specify the USB
	Returns:
		1, the card is plugged
		2, the card is mounted
		otherwise 0 
*/
int	
usb_is_in_place(int num)
{
	char	fname[128];
	int	status = 0;

	if (num) num = 1;
	/* pulgged or not */
	sprintf(fname, "/proc/scsi/usb-storage-%d/%d", num, num);
	if (device_is_mounted(fname, "Attached: No"))
	{
		status = 1;
		/* mounted or not */
		if (device_is_mounted("/proc/mounts", ( num == 0 )? "/dev/sda1":"/dev/sdb1"))
			status = 2;
	}
	return status;
}

