/*  Copyright (C) MOXA Inc. All rights reserved.
 *
 *  This software is distributed under the terms of the
 *  MOXA License.  See the file COPYING-MOXA for details.
 */
/*
 *  netiface_win.c
 *
 *  Routines to process network interfaces
 *
 *  Modification History:
 *    #2008-09-03 Peter Wu	new release
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <winsock2.h>
#include <windows.h>
#include <iphlpapi.h>
#include <tchar.h>
#include <shellapi.h>
#include "netiface.h"
#include "registry.h"

/*
 * ########################
 * MACRO
 * ########################
 */
#ifdef UNICODE
#define mbyte2tchar(src, dest, size)    MultiByteToWideChar(CP_ACP, 0, src, -1, dest, size)
#define tchar2mbyte(src, dest, size)    WideCharToMultiByte(CP_ACP, 0, src, -1, dest, size, NULL, NULL)
#else
#define mbyte2tchar(src, dest, size)    strcpy(dest, src)
#define tchar2mbyte(src, dest, size)    strcpy(dest, src)
#endif

#define LAN_PATH		_T("SYSTEM\\CurrentControlSet\\Control\\Network")


/*
 * ########################
 * TYPE
 * ########################
 */


/*
 * ########################
 * GLOBAL VARIABLES
 * ########################
 */


/*
 * ########################
 * UTILITY FUNCTIONS
 * ########################
 */
static int
mxiface_exec_netsh(char *param)
{
	SHELLEXECUTEINFO ExecInfo;
	TCHAR	buffer[MAX_PATH];


	mbyte2tchar(param, buffer, sizeof(buffer));

	ExecInfo.cbSize = sizeof(ExecInfo);
	ExecInfo.lpDirectory = 0;
	ExecInfo.fMask = SEE_MASK_FLAG_NO_UI;
	ExecInfo.hProcess = NULL;
	ExecInfo.lpVerb = _T("Open");
	ExecInfo.lpFile = _T("netsh.exe");
	ExecInfo.lpParameters = buffer;

	if (ShellExecuteEx(&ExecInfo) == TRUE)
		return 0;

	return -1;
}


static int
mxiface_get_dns_list(char *adapter, char *dns_list, int size)
{
	TCHAR	buffer[MAX_PATH];
	TCHAR	keypath[MAX_PATH];
	char	*p;


	mbyte2tchar(adapter, buffer, sizeof(buffer));
	wsprintf(keypath, _T("SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Parameters\\Interfaces\\%s"), buffer);
	if (registry_machine_get(keypath, _T("NameServer"), buffer, MAX_PATH) < 0)
		return -1;
	tchar2mbyte(buffer, dns_list, size);

	for (p = dns_list; *p; p++) if (*p == ',') *p = ' ';
	return 0;
}


int
mxiface_get_net_name(char *adapter, char *netname, int size)
{
	TCHAR	data[MAX_PATH];
	TCHAR	name[MAX_PATH];
	int		len;


	mbyte2tchar(adapter, name, sizeof(name));
	len = registry_machine_find_data(LAN_PATH, name, _T("Connection"), _T("Name"), data);
	if (len <= 0)
        return -1;
	data[len] = 0;

	tchar2mbyte(data, netname, size);
	return 0;
}


int
mxiface_get_adapter_name(char *netname, char *adapter, int size)
{
    int     i;
    int     count;
    char    buffer[MAX_PATH];
    MXIFACE ifaces[MAX_IFACE];


    count = mxiface_get_info(NULL, ifaces, MAX_IFACE);
    if (count == 0)
        return -1;

    for (i = 0; i < count; i++)
    {
        if (mxiface_get_net_name(ifaces[i].ifname, buffer, sizeof(buffer)) == 0 &&
            strcmp(netname, buffer) == 0)
        {
            strncpy(adapter, ifaces[i].ifname, size);
            return 0;
        }
    }
    return -2;
}


/*
 * ########################
 * FUNCTIONS           
 * ########################
 */
/*	Get the interface configuration from the kernel/file. 
	Inputs:
		<ifname> the name (netname) of the interface, NULL for all interfaces
		<max> the maximum interfaces
	Outputs:
		<ifaces> the buffer containing interface information
	Returns:
		the number of interfaces
*/
int
mxiface_get_info(char *ifname, MXIFACE *ifaces, int max)
{
	IP_ADAPTER_INFO *pAdapterInfo;
	IP_ADAPTER_INFO *pAdapter;
	ULONG			ulOutBufLen;
	MXIFACE *ifr = ifaces;
	int num = 0;

	if (ifaces == NULL || max == 0) return 0;

	memset(ifaces, 0, max * sizeof(MXIFACE));

	pAdapterInfo = NULL;
	ulOutBufLen = 0;
	if (GetAdaptersInfo(pAdapterInfo, &ulOutBufLen) != ERROR_BUFFER_OVERFLOW) 
		return 0;
	/* assume ulOutBufLen greater than 0 */
	pAdapterInfo = (IP_ADAPTER_INFO *) malloc ( ulOutBufLen );
	if (pAdapterInfo == NULL) {
		return 0;
	}
	if (GetAdaptersInfo(pAdapterInfo, &ulOutBufLen) != ERROR_SUCCESS) {
		free(pAdapterInfo);
		return 0;
	}
	pAdapter = pAdapterInfo;
	while (pAdapter) {
#if 1
		mxiface_get_net_name(pAdapter->AdapterName, ifr->ifname, sizeof(ifr->ifname));
#else
		 strcpy(ifr->ifname,  pAdapter->AdapterName);
#endif
		strcpy(ifr->ipaddr,  pAdapter->IpAddressList.IpAddress.String);
		strcpy(ifr->netmask, pAdapter->IpAddressList.IpMask.String);
		strcpy(ifr->gateway, pAdapter->GatewayList.IpAddress.String);
		ifr->enable_dhcp = pAdapter->DhcpEnabled;
		if (pAdapter->DhcpEnabled)
			strcpy(ifr->dhcpsrv, pAdapter->DhcpServer.IpAddress.String);
		else
			ifr->dhcpsrv[0] = 0;
		memcpy(ifr->macaddr, pAdapter->Address, 6);
		if (!ifname)
		{
			num++;
			if (num >= max) break;
			ifr++;
		}
		else
		{
			/* specified iface name found */
			if (!strcmp(ifr->ifname, ifname)) 
			{
				num = 1;
				break;
			}
		}
		pAdapter = pAdapter->Next;
	}
	free (pAdapterInfo);
	return num;
}

/*	get the mac address of an interface
	Inputs:
		<ifname> the name (netname) of the interface
	Outputs:
		<mac> mac address
	Returns:
		0 on success
*/
int
mxiface_get_mac_address(char *ifname, unsigned char *mac)
{
	MXIFACE ifaces, *ifr;

	ifr = &ifaces;
	if (mxiface_get_info(ifname, ifr, 1))
	{
		memcpy(mac, ifr->macaddr, 6);
		return 0;
	}
	else
		return -1;
}

/*  Update the configuration of an interface onto the file.
    Inputs:
        <iface> info of the interface
*/
int
mxiface_update_info(MXIFACE *ifr)
{
	char	buf[MAX_PATH];


	memset(buf, 0x0, sizeof(buf));
	if (!ifr->enable_dhcp)
	{
		if (*ifr->gateway)
			sprintf(buf, "interface ip set address name=\"%s\" static %s %s %s 1", ifr->ifname, ifr->ipaddr, ifr->netmask, ifr->gateway);
		else
			sprintf(buf, "interface ip set address name=\"%s\" static %s %s none", ifr->ifname, ifr->ipaddr, ifr->netmask);
	}
	else
		sprintf(buf, "interface ip set address name=\"%s\" dhcp", ifr->ifname);

	mxiface_exec_netsh(buf);
	return 0;
}

/*  restart the network interfaces 
    Returns:
        0 on success, otherwise on failure
*/
int
mxiface_restart(void)
{
	int i;
	int num;
	MXIFACE ifaces[MAX_IFACE];


	num = mxiface_get_info(NULL, ifaces, MAX_IFACE);
	if (num <= 0)
		return -1;

	for (i = 0; i < num; i++)
		mxiface_update_info(&ifaces[i]);

	return 0;
}

/*  Get the list of DNS servers from a networking interface
	Inputs:
		<ifname> the name of an interface
		<dns_list> buffer for the list of DNS servers
		<size> size of buffer for the list of DNS servers
	Outputs:
		<dns_list> the list of DNS servers separated by a space
	Returns:
		0 on success
*/
int
mxiface_get_dns(char *ifname, char *dns_list, int size)
{
	IP_ADAPTER_INFO *pAdapterInfo;
	IP_ADAPTER_INFO *pAdapter;
	ULONG			ulOutBufLen;
	int		found = 0;


	pAdapterInfo = NULL;
	ulOutBufLen = 0;
	if (GetAdaptersInfo(pAdapterInfo, &ulOutBufLen) != ERROR_BUFFER_OVERFLOW) 
		return -1;
	/* assume ulOutBufLen greater than 0 */
	pAdapterInfo = (IP_ADAPTER_INFO *) malloc ( ulOutBufLen );
	if (pAdapterInfo == NULL) {
		return -1;
	}
	if (GetAdaptersInfo(pAdapterInfo, &ulOutBufLen) != ERROR_SUCCESS) {
		free (pAdapterInfo);
		return -1;
	}
	pAdapter = pAdapterInfo;
	while (pAdapter) {
		char NetName[MAX_PATH];
		if (mxiface_get_net_name(pAdapter->AdapterName, NetName, sizeof(NetName)) == 0) {
			if (!strcmp(NetName, ifname))
			{
				found = 1;
				break;
			}
		}
		pAdapter = pAdapter->Next;
	}

	if (!found || mxiface_get_dns_list(pAdapter->AdapterName, dns_list, size) < 0)
		return -1;

	free (pAdapterInfo);
	return 0;
}


/*  Update the list of DNS servers onto a networking interface
	Inputs:
		<ifname> the name of the updated interface
		<dns_list> the list of DNS servers separated by a space
    Returns:
	0 on success
*/
int
mxiface_update_dns(char *ifname, char *dns_list)
{
	int		i;
	int		num = 0;
	char	buf[MAX_PATH];
	char	dns_buf[MAX_PATH];		/* backup the dns list for separate */
	char	delim[] = " ";			/* delimiter of dns list */
	char *	dns[32];


	/* separate the dns from dns list */
	if (dns_list)
	{
		strcpy(dns_buf, dns_list);

		dns[num] = strtok(dns_buf, delim);
		while (dns[num] && num < 32)
			dns[++num] = strtok(NULL, delim);
	}

	/* setup the dns one-by-one */
	for (i = 0; i < num; i++)
	{
		if (i == 0)
			sprintf(buf, "interface ip set dns name=\"%s\" static %s",ifname, dns[i]);
		else
		{
			sprintf(buf, "interface ip add dns name=\"%s\" %s", ifname, dns[i]);
			/* how many are enough? */
			Sleep(30000);
		}
		if (mxiface_exec_netsh(buf) < 0)
			return -1;
	}
	sprintf(buf, "interface ip set dns name=\"%s\" static none", ifname);
	if (mxiface_exec_netsh(buf) < 0)
		return -1;
	return 0;
}
