/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    mserial_port.c

    Routines to operate serial ports

    2008-05-08	CF Lin
		new release
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include "mserial_port.h"


#if defined(_WIN32_WCE)
#define MOXA_SET_OP_MODE                (0x400+66)
#define MOXA_GET_OP_MODE                (0x400+67)
#define UC_SET_SPECIAL_BAUD_RATE        (0x400+68)
#define UC_GET_SPECIAL_BAUD_RATE        (0x400+69)
#define MOXA_SET_SPECIAL_BAUD_RATE      (0x400+100)
#define MOXA_GET_SPECIAL_BAUD_RATE      (0x400+101)
#else
#if defined(V21XX)
#define GPD_TYPE            40000
#define MOXA_SET_OP_MODE    CTL_CODE( GPD_TYPE, 0x916, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define MOXA_GET_OP_MODE    CTL_CODE( GPD_TYPE, 0x917, METHOD_BUFFERED, FILE_ANY_ACCESS)
#elif defined(V46X)
#define GPD_TYPE            40000
#define MOXA_SET_OP_MODE    CTL_CODE( GPD_TYPE, 0x914, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define MOXA_GET_OP_MODE    CTL_CODE( GPD_TYPE, 0x915, METHOD_BUFFERED, FILE_ANY_ACCESS)
#else
#define MOXA_IOCTL          0x800
#define MOXA_SET_OP_MODE    CTL_CODE(FILE_DEVICE_SERIAL_PORT, MOXA_IOCTL + 23, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define MOXA_GET_OP_MODE    CTL_CODE(FILE_DEVICE_SERIAL_PORT, MOXA_IOCTL + 24, METHOD_BUFFERED, FILE_ANY_ACCESS)
#endif
#endif

/* special for setting interface mode on V21XX and V46X */
#if defined(V21XX) || defined(V46X)
#define MAX_PORT_NUM        4
unsigned int moxa_port_handle[MAX_PORT_NUM] = { 0, 0, 0, 0 };

/* get the port number according to the given handle */
static int
_mxsp_get_port(unsigned int fd)
{
    int i;
    for (i = 0; i < MAX_PORT_NUM; i++)
    {
        if (moxa_port_handle[i] == fd)
            return i;
    }
    return -1;
}
#endif


#if defined( _WIN32_WCE)
static int moxa_1st_uart_port = 99;

static int
_mxsp_get_max_port(void)
{
	DEVMGR_DEVICE_INFORMATION dev;
	HANDLE	fDevice = NULL;
	wchar_t *p;
	int	num = 0;

	dev.dwSize = sizeof(DEVMGR_DEVICE_INFORMATION);
	fDevice = FindFirstDevice(  DeviceSearchByDeviceName, L"COM*",  &dev );
	while ( fDevice )
	{
		p = wcsstr( dev.szDeviceName, L"COM" );
#ifdef V46X
		if ( p && (wcsstr(dev.szDeviceKey, L"MU860") || wcsstr(dev.szDeviceKey, L"MU150") || wcsstr(dev.szDeviceKey, L"Serial")) )
#else
		if ( p && (wcsstr(dev.szDeviceKey, L"MU860") || wcsstr(dev.szDeviceKey, L"MU150")) )
#endif
		{
			long com = wcstol( p+3, NULL, 10 );

			if (com < moxa_1st_uart_port) moxa_1st_uart_port = com;
			num++;
		}
		if ( !FindNextDevice(fDevice, &dev ) )
			break;
	}
	return num;
}
#endif

/* set the baudrate on an open port
	Inputs:
		<comH> handler of the open port
		<baudrate> a specified baudrate
	Returns:
		0 on success, otherwise failure
*/
int
mxsp_set_baudrate( unsigned int fd, int baudrate)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	/* check and setup configuration */
	FillMemory(&dcb, sizeof(dcb), 0);
	if (GetCommState( comH, &dcb ) == 0)
		return -1;
	/* non-indexed buad rates have been handled in the driver level */
	dcb.DCBlength = sizeof(dcb);
	dcb.BaudRate  = baudrate;
	return (SetCommState(comH, &dcb) == 0) ? -2 : 0;
}

/* get the baudrate on an open port
	Inputs:
		<comH> handler of the open port
	Returns:
		negative value on failure, otherwise the baudrate
*/
int
mxsp_get_baudrate( unsigned int fd )
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	/* check and setup configuration */
	if (GetCommState( comH, &dcb ) == 0)
		return -1;
	else
		return dcb.BaudRate;
}

/* set the data bits on an open port
	Inputs:
		<comH> handler of the open port
		<bits> 5, 6, 7, or 8
	Returns:
		0 on success, otherwise failure
*/
int
mxsp_set_databits( unsigned int fd, int bits)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	if (bits < 4 || bits > 8)
		return -99;

	if (GetCommState( comH, &dcb ) == 0)
		return -1;
	dcb.DCBlength = sizeof(dcb);
	dcb.ByteSize  = bits;
	return (SetCommState(comH, &dcb) == 0) ? -2 : 0;
}

/* get the data bits on an open port
	Inputs:
		<comH> handler of the open port
	Returns:
		5~8 on success, otherwise failure
*/
int
mxsp_get_databits( unsigned int fd )
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	if (GetCommState( comH, &dcb ) == 0)
		return -1;
	else
		return (int) dcb.ByteSize;
}

/* set the # of stop bits on an open port
	Inputs:
		<comH> handler of the open port
		<bits> stop bits, 1, 2, or 3 (for 1.5)
	Returns:
		0 on success, otherwise failure
*/
int
mxsp_set_stopbits( unsigned int fd, int bits)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	/* check and setup configuration */
	if (GetCommState( comH, &dcb ) == 0)
		return -1;
	dcb.DCBlength	= sizeof(dcb);

	if (bits==1)		dcb.StopBits = ONESTOPBIT;
	else if (bits==2)	dcb.StopBits = TWOSTOPBITS;
	else if (bits==3)	dcb.StopBits = ONE5STOPBITS;
	else				
		return -99;

	return (SetCommState(comH, &dcb) == 0) ? -2 : 0;
}

/* get the # of stop bits of an open port
	Inputs:
		<comH> handler of the open port
	Returns:
		negative value on failure, or stop bits, 1, 2, and 3 (for 1.5)
*/
int
mxsp_get_stopbits( unsigned int fd)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;
	int	bits;

	/* check and setup configuration */
	if (GetCommState(  comH, &dcb )==0)
		return -1;

	if (dcb.StopBits==ONESTOPBIT)			bits = 1;
	else if (dcb.StopBits==TWOSTOPBITS)		bits = 2;
	else if (dcb.StopBits==ONE5STOPBITS)	bits = 3;
	else
		bits = -99;

	return bits;
}

/* set the parity of an open port
	Inputs:
		<comH> handler of the open port
		<parity> parity code , 1:none, 2:odd, 3:even, 4:space, 5:mark, otherwise:none
	Returns:
		negative value on failure, or the parity code
*/
int
mxsp_set_parity( unsigned int fd, int parity)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	/* check and setup configuration */
	if (GetCommState(  comH, &dcb ) == 0)
		return -1;
	dcb.DCBlength = sizeof(dcb);

	if (parity == MSP_PARITY_NONE)			dcb.Parity = NOPARITY;
	else if (parity == MSP_PARITY_ODD)		dcb.Parity = ODDPARITY;
	else if (parity == MSP_PARITY_EVEN)		dcb.Parity = EVENPARITY;
	else if (parity == MSP_PARITY_SPACE)	dcb.Parity = SPACEPARITY;
	else if (parity == MSP_PARITY_MARK)		dcb.Parity = MARKPARITY;
	else
		return -99;

	/* Specifies if parity checking is enabled. */
	dcb.fParity	= (dcb.Parity != NOPARITY) ? TRUE : FALSE; 
	return (SetCommState(comH, &dcb) == 0) ? -2 : 0;
}

/* get the parity of an open port
	Inputs:
		<comH> handler of the open port
	Returns:
		negative value on failure, or the parity code
*/
int
mxsp_get_parity( unsigned int fd)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;
	int	parity;

	/* check and setup configuration */
	if (GetCommState(  comH, &dcb ) == 0)
		return -1;

	if (dcb.Parity == NOPARITY)			parity = MSP_PARITY_NONE;
	else if (dcb.Parity == ODDPARITY)	parity = MSP_PARITY_ODD;
	else if (dcb.Parity == EVENPARITY)	parity = MSP_PARITY_EVEN;
	else if (dcb.Parity == SPACEPARITY)	parity = MSP_PARITY_SPACE;
	else if (dcb.Parity == MARKPARITY)	parity = MSP_PARITY_MARK;
	else								
		parity = -99;

	return parity;
}

/* set the flow control mode of an open port
	Inputs:
		<comH> handler of the open port
		<mode> 1:software, 2:hardware, otherwise:none
	Returns:
		0 on success, otherwise failure
*/
int
mxsp_set_flow_control( unsigned int fd, int mode)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	/* check and setup configuration */
	if (GetCommState(  comH, &dcb ) == 0)
		return -1;
	dcb.DCBlength = sizeof(dcb);
	if (mode == MSP_FLOWCTRL_NONE)			
	{
		dcb.fRtsControl	= RTS_CONTROL_DISABLE;
		dcb.fOutxCtsFlow= FALSE;
		dcb.fOutX		= FALSE;
		dcb.fInX		= FALSE;
		dcb.fErrorChar	= FALSE;
		dcb.fNull		= FALSE;
	}
	else if (mode == MSP_FLOWCTRL_SW)	
	{
		dcb.fRtsControl	= RTS_CONTROL_DISABLE;
		dcb.fOutxCtsFlow= FALSE;
		dcb.fOutX		= TRUE;
		dcb.fInX		= TRUE;
//		dcb.fTXContinueOnXoff = FALSE;
	}
	else if (mode == MSP_FLOWCTRL_HW) 
	{
		dcb.fRtsControl	= RTS_CONTROL_HANDSHAKE;
		dcb.fOutxCtsFlow= TRUE;
		dcb.fOutX		= FALSE;
		dcb.fInX		= FALSE;
		dcb.fErrorChar	= FALSE;
		dcb.fNull		= FALSE;
	}
	else
		return -99;

	return (SetCommState(comH, &dcb) == 0) ? -2 : 0;
}

/* get the flow control mode of an open port
	Inputs:
		<comH> handler of the open port
	Returns:
		negative value on failure, or the flow control mode
*/
int
mxsp_get_flow_control( unsigned int fd)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;
	int	mode;

	/* check and setup configuration */
	if (GetCommState(  comH, &dcb ) == 0)
		return -1;
	if ( dcb.fRtsControl == RTS_CONTROL_HANDSHAKE ) mode = MSP_FLOWCTRL_HW;
	else if ( (dcb.fRtsControl==RTS_CONTROL_DISABLE) && (dcb.fOutX==TRUE) ) mode = MSP_FLOWCTRL_SW;
	else if ( (dcb.fRtsControl==RTS_CONTROL_DISABLE) && (dcb.fOutX==FALSE) ) mode = MSP_FLOWCTRL_NONE;
	else
		mode = -99;
	return mode;
}

/* set the operational interface of an open port
	Inputs:
		<comH> handler of the open port
		<mode> interface mode , 1:software, 2:hardware, otherwise:none
	Returns:
		0 on success, otherwise failure
*/
int
mxsp_set_interface( unsigned int fd, int mode)
{
	HANDLE comH = (HANDLE) fd;
	BYTE iface = (BYTE) mode;
	DWORD bytesReturned;
#if defined(V21XX)
	HANDLE modeH;
	int port = _mxsp_get_port(fd);
	int mask_array[] = { 0x8, 0x2, 0x4, 0x4 };  /* {RS232, RS485-2W, RS485-4W, RS485-4W} */
#elif defined(V46X)
	HANDLE modeH;
	int port = _mxsp_get_port(fd);
#endif

	if (mode < MSP_RS232_MODE || mode > MSP_RS485_4WIRE_MODE)
		return -99;

#if defined(V21XX)
    if (port == -1)
        return -1;

    modeH = CreateFile(L"\\\\.\\MODDev", 
                        GENERIC_WRITE | GENERIC_READ, 
                        FILE_SHARE_WRITE | FILE_SHARE_READ, 
                        NULL, 
                        OPEN_EXISTING, 
                        0, 
                        NULL);
    if (modeH == INVALID_HANDLE_VALUE)
        return -1;

    moxa_port_handle[port] = 0;
#elif defined(V46X)
    if (port == -1)
        return -1;

    modeH = CreateFile(L"\\\\.\\ModeDev", 
                        GENERIC_WRITE | GENERIC_READ, 
                        FILE_SHARE_WRITE | FILE_SHARE_READ, 
                        NULL, 
                        OPEN_EXISTING, 
                        0, 
                        NULL);
    if (modeH == INVALID_HANDLE_VALUE)
        return -1;

#endif

	if (DeviceIoControl( comH, MOXA_SET_OP_MODE, &iface, sizeof(iface), NULL,0,&bytesReturned,NULL) == 0)
		return -1;

	return 0;
}

/* get the operational interface of an open port
	Inputs:
		<comH> handler of the open port
	Returns:
		negative value on failure, otherwise the operational interface mode
*/
int
mxsp_get_interface( unsigned int fd)
{
	HANDLE comH = (HANDLE) fd;
	BYTE mode;
    DWORD bytesReturned;

	return (DeviceIoControl( comH, MOXA_GET_OP_MODE, NULL, 
		sizeof(mode), &mode,0,&bytesReturned,NULL))? (int) mode:-1;
}

/* purge the buffers of an open port
	Inputs:
		<purge> 1: receive data buffer, 2: transmit data buffer, 3: both
	Returns:
		0 on success, otherwise failure
*/
int 
mxsp_purge_buffer( unsigned int fd, int purge)
{
	HANDLE comH = (HANDLE) fd;
	BYTE mode;

	/* no overlapped no MSP_PURGE_RXABORT or MSP_PURGE_TXABORT */
	if ( purge == MSP_PURGE_RX)		mode = PURGE_RXCLEAR;
	else if (purge == MSP_PURGE_TX) mode = PURGE_TXCLEAR;
	else if (purge == MSP_PURGE_RXTX) mode = PURGE_TXCLEAR | PURGE_RXCLEAR;
	else
		return -99;

	return (PurgeComm( comH, mode )==0)? -2:0;
}

/* read data onto a buffer from an open port (nonblocking mode)
	Inputs:
		<fd> handler of the open port
		<buffer> point to the buffer
		<size> maximum size to be read
		<dummy> reserved argument
	Returns:
		< 0, on failure
		0, no data ready
		otherwise the number of bytes read
*/
int
mxsp_read( unsigned int fd, char *buffer, int size, void *dummy)
{
	HANDLE comH = (HANDLE) fd;
	COMSTAT	stat;
	DWORD	errcode;

	(void) dummy;

	if (ClearCommError(comH,&errcode,&stat)==0)
		return -1;

	if (stat.cbInQue > 0) 
	{
		int	len = 0;

		if ((unsigned int) size > stat.cbInQue) 
			size = stat.cbInQue;
		if (ReadFile(comH, buffer, size, &len, 0)==0)
			return -2;
		else
			return len;
	}
	return 0;
}

/* read data onto a buffer from an open port (nblocking mode)
	Inputs:
		<comH> handler of the open port
		<buffer> point to the buffer
		<size> maximum size to be read
	Returns:
		< 0, on failure
		otherwise the number of bytes read
*/
int
mxsp_blocking_read( unsigned int fd, char *buffer, int size, void *hndl)
{
	HANDLE comH = (HANDLE) fd;
	int	len = 0;

	(void) hndl;

	if (ReadFile(comH, buffer, size, &len, NULL))
		return len; /* len==0 end of file */
	else
	{
		DWORD	errcode;

		if (ClearCommError(comH,&errcode,NULL)==0)
			return 0;
		else
			return -1;
	}
}

/* write data to the open port
	Inputs:
		<fd> handler of the open port
		<buffer> point to the buffer
		<size> size of the data to be written
		<dummy> reserved argument
	Returns:
		< 0 on failure, otherwise the number of bytes written
*/
int
mxsp_write( unsigned int fd, char *buffer, int size, void *dummy)
{
	HANDLE comH = (HANDLE) fd;
	int	len = 0;

	(void) dummy;

	if (WriteFile( comH, buffer, size, &len, NULL)) 
		return len;
	else
		return -1;
}

/* close a serial port by the port number, not the name 
	Inputs:
		<port> port number
	Returns:
		0 on success, otherwise a negative value
*/
int
mxsp_close( unsigned int fd )
{
	HANDLE comH = (HANDLE) fd;

#if defined(V21XX) || defined(V46X)
    int port = _mxsp_get_port(fd);
    if (port != -1)
        moxa_port_handle[port] = 0;
#endif
	CloseHandle(comH);
	return 0;
}

/* byte to byte timeout  = ReadIntervalTimeout */
/* read operation timeout  = ReadTotalTimeoutConstant + ReadTotalTimeoutMultiplier x (bytes to read) */
/* set the time-out for read operations on the open port
	Inputs:
		<fd> handler of the open port
		<mSec> time-out, in milliseconds
*/
int 
mxsp_set_timeout_read(unsigned int fd, int mSec)
{
	HANDLE comH = (HANDLE) fd;
	COMMTIMEOUTS commtimeouts;

	if (GetCommTimeouts(comH, &commtimeouts) == 0)
		return -1;
	commtimeouts.ReadIntervalTimeout		= 10;
	commtimeouts.ReadTotalTimeoutMultiplier = 1;
	commtimeouts.ReadTotalTimeoutConstant	= mSec;		
	return SetCommTimeouts(comH, &commtimeouts) ? 0 : -1;
}

/* write operation timeout = WriteTotalTimeoutConstant + WriteTotalTimeoutMultiplier x (bytes to write) */
/* set the time-out for write operations on the open port
	Inputs:
		<fd> handler of the open port
		<mSec> time-out, in milliseconds
*/
int 
mxsp_set_timeout_write(unsigned int fd, int mSec)
{
	HANDLE comH = (HANDLE) fd;
	COMMTIMEOUTS commtimeouts;

	if (GetCommTimeouts(comH, &commtimeouts) == 0)
		return -1;
	commtimeouts.WriteTotalTimeoutMultiplier = 1;//MAXDWORD
	commtimeouts.WriteTotalTimeoutConstant	= mSec;
	return SetCommTimeouts(comH, &commtimeouts) ? 0 : -1;
}

/* open a serial port by the port number, not the name 
	Inputs:
		<port> port number
	Returns:
		0 on failure, otherwise the handle of an open port
	Notes:
		the default communication setting: RS232/HW Flow Control/N/8/1/9600
*/
unsigned int
mxsp_open(int port)
{
	HANDLE comH;
	WCHAR	sPort[20];

#ifdef _WIN32_WCE
	if (moxa_1st_uart_port == 99) _mxsp_get_max_port();
 	wsprintf( sPort, L"$device\\COM%d", port+moxa_1st_uart_port-1 );
#else
 	wsprintf( sPort, L"\\\\.\\COM%d", port );
#endif
    comH = CreateFile ( sPort, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_FLAG_WRITE_THROUGH, NULL);
	if (comH == INVALID_HANDLE_VALUE)
	{
		dbgprintf("fail to open serial port %S\n", sPort);
		return 0;
	}
	else
	{
		unsigned int fd = (unsigned int) comH;
		DCB dcb;

#if defined(V21XX) || defined(V46X)
        if (port < MAX_PORT_NUM)
            moxa_port_handle[port] = fd;
#endif

		/* check and setup configuration */
		GetCommState( comH, &dcb );   
		dcb.DCBlength = sizeof(dcb);

		dcb.fOutX			= FALSE;
		dcb.fInX			= FALSE;
		dcb.fOutxCtsFlow	= FALSE;//TRUE;
		dcb.fErrorChar		= FALSE;
		dcb.fNull			= FALSE;
		dcb.fRtsControl		= RTS_CONTROL_ENABLE;
		dcb.Parity			= NOPARITY;
		dcb.fParity			= FALSE;  /* Specifies if parity checking is enabled. */
		dcb.ByteSize		= 8;
		dcb.StopBits		= ONESTOPBIT;
		dcb.BaudRate		= 9600;
		/* setup configuration */
		SetCommState( comH, &dcb);

		mxsp_set_timeout_read(fd, 1);
		mxsp_set_timeout_write(fd, 1);

		mxsp_purge_buffer( fd, MSP_PURGE_RXTX);
		mxsp_set_interface( fd, MSP_RS232_MODE);

		return fd;
	}
}

/* the number of bytes received by the serial. 
	Inputs:
		<comH> the open port
	Returns:
		< 0 on failure, the number of bytes
*/
int	
mxsp_inqueue(unsigned int fd)
{
	HANDLE comH = (HANDLE) fd;
	COMSTAT	stat;
	DWORD	errcode;

	if (ClearCommError( comH,&errcode,&stat) != 0)
		return stat.cbInQue;
	else
		return -1;
}

/* the number of bytes user data remaining to be transmitted
	Inputs:
		<comH> the open port
	Returns:
		< 0 on failure, the number of bytes
*/
int
mxsp_outqueue( unsigned int fd )
{
	HANDLE comH = (HANDLE) fd;
	COMSTAT	stat;
	DWORD	errcode;

	if (ClearCommError( comH,&errcode,&stat) != 0)
		return stat.cbOutQue;
	else
		return -1;
}

#if 0

static void 
uart_set_timeout(HANDLE Port, int mSec)
{
	COMMTIMEOUTS commtimeouts;
	GetCommTimeouts(Port, &commtimeouts);
	commtimeouts.ReadIntervalTimeout		= mSec;
	commtimeouts.ReadTotalTimeoutMultiplier = 1;
	commtimeouts.ReadTotalTimeoutConstant	= mSec;		
	commtimeouts.WriteTotalTimeoutMultiplier = 1;
	commtimeouts.WriteTotalTimeoutConstant	= mSec;
	SetCommTimeouts(Port, &commtimeouts);
}
#endif

/*
Remarks
If (ReadIntervalTimeout==MAXDWORD && ReadTotalTimeoutMultiplier == MAXDWORD && 
	ReadTotalTimeoutConstant>0 && ReadTotalTimeoutConstant < MAXDWORD)
If there are characters in the input buffer, ReadFile returns immediately with the characters in the buffer.
If there are no characters in the input buffer, ReadFile waits until a character arrives and then returns immediately.
If no characters arrive within the time specified by ReadTotalTimeoutConstant, ReadFile times out.
*/

/***************************************************************************************************************
  a sub-routine that is called to set flow control according to NPPI command parameter
	Inputs:
        <comH> the Handle of open port
        <cts> the CTS value
        <rts> the RTS value
        <stx> Indicates whether XON/XOFF flow control is used during transmission
        <srx> Indicates whether XON/XOFF flow control is used during reception
	Returns:
        0 on success
******************************************************************************************************************/
int 
mxsp_set_flow_control_ex( unsigned int fd ,int cts, int rts, int stx, int srx)
{
	HANDLE comH = (HANDLE) fd;
	DCB		dcb;

	/* check and setup configuration */
	dcb.DCBlength = sizeof(dcb);
    if ( GetCommState(comH,&dcb) == 0)
        return -1;

    dcb.fOutxCtsFlow = cts;
    dcb.fRtsControl = rts;
    dcb.fOutX       = stx;
    dcb.fInX        = srx;

    return (SetCommState(comH, &dcb)? 0:-2);
}

/***************************************************************************************************************
  a sub-routine that is called to set LCTRL status
     Inputs:
        <comH> the Handle of open port
        <dwFunc> the desired value
	Returns:
        0 on success
******************************************************************************************************************/
int 
mxsp_set_io_lctrl( unsigned int fd, unsigned int dwFunc)
{
	HANDLE comH = (HANDLE) fd;
    return (EscapeCommFunction(comH,dwFunc)? 0:-1);
}

/***************************************************************************************************************
  a sub-routine that is called to get lstatus status
     Inputs:
        <fd> the Handle of open port
        <lstatus> buffer to restore lstatus
	Returns:
        0 on success
******************************************************************************************************************/
int
mxsp_get_lstatus(unsigned int fd, unsigned int *lstatus)
{
	HANDLE comH = (HANDLE) fd;
	return (GetCommModemStatus( comH, (DWORD*) lstatus)? 0:-1);
}

/* set the xon/xoff characters for software flow control
	Inputs:
		<fd> the open port
		<xon> xon character
		<xoff> xoff character
	Returns:
		0 on success, otherwise failure
*/
int 
mxsp_set_xonxoff( unsigned int fd ,char xon, char xoff)
{
	HANDLE comH = (HANDLE) fd;
	DCB		dcb;

	dcb.DCBlength = sizeof(dcb);

    /* check and setup configuration */
    if (GetCommState(comH, &dcb ) == 0)//restore dcb
        return -1;

    dcb.XonChar   = xon;
    dcb.XoffChar  = xoff;

    return (SetCommState(comH, &dcb)? 0:-2);
}

/***************************************************************************************************************
  a sub-routine that is called to set break : suspends character transmission 
  for a specified communications device and places the transmission line in a 
  break state until the ClearCommBreak function is called.
     Inputs:
        <comH> the Handle of open port
    Return:
        0 on success
******************************************************************************************************************/
int 
mxsp_set_break( unsigned int fd )
{
	HANDLE comH = (HANDLE) fd;
    return (SetCommBreak(comH)? 0:-1);
}

/***************************************************************************************************************
  a sub-routine that is called to clear break
     Inputs:
        <comH> the Handle of open port
    Return:
        0 on success
******************************************************************************************************************/
int 
mxsp_clear_break( unsigned int fd )
{
	HANDLE comH = (HANDLE) fd;
    return (ClearCommBreak(comH)? 0:-1);
}

int
mxsp_get_errors(unsigned int fd, unsigned int *err)
{
	HANDLE comH = (HANDLE) fd;
	return (ClearCommError( comH, (DWORD*) err, NULL)? 0:-1);
}

int
mxsp_set_mask(unsigned int fd, unsigned int mask)
{
	HANDLE comH = (HANDLE) fd;
	return (SetCommMask( comH, (DWORD) mask)? 0:-1);
}

/***** added by Carl on 2009-08-28 *****/
int
mxsp_set_uart_arguments( unsigned int fd, int baudrate, int dbits, int parity, int sbits, int fc)
{
	HANDLE comH = (HANDLE) fd;
	DCB	dcb;

	/* check and setup configuration */
	FillMemory(&dcb, sizeof(dcb), 0);
	if (GetCommState( comH, &dcb ) == 0)
		return -1;
	/* non-indexed buad rates have been handled in the driver level */
	dcb.DCBlength = sizeof(dcb);
	dcb.BaudRate  = baudrate;

	dcb.ByteSize  = dbits;

	if (sbits==1)		dcb.StopBits = ONESTOPBIT;
	else if (sbits==2)	dcb.StopBits = TWOSTOPBITS;
	else if (sbits==3)	dcb.StopBits = ONE5STOPBITS;
	else				
		return -99;

	if (parity == MSP_PARITY_NONE)			dcb.Parity = NOPARITY;
	else if (parity == MSP_PARITY_ODD)		dcb.Parity = ODDPARITY;
	else if (parity == MSP_PARITY_EVEN)		dcb.Parity = EVENPARITY;
	else if (parity == MSP_PARITY_SPACE)	dcb.Parity = SPACEPARITY;
	else if (parity == MSP_PARITY_MARK)		dcb.Parity = MARKPARITY;
	else
		return -99;

	if (fc == MSP_FLOWCTRL_NONE)			
	{
		dcb.fRtsControl	= RTS_CONTROL_DISABLE;
		dcb.fOutxCtsFlow= FALSE;
		dcb.fOutX		= FALSE;
		dcb.fInX		= FALSE;
		dcb.fErrorChar	= FALSE;
		dcb.fNull		= FALSE;
	}
	else if (fc == MSP_FLOWCTRL_SW)	
	{
		dcb.fRtsControl	= RTS_CONTROL_DISABLE;
		dcb.fOutxCtsFlow= FALSE;
		dcb.fOutX		= TRUE;
		dcb.fInX		= TRUE;
		dcb.fTXContinueOnXoff = FALSE;
	}
	else if (fc == MSP_FLOWCTRL_HW) 
	{
		dcb.fRtsControl	= RTS_CONTROL_HANDSHAKE;
		dcb.fOutxCtsFlow= TRUE;
		dcb.fOutX		= FALSE;
		dcb.fInX		= FALSE;
		dcb.fErrorChar	= FALSE;
		dcb.fNull		= FALSE;
	}
	else
		return -99;

	return (SetCommState(comH, &dcb) == 0) ? -2 : 0;
}


/**********************************************/
