/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/

#ifndef _COMMON_H
#define _COMMON_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "connection.h"
#include "buffer.h"
#include "timer.h"

typedef struct _CONNPRT CONNPRT;

typedef void (*handle_close_t) (CONNPRT*);
typedef int  (*handle_packet_t) (unsigned int, char*, int, void*);

typedef struct _EVENTMR
{
	unsigned int event_mask;		
	conn_event_t event_callback;
	unsigned int event_timer_id;		
	unsigned int event;		
} EVENTMR;

#define OPTION_SELECTABLE	1
#define OPTION_CALL_OPEN	(1 << 1)

struct _CONNPRT
{
	unsigned int	fd;
    unsigned short	id; /* index to a pool of connections */
    unsigned int	port:16;
    unsigned int	timer_id:16; /* user defined timer */

    unsigned int	options:16;
    unsigned int	type:8;
    unsigned int	destroy:1;
    unsigned int	selectable:1; /* if fd is functionable in select() */
    unsigned int	xxx:6;

    BUFFER	*sendbuf;	/* sending buffer */
    BUFFER	*recvbuf;	/* receiving buffer */

	/* callback functions for handling file descriptor fd */
    handle_close_t	close;	/* closing file descriptor */
    handle_packet_t	recv_packet; /* handling received data packet */
    handle_packet_t	send_packet; /* sending data packet */

	USERFUN user_functions; /*  user defined functions */

    void	*private_data; /* carry user-defined data */

	void	*param_opts; /* parameters for each type of connections */
	void	*meta_opts; /* meta data for each type of connections */

    /* a server connection remembers a list of client connections */
	struct _CONNPRT *next, *prev;
};

typedef struct _GLOBAL
{
	struct timeval current_tm;
#define current_time current_tm.tv_sec

	/* for select function */
	unsigned int max_fd;
	fd_set  read_fds;
	fd_set  write_fds;

	/* the connection pool */
	CONNPRT  **connections;
	unsigned int num_connections;
	unsigned int max_connections;
	unsigned int num_selectable_connections;

	CONNPRT  **destroy_connections;
	unsigned int max_destroy_connections;
	unsigned int num_destroy_connections;
} GLOBAL;

extern GLOBAL global;

#ifdef __cplusplus
extern "C" {
#endif

CONNPRT* connection_lookup (int port);
CONNPRT* connection_add (int type, unsigned int fd, int port, unsigned int options, 
						 USERFUN *funs, void *param, void *meta,
						 handle_packet_t recv_packet, handle_packet_t send_packet, handle_close_t close);

void  connection_add_next(CONNPRT *xprt, CONNPRT *next);
void  connection_get_user_functions(CONNPRT *conp, USERFUN *funs);
void* connection_get_parameters(CONNPRT *conp);
void* connection_get_metadata(CONNPRT *conp);
unsigned int connection_add_timer(timer_cb_t timer, unsigned int timer_interval, void *arg);

void log_message (const char *fmt, ...);
int split_line (char **temp, int tempcount, char *pkt);
void* my_malloc(void *mem, size_t size);

#ifdef __cplusplus
}
#endif

#endif

