/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    keypad_wce.c

    Routines to operate the keypad buttons on an embedded computer.

    2009-02-15	CF Lin
*/
#include <windows.h>
#include <stdio.h>
#include "mxkeypad.h"

/* 7420 & 7410 Only */
typedef BOOL (WINAPI* fpInterruptInitialize)( DWORD idInt, HANDLE hEvent, LPVOID pvData, DWORD cbData );
typedef VOID (WINAPI* fpInterruptDisable)( DWORD idInt );

#define KEYPAD_1ST_SYSINTR	50

static HMODULE	gHCore=NULL;
static HANDLE ghEvents[MAX_KEYPADS];

/*	open a handle to the keypad device
	Returns:
		on success, a positive value for the handle
		otherwise, a negative value
*/
int
keypad_open(void)
{
	fpInterruptInitialize fpIntrInit=NULL;
    int i = 0;

	if (gHCore==NULL)
	{
		gHCore = LoadLibrary( TEXT("coredll.dll") );
		if (gHCore==NULL)
			return -1;
		fpIntrInit = (fpInterruptInitialize) GetProcAddress(gHCore, TEXT("InterruptInitialize") );
		if (fpIntrInit==NULL)
		{
			FreeLibrary( gHCore );
			gHCore=NULL;
			return -2;
		}
		/* initialize interrupts */
		for (i = 0; i < MAX_KEYPADS; i++)
		{
			ghEvents[i] = CreateEvent( NULL, FALSE, FALSE, NULL );
			if (ghEvents[i]!=INVALID_HANDLE_VALUE)
				fpIntrInit( KEYPAD_1ST_SYSINTR + i, ghEvents[i], NULL, 0 );
			else
			{
				while(i > 0)
				{
					i--;
					CloseHandle(ghEvents[i]);
				}
				FreeLibrary( gHCore );
				gHCore=NULL;
				return -3;
			}
		}
	}
	return 1;
}

/*	close a handle to the keypad device
	Inputs:
		<fd> the opend handle
*/
void
keypad_close(int fd)
{
	fpInterruptDisable fpIntrDown;
	int i;

	(void) fd;

	if (!gHCore)
		return;

	fpIntrDown = (fpInterruptDisable) GetProcAddress(gHCore, TEXT("InterruptDisable") );
	if (fpIntrDown)
	{
		/* diable interrupts */
		for (i = 0; i < MAX_KEYPADS; i++)
			fpIntrDown(KEYPAD_1ST_SYSINTR + i);
	}
	/* close handlers */
	for (i = 0; i < MAX_KEYPADS; i++)
		CloseHandle(ghEvents[i]);
	/* free up */
	FreeLibrary( gHCore );
	gHCore = NULL;
}

static int
keypad_get_key()
{
	static int i = 0;
	int j;

	for (; i < MAX_KEYPADS; i++)
	{
		j = WaitForSingleObject( ghEvents[i], 200 );
		if (j == WAIT_TIMEOUT)
			continue;
		//printf("i=%d,j=%d\n", i, j);
		return i;
	}
	i %= MAX_KEYPADS;
	return -1;
}

/*	obtain which key has been pressed
	Inputs:
		<fd> the opend handle
	Returns:
		key number starting from 1, or 0 meaning no key been pressed
*/
int
keypad_get_pressed_key(int fd)
{
	int key = -1;
#if 0
	DWORD i;
#endif

	(void) fd;

    while(1) 
    {
#if 1
		if ((key = keypad_get_key()) >= 0) break;
#else
		i = WaitForMultipleObjects( MAX_KEYPADS, ghEvents, FALSE, 1000 );
		printf("i=%d\n", i);
		if (i==WAIT_TIMEOUT)
			continue;
		if (i==WAIT_FAILED)
			break;
		if (i>=WAIT_OBJECT_0 && i < (WAIT_OBJECT_0+MAX_KEYPADS))
		{
			key = i - WAIT_OBJECT_0;
			break;
		}
#endif
	}
    return key;
}

