/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    timer.c

    Routines to have timers.

    2008-04-15	CF Lin
		new release
*/
#include <stdlib.h>
#include <stdio.h>
#include "timer.h"

static TIMER *Pending_Timers = NULL;
static unsigned int TimerRef = 1;

#if defined (_WIN32_WCE) || defined (WIN32)

#if defined(_MSC_VER) || defined(_MSC_EXTENSIONS)
  #define EPOCHFILETIME     11644473600000000Ui64
#else
  #define EPOCHFILETIME     11644473600000000ULL
#endif

struct timezone {
	int tz_minuteswest; /* minutes W of Greenwich */
	int tz_dsttime;     /* type of dst correction */
};

#if !defined(_WIN32_WCE)

int
gettimeofday(struct timeval *tv, struct timezone *tz)
{
	FILETIME                ft;
	LARGE_INTEGER           li;
	unsigned __int64        tmpres = 0;
	static int              tzflag;


	if (NULL != tv)
	{
		GetSystemTimeAsFileTime(&ft);

		li.LowPart  = ft.dwLowDateTime;
		li.HighPart = ft.dwHighDateTime;
		tmpres = li.QuadPart;           /* In 100-nanosecond intervals */

		/*converting file time to unix epoch*/  
		tmpres -= EPOCHFILETIME;        /* Offset to the Epoch time */
		tmpres /= 10;                   /* In microseconds */
		tv->tv_sec = (long)(tmpres / 1000000UL);
		tv->tv_usec = (long)(tmpres % 1000000UL);
	}

	if (NULL != tz)
	{
		if (!tzflag)
		{
			_tzset();
			tzflag++;
		}
		tz->tz_minuteswest = _timezone / 60;
		tz->tz_dsttime = _daylight;
	}

	return 0;
}

#else

int
gettimeofday(struct timeval *tv, struct timezone *tz)
{
	SYSTEMTIME              st;
	FILETIME                ft;
	TIME_ZONE_INFORMATION   tzi;
	LARGE_INTEGER           li;
	unsigned __int64        tmpres = 0;
	static int              tzflag;


	if (NULL != tv)
	{
		GetSystemTime(&st);
		SystemTimeToFileTime(&st, &ft);

		li.LowPart  = ft.dwLowDateTime;
		li.HighPart = ft.dwHighDateTime;
		tmpres = li.QuadPart;           /* In 100-nanosecond intervals */

		/*converting file time to unix epoch*/  
		tmpres -= EPOCHFILETIME;        /* Offset to the Epoch time */
		tmpres /= 10;                   /* In microseconds */
		tv->tv_sec = (long)(tmpres / 1000000UL);
		tv->tv_usec = (long)(tmpres % 1000000UL);
	}

	if (NULL != tz)
	{
		GetTimeZoneInformation(&tzi);

		tz->tz_minuteswest = tzi.Bias;
		if (tzi.StandardDate.wMonth != 0)
			tz->tz_minuteswest += tzi.StandardBias * 60;

		if (tzi.DaylightDate.wMonth != 0)
			tz->tz_dsttime = 1;
		else
			tz->tz_dsttime = 0;
	}

	return 0;
}

#endif  /* _WIN32_WCE */
#endif  /* _WIN32_WCE || WIN32 */

void
timer_print(char *s, struct timeval *tm)
{
printf("%s: %lu %lu\n", s, tm->tv_sec, tm->tv_usec);
}

/* add a time interval, in milliseconds */
#define timer_add_msec(tm, interval) \
{ \
    tm->tv_usec += 1000*interval; \
    while (tm->tv_usec>=1000000) \
    { \
       tm->tv_sec ++; \
       tm->tv_usec -= 1000000; \
    } \
}

static int
timer_compare(struct timeval *t1, struct timeval *t2)
{
    if (t1->tv_sec < t2->tv_sec)
	return -1;
    if (t1->tv_sec > t2->tv_sec)
        return 1;
    if (t1->tv_usec < t2->tv_usec)
        return -1;
    if (t1->tv_usec > t2->tv_usec)
        return 1;
    return 0;
}

static void
schedule_timer (TIMER * ntimer)
{
    TIMER **slot;

    if (!ntimer->events)
	{
		free(ntimer);
		return;
	}

    /* we've created it, now put it in order */
    for (slot = &Pending_Timers; *slot; slot = &(*slot)->next)
    {
        if (timer_compare(&ntimer->next_time, &(*slot)->next_time) < 0)
	    break;
    }
    ntimer->next = *slot;
    *slot = ntimer;
}

/* returns the refnum */
unsigned int
add_timer (unsigned int interval, int events, timer_cb_t func, void *arg)
{
    TIMER  *newtm;
    struct timeval *p;

    if (!events)
	return 0;
    newtm = (TIMER*) calloc (1, sizeof (TIMER));
    if (!newtm)
    {
	return 0;
    }
    p = &newtm->next_time;
    timer_now(p);
    timer_add_msec(p, interval);
    newtm->interval = interval;
    newtm->func = func;
    newtm->arg = arg;
    newtm->events = events;
    newtm->refnum = TimerRef++;
    schedule_timer (newtm);

    return newtm->refnum;
}

void
reset_timers ()
{
    TIMER *slot, *ntimer;
    struct timeval *p;

    slot = Pending_Timers;
    Pending_Timers = NULL;

    /* reset the timer */
    while (slot)
    {
        ntimer = slot->next;

        p = &slot->next_time;
        timer_now(p);
        timer_add_msec(p, slot->interval);
        schedule_timer (slot);

        slot = ntimer;
    }
}

void
exec_timers (struct timeval *now)
{
    TIMER  *current;

    while (Pending_Timers && timer_compare(&Pending_Timers->next_time, now) <= 0)
    {
	/* who is the header item */
	current = Pending_Timers;
	switch (current->events)
	{
	case 0:
		Pending_Timers = current->next;
	    free (current);
	    break;
	default:
	    current->events--;
	case -1:
	    {
			struct timeval *p = &current->next_time;
			/* reschedule */
            timer_add_msec(p, current->interval);
	    }
		if (current->func) (*current->func) (current->arg); 
		Pending_Timers = current->next;
	    schedule_timer (current);
	    break;
	}
    }
}

/* returns the time offset at which the next pending event is scheduled */
int 
next_timer (struct timeval *now, struct timeval *to)
{
    timer_now(now);
    if (Pending_Timers)
    {
        struct timeval *p = &Pending_Timers->next_time;

        if (timer_compare(p, now) <= 0)
	{
	    to->tv_sec = to->tv_usec = 0;
	    return 0;		/* now! */
	}
	to->tv_sec = p->tv_sec-now->tv_sec;
	if (p->tv_usec < now->tv_usec)
	{
	   to->tv_sec--;
	   to->tv_usec = 1000000+p->tv_usec-now->tv_usec;
	}
	else
	   to->tv_usec = p->tv_usec-now->tv_usec;
	   
        return 1;
    }
    return -1;
}

void
free_timers (void)
{
    TIMER  *ptr;

    while (Pending_Timers)
    {
	ptr = Pending_Timers;
	Pending_Timers = Pending_Timers->next;
	free (ptr);
    }
}

/* reschedule a timer whose callback function is not the caller
 */
void
timer_reschedule (unsigned int refnum, int interval)
{
    TIMER **ptr, *tmp;

    for (ptr = &Pending_Timers; *ptr; ptr=&(*ptr)->next)
    if (refnum == (*ptr)->refnum)
    {
 		struct timeval *p;

		tmp = *ptr;
		/* reschedule */
		p = &tmp->next_time;
		tmp->interval = interval;
		timer_now(p); /* change the next_time to be current time */
		if (tmp != Pending_Timers) /* the caller is not the timer */
		{
			*ptr = (*ptr)->next; /* off hook */
			timer_add_msec(p, interval);
			schedule_timer (tmp);
		}
		break;
    }
}

void
timer_set_interval (unsigned int refnum, int interval)
{
    TIMER  *t = Pending_Timers;

    for (; t; t = t->next)
	if (t->refnum == refnum)
	{
	    t->interval = interval;
	    break;
	}
}

/* directly call the timer function */
void
timer_exec_now(unsigned int refnum)
{
    TIMER  *t = Pending_Timers;

    for (; t; t = t->next)
    if (t->refnum == refnum)
    {
	t->func (t->arg);
        break;
    }
}

#if 0
/* remove a timer from the global list */
void
timer_remove(unsigned int refnum)
{
    TIMER **ptr, *tmp;

    for (ptr = &Pending_Timers; *ptr; ptr=&(*ptr)->next)
    if (refnum == (*ptr)->refnum)
    {
    tmp = *ptr;
    *ptr = (*ptr)->next;
    free (tmp);
    break;
    }
}
#else
/* remove a timer from the global list, schedule_timer will free it */
void
timer_remove(unsigned int refnum)
{
    TIMER *tmp;

    for (tmp = Pending_Timers; tmp; tmp=tmp->next)
    if (refnum == tmp->refnum)
    {
		tmp->events = 0;
		break;
    }
}
#endif



