/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    tcp_server.c

    Routines to simulate a simple TCP server.

    2008-04-15	CF Lin
		new release
*/
#include <time.h>
#include <ctype.h>
#include "common.h"

static void
udp_connection_close(CONNPRT *con)
{
	UDPXPRM *param;
	struct sockaddr_in *saddr;

	param = (UDPXPRM*) connection_get_parameters(con);
	if (param) free(param);
	saddr = (struct sockaddr_in *) connection_get_metadata(con);
	if (saddr) free(saddr);
	closesocket(con->fd);
}

/* via an established UDP socket, send data to a networking peer 
	Inputs:
		<fd> the socket
		<data> point to the data
		<dlen> the length of the data
		<xp> a structure specifying the peer, i.e, peer information is stored in the structure
	Returns:
		the number of bytes sent
*/
static int
udp_connection_send(unsigned int fd, char *data, int dlen, void *arg)
{
	CONNPRT *conp = (CONNPRT *) arg;
	struct sockaddr_in *saddr;

	saddr = (struct sockaddr_in *) connection_get_metadata(conp);
	if (saddr)
		return sendto((int) fd, data, dlen, 0, (struct sockaddr*) saddr, sizeof(struct sockaddr));
	else
		return -1;
}

/*  via an established UDP connection, receive data from a peer 
	Inputs:
		<fd> the socket
		<data> point to the buffer to store data
		<dlen> the length of the data
	Outputs:
		<xp> obtain peer information via this structure
	Returns:
		the number of bytes received
*/
static int
udp_connection_recv(unsigned int fd, char *data, int dlen, void *arg)
{
	CONNPRT *conp = (CONNPRT *) arg;
	int len, slen = sizeof(struct sockaddr);
	struct sockaddr_in *saddr;
#ifdef _WIN32_WCE
	struct sockaddr_in tmp;	// used to avoid cleaning saddr on WinCE platform by Carl 2009-04-10.
#endif

	saddr = (struct sockaddr_in *) connection_get_metadata(conp);
	if(saddr)
	{
#ifdef _WIN32_WCE
		memcpy(&tmp, saddr, sizeof(struct sockaddr_in));
		//dbgprintf("addr=%s", inet_ntoa(tmp.sin_addr));
#endif
		len = recvfrom((int) fd, data, dlen, 0, (struct sockaddr*) saddr, &slen);
#ifdef _WIN32_WCE
		//dbgprintf("len=%d, err=%d", len, WSAGetLastError());
		if(len < 0)
		{
			/* saddr->sin_addr is reset to 0.0.0.0 after error */
			//dbgprintf("len=%d, addr=%s", len, inet_ntoa(saddr->sin_addr));
			memcpy(saddr, &tmp, sizeof(struct sockaddr_in));
		}
#endif
	}
	else
		len = -1;

	return (len < 0)? 0:len;
}

static CONNPRT*
udp_connection_add(int type, UDPXPRM *param, USERFUN *funs)
{
	int fd;
	CONNPRT *xp=NULL;
	struct sockaddr_in addr,*saddr;

dbgprintf("++udp_connection_add: (port=%d)", param->listen_port);
	saddr = &addr;
	if (type==CONNECTION_TYPE_UDPCLIENT)
		fd = udp_make_client(param->host, param->listen_port, saddr);
	else
		fd = udp_startup_server(0, param->listen_port, saddr);
	if (fd <= 0)
		return NULL;

	param->ip = saddr->sin_addr.s_addr;
	param = (UDPXPRM*) my_malloc(param, sizeof(UDPXPRM));
	if (!param)
		return NULL;
	saddr = (struct sockaddr_in *) my_malloc(saddr, sizeof(struct sockaddr_in));
	if (!saddr)
	{
		free(param);
		return NULL;
	}
	/* add this client into the pool */
	xp = connection_add(type, (unsigned int) fd, param->listen_port, OPTION_SELECTABLE | OPTION_CALL_OPEN, 
			funs, param, saddr, udp_connection_recv, udp_connection_send, udp_connection_close);
	if (xp)
	{
dbgprintf("--udp_connection_add:");
	}
	else
	{
		closesocket(fd);
		free(param);
		free(saddr);
	}
	return xp;
}

/* start a udp server at a port without specifying the interface
*/
MHANDLE
udp_connection_startup_server (UDPXPRM *param, USERFUN *funs)
{
	return udp_connection_add(CONNECTION_TYPE_UDPSERVER, param, funs);
}

/* make a client connection to a server and add this connection 
   into the golbal connection pool
	Inputs:
		<host> the host name of the server
		<port> the listen port of the server
*/
MHANDLE
udp_connection_make_client (UDPXPRM *param, USERFUN *funs)
{
	return udp_connection_add(CONNECTION_TYPE_UDPCLIENT, param, funs);
}

