/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    connection.c

    Routines of connection functions: such as server start up, client connection...

    2008-04-15	CF Lin
		new release
*/
#if !defined (_WIN32_WCE) && !defined (WIN32)
#include <errno.h>
#endif
#include "common.h"
#include "moxa_config.h"
#include "mxthread.h"

int	SigCaught = 1;
GLOBAL global;

/* add a socket to the write set */
void
connection_set_write (CONNPRT *xp)
{
	if (xp->selectable)	
	{
		FD_SET (xp->fd, &global.write_fds);
		if (global.max_fd < xp->fd)	global.max_fd = xp->fd;
	}
}

/* remove a socket from the read set */
void
connection_clear_write (CONNPRT *xp)
{
	if (xp->selectable)	FD_CLR (xp->fd, &global.write_fds);
}

/* add a socket to the read set */
void
connection_set_read (CONNPRT *xp)
{
	if (xp->selectable)	
	{
		FD_SET (xp->fd, &global.read_fds);
		if (global.max_fd < xp->fd)	global.max_fd = xp->fd;
	}
}

/* remove a socket from the read set */
void
connection_clear_read (CONNPRT *xp)
{
	if (xp->selectable)	FD_CLR (xp->fd, &global.read_fds);
}

/*	lookup a connection by port #
	Inputs:
		<port> port #
	Returns:
		the connection if there is 
*/
CONNPRT*
connection_lookup (int port)
{
	CONNPRT *xp;
	unsigned int i;

	for (i=0; i < global.num_connections; i++)
	{
		xp = global.connections[i];
		if (xp->port == (unsigned int) port)
			return xp;
	}
	return NULL;
}

/* walk through each connection with a callback function */
void
connection_foreach (unsigned int type, conn_close_t cb, void *arg)
{
	CONNPRT *xp;
	unsigned int i;

	for (i=0; i < global.num_connections; i++)
	{
		xp = global.connections[i];
		if (xp->type == type || type==0)
			cb(xp, arg);
	}
}

/* mark a destroyed connection */
void
connection_destroy(MHANDLE hndl)
{
    CONNPRT *xp = (CONNPRT*) hndl;

	if (xp->destroy)
		return;
	if (global.num_destroy_connections==global.max_destroy_connections)
	{
		global.max_destroy_connections += 8;
		global.destroy_connections = realloc(global.destroy_connections, global.max_destroy_connections*sizeof(CONNPRT*));
		if (global.destroy_connections==NULL)
			return;
	}
    if (xp->destroy==0)	xp->destroy = 1;
	global.destroy_connections[global.num_destroy_connections++] = xp;
dbgprintf("--connection_destroy:");
}

/*	remove the timer defined in a connection 
	Inputs:
		<xp> connection
*/
void
connection_remove_timer(CONNPRT *xp)
{
	if (xp->timer_id) 
	{
		timer_remove(xp->timer_id);
		xp->timer_id = 0;
	}
}

/*  for a connection, add a timer function
	Inputs:
		<xp> the connection
		<timer> the timer function

	Returns:
		0 on success, otherwise failure
	Notes:
		the timer interval has given when the connection
		is established if there is
*/
unsigned int
connection_add_timer(timer_cb_t timer, unsigned int timer_interval, void *arg)
{
	/* add a new timer */
	return add_timer(timer_interval, -1, timer, arg); /* register a timer */
}

int
connection_get_client_number(CONNPRT *conp)
{
    int n = 0;

    while(conp->next) 
    {
        conp = conp->next;
		n++;
    }
    return n;
}

void
connection_add_next(CONNPRT *xprt, CONNPRT *next)
{
	next->prev = xprt;
	next->next = xprt->next;
	xprt->next = next;
	if (next->next)
		next->next->prev = next;
}

void
connection_get_user_functions(CONNPRT *conp, USERFUN *funs)
{
	memcpy(funs, &conp->user_functions, sizeof(USERFUN));
}

void*
connection_get_parameters(CONNPRT *conp)
{
	return conp->param_opts;
}

void*
connection_get_metadata(CONNPRT *conp)
{
	return conp->meta_opts;
}

/*	create a connection and put it into the global pool 
	Inputs:
		<fd> the device id of the connection
		<port> the port number of the device
		<type> the type of the connection
		<selectable> is the connection selectable
		<funs> user-defined functions
	Returns:
		the connection
*/
CONNPRT*
connection_add (int type, unsigned int fd, int port, unsigned int options,
				USERFUN *funs, void *param, void *meta,
				handle_packet_t recv_packet, handle_packet_t send_packet, handle_close_t close) 
{
	CONNPRT *xp = NULL;
	
dbgprintf("++connection_add:");

	xp = (CONNPRT*) calloc(1, sizeof(CONNPRT));
	if (xp)
	{
		xp->recv_packet = recv_packet;
		xp->send_packet = send_packet;
		xp->close = close;
		xp->fd = fd;
	    xp->type = type;
		xp->port = (unsigned int) port;
		xp->meta_opts = meta;
		xp->param_opts = param;
		if (fd)
		{
			/* add this client into the pool */
			if (global.num_connections == global.max_connections)
			{
				global.max_connections += 16;
				global.connections = realloc(global.connections, global.max_connections*sizeof(CONNPRT*));
				if (global.connections==NULL)
				{
					free(xp);
					return NULL;
				}
			}			
			/* put this connection into the pool */
			global.connections[global.num_connections] = xp;
			xp->id = global.num_connections++;

			if (options & OPTION_SELECTABLE)
			{
				/* count the # of selectable connections */
				xp->selectable = 1;
				global.num_selectable_connections++;
				/* put it into readable list */
				connection_set_read(xp); 
			}
			if (funs)
			{
				memcpy(&xp->user_functions, funs, sizeof(USERFUN));
				if((options & OPTION_CALL_OPEN) && xp->user_functions.open)
				{
					unsigned int timer_interval = 0;
					/* call a user-defined initialization function */
					xp->private_data = xp->user_functions.open(xp, param, &timer_interval);
					/* call a user-defined timer function */
					if(timer_interval && xp->user_functions.timer)
						xp->timer_id = connection_add_timer(xp->user_functions.timer, timer_interval, xp->private_data);
				}
			}
		}
dbgprintf("--connection_add: type (%d) port (%d)", type, port);
	}
	else
	{
		dbgprintf("connection_add: out of memory");
	}
	return xp;
}

/*	remove an open port from the global set 
	Inputs:
		<xp> connection
*/
static void
connection_remove (CONNPRT *xp)
{
	unsigned int id = xp->id;

dbgprintf("++connection_remove: remove an id [%d] port [%d].", id, xp->port);

	/* call user-defined function to release resource */
	if (xp->user_functions.close) 
		xp->user_functions.close(xp, xp->private_data);

	/* remove the timer if there is */
	connection_remove_timer(xp);

	/* unhook itself */
	if (xp->prev)
		xp->prev->next = xp->next;
	if (xp->next)
		xp->next->prev = xp->prev;

	if (xp->fd > 0)
	{
		if (xp->close) xp->close(xp);

		connection_clear_read(xp);
		connection_clear_write(xp);

		if (xp->selectable)
			global.num_selectable_connections--;

		if (global.num_connections > 0) 
			global.num_connections--; /* the last one */

		if (id < global.num_connections) /* if it is not the last one */
		{
			/* swap this place with the last in the array */
			global.connections[id] = global.connections[global.num_connections];
			global.connections[id]->id = id;
		}
	}

	if (xp->recvbuf) buffer_free(xp->recvbuf);
	if (xp->sendbuf) buffer_free(xp->sendbuf);

dbgprintf("--connection_remove: remove an id [%d] port [%d].", id, xp->port);
	free(xp);
}

/*	flush queued data to the specified connection 
	Inputs:
		<xp> connection
*/
static void
connection_flush_data (CONNPRT *xp)
{
    int  n, len;
	BUFFER *b;

    /* write until the queue is consumed, or we would block */
    while ((b = xp->sendbuf) != NULL)
    {
		b = xp->sendbuf;
		len = b->datasize - b->consumed; /* data left */
		if (len == 0)
		{ /* empty */
			b->consumed = b->datasize = 0;
			break;
		}
		n = xp->send_packet (xp->fd, b->data+b->consumed, len, xp);
		if (n == -1)
		{
			connection_destroy (xp);
			return;
		}
		else if (n==0)
			break;
dbgprintf("FLUSH[%4d,%4d]", xp->port, n); 
		xp->sendbuf = buffer_consume(xp->sendbuf, n);
	}
}

/*	send data to the specified connection 
	Inputs:
		<xp> connection
		<buf> data
		<len> the length of the data
*/
int
connection_send_data (MHANDLE hndl, char *buf, int len)
{
	CONNPRT *xp = (CONNPRT*) hndl;
	BUFFER *b = xp->sendbuf;
	int     n = 0;

//dbgprintf("SEND[%4d,%4d]", xp->port, len); 
	if (!b || b->datasize == b->consumed) /* empty, write data out directly */
	{
		n = xp->send_packet (xp->fd, buf, len, xp);
		if (n < 0)
		{
#if !defined (_WIN32_WCE) && !defined (WIN32)
			dbgprintf ("connection_send_data : send_packet %d", errno);
#else
			dbgprintf ("connection_send_data : send_packet %d", WSAGetLastError());
#endif
			connection_destroy (xp);
			return 0;
		}
		else
		{
			if (n == len) /* all sent */
			return len;
			/* portion of data */
			buf += n;
			len -= n;
		}
	}
	if (len > 0)
	{
		/* wait for the socket becomes writable again */
		connection_set_write(xp);
		if (buffer_size(xp->sendbuf) >= 102400)
		{
			return 0;
		}
		else
		{
			/* there was data or unsent data, queue it */
			xp->sendbuf = buffer_queue(b, buf, len);
			if (!xp->sendbuf) /* overflow */
			{
				dbgprintf ("connection_send_data : buffer overflow");
				connection_destroy (xp);
				return 0;
			}
		}
	}
	return n;
}

/*	read coming data from a connection and call proper callback function
	Inputs:
		<xp> the connection
*/
static void
connection_read_data (CONNPRT *xp)
{
	int len;
	BUFFER	*b;
	DATAPKT dpkt;

	if (xp->recvbuf==NULL)
	{
		xp->recvbuf = buffer_new();
		if (xp->recvbuf==NULL)
			return;
	}

	b = xp->recvbuf;
	if (b->datamax == b->datasize)
		return;

	while(b->datamax > b->datasize)
	{
		/* read from the open socket */
		len = xp->recv_packet (xp->fd, b->data+b->datasize, b->datamax - b->datasize, xp);
		if (len == 0)
			break;
//dbgprintf("RECV[%4d,%2d/%4d]", xp->port,len,b->datamax-b->datasize);
		if (len < 0)
		{
			connection_destroy (xp);
			return;
		}
		b->datasize += len;
	}
	while (b->datasize > b->consumed)
	{
		dpkt.packet_data = b->data+b->consumed;
		dpkt.packet_size = b->datasize-b->consumed;
		dpkt.packet_consumed = dpkt.packet_size;
		/* call the user-defined function which consumes bytes */
		len = xp->user_functions.dispatch(xp, xp->private_data, &dpkt);
		if (len < 0)
		{
			connection_destroy(xp);
			return;
		}
		b->consumed += dpkt.packet_consumed; /* important here */
		if (len == 0) /* intentional out */
			break;
	}
	len = b->datasize-b->consumed; /* data left in the buffer */
	if (len)
	{
		/* move ahead */
		memmove (b->data, b->data+b->consumed, len); 
	}
	b->datasize=len;
	b->consumed=0;
}

#if !defined(WIN32) && !defined(_WIN32_WCE)
#include <signal.h>
static void
sighandler (int sig)
{
	dbgprintf ("sig: caught signal %d", sig);
    switch (sig)
	{
	case SIGINT:
	case SIGQUIT:
	case SIGTERM:
	case SIGPIPE:
		SigCaught = 1; /* quit the main loop */
		break;
    case SIGHUP:
		break;
	case SIGUSR1:
		break;
	}
}

static void
sig_init(void)
{
    struct sigaction sa;

    memset (&sa, 0, sizeof (sa));
    sa.sa_handler = sighandler;
    sigaction (SIGHUP, &sa, NULL);
    sigaction (SIGTERM, &sa, NULL);
    sigaction (SIGINT, &sa, NULL);
    sigaction (SIGPIPE, &sa, NULL);
    sigaction (SIGUSR1, &sa, NULL);
    sigaction (SIGALRM, &sa, NULL);
}
#endif

static void
connection_dispatch(void *arg)
{
	CONNPRT *xp;
	struct  timeval to;
	fd_set  read_fds, write_fds;
	time_t	tick;
	int     numfds, select_errors=0;
	unsigned int i;
	app_release_t app_release = (app_release_t) arg;
#if defined(_WIN32_WCE) || defined(WIN32)
	HANDLE	dummy = NULL;
#endif

 	dbgprintf("Program Start %d %d", SigCaught, global.num_connections);
	while(!SigCaught && global.num_connections)
	{
		tick = next_timer (&global.current_tm, &to);
		if (tick < 0)
		{
			to.tv_sec = 2;
			to.tv_usec = 0;
		}
		if (global.num_selectable_connections == 0)
		{
#if defined(_WIN32_WCE) || defined(WIN32)
			/* we need a timeout, without any selectable fd, select function 
			    doesn't work as a timeout in Win32 
			*/
			if (global.num_connections > 0)
			{
				if (dummy==NULL) {
					//printf("create event\n");
					dummy = CreateEvent(NULL, FALSE, FALSE, NULL);
				}
				WaitForSingleObject(dummy, to.tv_usec>>10);
			}
			else
#endif
			{
 				dbgprintf("no connection");
				break;
			}
		}
		else
		{
			/* copy the global sets */
			read_fds  = global.read_fds;
			write_fds = global.write_fds;
			numfds = select (global.max_fd + 1, &read_fds, &write_fds, 0, &to);
			if (numfds == -1) 
			{
				if (++select_errors < 5)
					continue;
				dbgprintf("ERROR: select errors (could be no connection)");
				break;
			}
			select_errors = 0; /* reset */
		}
		/* handle data read for each file descriptor */
		for (i = 0; i < global.num_connections; i++) 
		{
			xp = global.connections[i];
			if (xp->selectable && FD_ISSET(xp->fd, &read_fds) == 0)
				continue;
			connection_read_data (xp);
		}
		/* handle data write for each file descriptor */
		for (i = 0; i < global.num_connections; i++) 
		{
			xp = global.connections[i];
			if (xp->selectable && FD_ISSET(xp->fd, &write_fds) == 0)
				continue;
			connection_flush_data (xp);
		}
		/* execute any pending events now */
		exec_timers (&global.current_tm);

		/*	reap destoryed connections, must be last operations because 
			previous actions including timer functions might put connections 
			in the destroy list
		*/
		i = 0;
		while(i < global.num_destroy_connections)
		{
			connection_remove (global.destroy_connections[i++]);
		}
		global.num_destroy_connections = 0;
	}

	dbgprintf("Program Exit");

	if (app_release) app_release();

	free_timers();

	/* the routine releasing all possible resource */
	while (global.num_connections)
	{
		connection_remove (global.connections[0]);
	}
	if (global.connections)	free(global.connections);
	if (global.destroy_connections)	free(global.destroy_connections);

#if defined(_WIN32_WCE) || defined(WIN32)
	if (dummy) CloseHandle(dummy);
#endif

	/* release possible windows socket dll */
	net_library_release();
}

int
connection_dispatch_loop(int argc, char **argv, app_init_t app_init, app_release_t app_release, int thread)
{
	CONNPRT *xp;
	unsigned int i;
	char buffer[128];

	get_program_build_time(CONNECTION_VERSION_NUMBER, buffer);
	dbgprintf("%s", buffer);
	if (!SigCaught)
	{
		dbgprintf("this function has been called");
		return 1;
	}

#if defined(WIN32) || defined(_WIN32_WCE)
	net_library_init();
#else
	sig_init();
#endif

	/* clean up global variables */
	memset(&global, 0, sizeof(GLOBAL));
	if (moxa_device()==0)
	{
 		dbgprintf("device error");
		return -1;
	}
	else if (app_init==NULL)
	{
		dbgprintf("initial function is not given");
		return -1;
	}
	/* call the user-defined initialization function */
	else if (app_init(argc, argv)==0)
	{
		unsigned int n = 0;

		/*	while supporting nonselectable ports, add a constant 
			timer to force periodical read on these ports */
		for (i = 0; i < global.num_connections; i++) 
		{
			xp = global.connections[i];
			if (xp->fd==0 || xp->selectable)
				continue;
			n++;
		}
		if (n)	add_timer(10, -1, NULL, NULL);

		SigCaught = 0;
	}
	else
		return -2;
	if (thread)
		mxthread_create( (thread_cb_t) connection_dispatch, app_release  );
	else
		connection_dispatch(app_release);
	return 0;
}

void
connection_dispatch_quit(void)
{
	SigCaught = 1;
}
