/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    mxwdg.c

    Routines to operate a watch dog timer.

    2008-08-20
*/

#include	<windows.h>
#if defined(UC74XX) || defined(DA66X)
#include	"fsioctl.h"	// For IOCTL_HAL_REBOOT definition.
#endif
#include	"mxwdg.h"

#if defined(UC74XX) || defined(DA66X)
#define	IOCTL_HAL_REBOOT CTL_CODE(FILE_DEVICE_HAL, 15, METHOD_BUFFERED, FILE_ANY_ACCESS)
#else /* IA26X, UC-712X */
#ifdef CTL_CODE
#undef CTL_CODE
#endif

#define CTL_CODE(DeviceType, Function, Method, Access) ( \
  ((DeviceType) << 16) | ((Access) << 14) | ((Function) << 2) | (Method) \
)

#define IOCTL_START_WDG			CTL_CODE( 0, 164, 0, 0)
#define IOCTL_STOP_WDG			CTL_CODE( 0, 165, 0, 0)
#define IOCTL_REFRESH_WDG		CTL_CODE( 0, 166, 0, 0)
#endif

#if defined(UC74XX) || defined(DA66X)
typedef BOOL (WINAPI* fpKernelIoControl)( DWORD dwIoControlCode, LPVOID lpInBuf, DWORD nInBufSize, LPVOID lpOutBuf, DWORD nOutBufSize, LPDWORD lpBytesReturned );
#else /* IA26X, UC-712X */
typedef struct _Wdg {
	HANDLE hWdg;
	DWORD dwPeriod;
} Wdg;
#endif

/*
	global variable 

*/
#if defined(UC74XX) || defined(DA66X)
static HMODULE hCoreDLL = NULL;
static fpKernelIoControl KernelIoControl = NULL;
#endif

int
mxwdg_open(unsigned long time)
{
#if defined(UC74XX) || defined(DA66X)
	DWORD dwWatchdogTime = time; /* in milliseconds */

	if (KernelIoControl==NULL)
	{
		// Load KernelIoControl library
		hCoreDLL = LoadLibrary( TEXT("coredll.dll") );
		if (hCoreDLL==NULL)
			return -1;
		KernelIoControl = (fpKernelIoControl)GetProcAddress(hCoreDLL, TEXT("KernelIoControl") );
		if (KernelIoControl==NULL)
		{
			FreeLibrary( hCoreDLL );
			return -2;
		}
		KernelIoControl( IOCTL_HAL_REBOOT, &dwWatchdogTime, sizeof(DWORD), NULL, 0, NULL );
	}
	return (int) dwWatchdogTime;
#else /* IA26X, UC-712X */
	Wdg *wdg;
	BOOL bolRet;

	if ((wdg = (Wdg *)malloc(sizeof(Wdg))) == NULL) return -1;
	wdg->hWdg = CreateFile( L"UCS1:", GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
	wdg->dwPeriod = time;

	bolRet = DeviceIoControl (wdg->hWdg, IOCTL_START_WDG, &wdg->dwPeriod, sizeof(DWORD), NULL, 0,  NULL, NULL);
	if ( bolRet )
		return (int)wdg;
	else
	{
		CloseHandle(wdg->hWdg);
		free(wdg);
		return -2;
	}
#endif
}

int 
mxwdg_refresh(int fd)
{
#if defined(UC74XX) || defined(DA66X)
	DWORD dwWatchdogTime = fd; /* in milliseconds */

	if (KernelIoControl==NULL)
		return -1;
	else 
	{
		KernelIoControl( IOCTL_HAL_REBOOT, &dwWatchdogTime, sizeof(DWORD), NULL, 0, NULL );
		return 0;
	}
#else /* IA26X, UC-712X */
	Wdg *wdg = (Wdg *)fd;
	BOOL bolRet;

	if ( wdg == NULL || wdg->hWdg == INVALID_HANDLE_VALUE || wdg->hWdg == NULL )
		return -1;
#ifdef UC712X
	bolRet = DeviceIoControl (wdg->hWdg, IOCTL_REFRESH_WDG, NULL, 0, NULL, 0,  NULL, NULL);
#else
	bolRet = DeviceIoControl (wdg->hWdg, IOCTL_REFRESH_WDG, &wdg->dwPeriod, sizeof(DWORD), NULL, 0,  NULL, NULL);
#endif
	if (bolRet == TRUE) return 0;
	return -2;
#endif
}

void
mxwdg_close(int fd)
{
#if defined(UC74XX) || defined(DA66X)
	DWORD dwWatchdogTime = 0; /* in milliseconds */
	
	(void) fd;

	if (KernelIoControl)
	{
		KernelIoControl( IOCTL_HAL_REBOOT, &dwWatchdogTime, sizeof(DWORD), NULL, 0, NULL );
		FreeLibrary( hCoreDLL );
	}
#else /* IA26X, UC-712X */
	Wdg *wdg = (Wdg *)fd;

	if ( wdg == NULL || wdg->hWdg == INVALID_HANDLE_VALUE || wdg->hWdg == NULL )
		return;
	DeviceIoControl (wdg->hWdg, IOCTL_STOP_WDG, NULL, 0, NULL, 0,  NULL, NULL);
	CloseHandle(wdg->hWdg);
	free(wdg);
#endif
}
