/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    mcast.c

    Routines of multicast functions

    2008-04-15 Peter Wu	new release
*/

#ifndef UC711X
#include <string.h>
#if !defined(_WIN32_WCE) && !defined(WIN32)
#include <unistd.h>
#endif
#include "mcast.h"

#if !defined(_WIN32_WCE) && !defined(WIN32)
#define closesocket(x)	close((int)x)
#define SOCKET_ERROR	-1
#endif

/* ----------------------- sender ----------------------- */

/*  create multicast socket
    Inputs: none
    Outputs: none
    Returns:
        -1 on failure, otherwise socket fd
*/
SOCKET
mcast_open_socket()
{
	return socket(AF_INET, SOCK_DGRAM, 0);
}

/*  set TTL
    Inputs:
        <fd> the socket fd
        <val> the TTL value
    Outputs: none
    Returns:
        -1 on failure, otherwise OK
*/
int
mcast_set_ttl(SOCKET fd, int val)
{
#if defined (_WIN32_WCE) || defined (WIN32)
	int ttl = val;
#else
	unsigned char ttl = (unsigned char)val;
#endif

	return (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL,
#if defined (_WIN32_WCE) || defined (WIN32)
												(const char *)
#else
												(void *)
#endif
												&ttl, sizeof ttl));
}

/*  set multicast packets sent to loopback interface on or off
    Inputs:
        <fd> the socket fd
        <onoff> the flag for on of value 1 or off of value 0
    Outputs: none
    Returns:
        -1 on failure, otherwise OK
*/
int
mcast_set_loop(SOCKET fd, int onoff)
{
#if defined (_WIN32_WCE) || defined (WIN32)
	int flag;
#else
	unsigned char flag;
#endif

#if 0
	if (onoff != 1 && onoff != 0) return -2;
#endif

	flag = onoff;

	return (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_LOOP,
#if defined (_WIN32_WCE) || defined (WIN32)
												(const char *)
#else
												(void *)
#endif
												&flag, sizeof flag));
}

/*  create a multicast client socket
    Inputs:
		<onoff> the flag to turn on (1) or off (0) multicast packets sent to
				the loopback interface
        <grp> the multicast group
        <port> the multicast port number
        <addr> point to the multicast socket structure
    Outputs: none
    Returns:
        socket fd, otherwise negative on failure
*/
SOCKET
mcast_make_client(int onoff, char *grp, int port, struct sockaddr_in *addr)
{
	SOCKET fd;

	/* Create a datagram socket */
	if ((fd = mcast_open_socket()) == -1) {
		return -1;
	}
#if defined(_WIN32_WCE)
	/* Associate the source socket's address with the socket fd */
	memset((char *)addr, 0, sizeof(struct sockaddr_in));
	addr->sin_family = AF_INET;
	addr->sin_addr.s_addr = htonl(INADDR_ANY);
	addr->sin_port = htons(0);
	if (bind(fd, (struct sockaddr *)addr, sizeof(struct sockaddr_in)) == -1) {
		return -4;
	}
#endif
	/* Fill out the desination socket's address information */
	memset(addr, 0, sizeof(struct sockaddr_in));
	addr->sin_family = AF_INET;
	addr->sin_addr.s_addr = inet_addr(grp);
	addr->sin_port = htons((unsigned short)port);

	/* Set the Time-to-Live of the multicast */
	if (mcast_set_ttl(fd, 3) == -1) {
		closesocket(fd);
		return -2;
	}
#if !defined(_WIN32_WCE)
	if (mcast_set_loop(fd, onoff) == -1) {
		closesocket(fd);
		return -3;
	}
#endif

	return fd;
}

/* ----------------------- receiver ----------------------- */

/*  create a multicast server socket
    Inputs:
        <grp> the multicast group
        <port> the multicast port number
        <addr> point to the multicast socket structure
    Outputs: none
    Returns:
        socket fd, otherwise negative on failure
*/
SOCKET
mcast_make_server(char *grp, int port, struct sockaddr_in *addr)
{
	SOCKET fd;
	int on = 1;

	/* create a datagram socket */
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
		return -1;
	}

	/* enable to allow multiple multicast servers to be started */
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *) &on, sizeof(int)) == -1) {
	}

	/* construct a multicast address structure */
	memset(addr, 0, sizeof(struct sockaddr_in));
	addr->sin_family = AF_INET;
#if defined(WIN32) || defined(_WIN32_WCE)
	addr->sin_addr.s_addr = htonl(INADDR_ANY);
#else
	/* avoid receiving other datagrams for the listening port */
	addr->sin_addr.s_addr = inet_addr(grp);
#endif
	addr->sin_port = htons((unsigned short)port);

	/* construct a multicast address structure */
	if (bind(fd, (struct sockaddr *)addr, sizeof(struct sockaddr_in)) == -1) {
		return -2;
	}

	return fd;
}

/*  join a multicast group
    Inputs:
        <fd> the socket fd
        <grp> the multicast group
        <mreq> point to the multicast request structure
    Outputs: none
    Returns:
        0 on success, -1 on failure
*/
int
mcast_join(SOCKET fd, char *grp, struct ip_mreq *mreq)
{
	mreq->imr_multiaddr.s_addr = inet_addr(grp);
	mreq->imr_interface.s_addr = htonl(INADDR_ANY);

	if (setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
#if defined(WIN32) || defined(_WIN32_WCE)
								(const char *)
#else
								(void *)
#endif
								mreq, sizeof(struct ip_mreq)) == SOCKET_ERROR) {
		return -1;
	}

	return 0;
}

/*  leave a multicast group
    Inputs:
        <fd> the socket fd
        <mreq> point to the multicast request structure
    Outputs: none
    Returns:
        0 on success, -1 on failure
*/
int
mcast_leave(SOCKET fd, struct ip_mreq *mreq)
{
	if (setsockopt(fd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
#if defined(WIN32) || defined(_WIN32_WCE)
								(const char *)
#else
								(void *)
#endif
								mreq, sizeof(struct ip_mreq)) == SOCKET_ERROR) {
		return -1;
	}

	return 0;
}
#endif /* UC711X */
