/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    tcp_connection.c

    Routines to simulate a simple TCP server.

    2008-04-15	CF Lin
		new release
*/
#include <time.h>
#include <ctype.h>
#include "common.h"

MHANDLE tcp_connection_make_client (CLNTPRM *param, USERFUN *funs);

typedef struct _TCPCLNT
{
    CLNTPRM param;
	USERFUN dfuns;
	unsigned int timer_id;
} TCPCLNT;

static void
tcp_connection_client_timer(void *data)
{
    TCPCLNT *clnt = (TCPCLNT*) data;

    tcp_connection_make_client(&clnt->param, &clnt->dfuns);
	free (clnt);
}

static void
tcp_connection_close(CONNPRT *con)
{
	CLNTPRM *param;
	TCPCLNT *clnt;

    closesocket(con->fd);
	param = (CLNTPRM*) connection_get_parameters(con);
	if (param)
	{
		if (con->type == CONNECTION_TYPE_TCPCLIENT && param->connection_retrials > 0 && param->reconnect_interval)
		{
			param->connection_retrials--;
			clnt = (TCPCLNT*) calloc(1, sizeof(TCPCLNT));
			if (clnt)
			{
				param->host = NULL; /* IP is enough */
				clnt->param = *param;
				connection_get_user_functions(con, &clnt->dfuns);
				clnt->timer_id = add_timer(param->reconnect_interval*1000, 1, tcp_connection_client_timer, clnt);
			}
		}
		free(param);
	}
}

/*  add a TCP connection into the golbal pool
	Inputs:
		<fd> the socket
		<port> port #
		<type> specify the type of connection
		<funs> user-defined functions
	Returns:
		always 0 in order to start processing data packet
*/
static CONNPRT*
tcp_connection_add(unsigned int fd, int port, int type, int call_open, USERFUN *funs, void *param, void *meta)
{
	unsigned int options;

	if (call_open)
		options = OPTION_SELECTABLE | OPTION_CALL_OPEN;
	else
		options = OPTION_SELECTABLE;

    /* add this client into the pool */
    return connection_add(type, fd, port, options, funs, param, meta, 
				tcp_nonblocking_read, tcp_nonblocking_write, tcp_connection_close);
}

/* verify if the ip is a valid one 
	Inputs:
		<pi> allowed range
		<ip> to be validated
	Returns:
		1, if a valid ip
*/
static int
valid_ip(unsigned int pi, unsigned int ip)
{
	unsigned int i, cpi, cip;

	for (i=0; i < sizeof(unsigned int); i++)
	{
	cpi = pi & (0xff << (8*i));
	cip = ip & (0xff << (8*i));
	if (cpi && cip != cpi)
		return 0;
	}
	return 1;
}

int connection_get_client_number(CONNPRT*); 

/*  after accepting a client, add a connection into the golbal pool
	Inputs:
		<sock> the server socket
		<buf> dummy buffer
		<len> dummy;
		<conp> the server connection
	Returns:
		always 0 in order to start processing data packet
*/
static int
tcp_connection_accept_client(unsigned int sock, char *buf, int len, void *data)
{
	CONNPRT *srvr = (CONNPRT*) data;
	int port;
	int fd;
	unsigned int ip;
	CONNPRT *clnt;
	SRVRPRM *param;
	TCPCPRM *cparm;
	USERFUN funs;

	(void) buf;
	(void) len;

dbgprintf("++tcp_connection_accept_client:");

	/* accept a socket */
	fd = tcp_accept_client (sock, &port, &ip, 1);
	if (fd < 0)
		return 0;

	param = connection_get_parameters(srvr);
	if (param)
	{
		unsigned int n, clients_ip_range=0;

		if (param->ip_range[0])
		{
			clients_ip_range = inet_addr(param->ip_range);
		}
		/* check the # of client connections of the server */
		/* not include current connected client */
		n = connection_get_client_number(srvr);
		/* check if the # of client connections exceeds a level */
		if ( param->max_clients && n >= param->max_clients ) 
		{
			closesocket(fd);
			return 0;
		}
		/* check if the IP of the client is allowed */
		else if ( clients_ip_range && !valid_ip(clients_ip_range, ip) )
		{
			closesocket(fd);
			return 0;
		}
	}
	cparm = (TCPCPRM*) malloc(sizeof(TCPCPRM));
	if (!cparm)
	{
		closesocket(fd);
		return 0;
	}
	cparm->ip = BSWAP32(ip);
	cparm->local_port = port;
	cparm->listen_port = param->listen_port;
	connection_get_user_functions(srvr, &funs);
	/* add this connection into the golbal connection pool */
	clnt = tcp_connection_add((unsigned int)fd, port, CONNECTION_TYPE_TCPACCEPT, 1, &funs, cparm, NULL);
	if (clnt==NULL)
	{
		free(cparm);
		closesocket(fd);
	}
	else
	{
		/* add a child */
		connection_add_next(srvr, clnt);
dbgprintf("--tcp_connection_accept_client: accept a tcp client port %d ip %s", port, iptoa(BSWAP32(ip)));
	}
	return 0; /* always */
}

/*  start a tcp server without specifying the interface
	Inputs:
		<param> the parameters of the host
		<funs> user-defined functions
	Returns:
		a handle to the connection
*/
MHANDLE
tcp_connection_startup_server (SRVRPRM *param, USERFUN *funs)
{
	CONNPRT *conp;
	SRVRPRM *srvp;
	int fd;
	int port = param->listen_port;

dbgprintf("++tcp_connection_startup_server:");

	srvp = (SRVRPRM*) my_malloc(param, sizeof(SRVRPRM));
	if (!srvp)
		return NULL;

	fd = tcp_startup_server(0, port);
	if (fd < 0)
	{
		free(srvp);
		return NULL;
	}
	conp = tcp_connection_add((unsigned int)fd, port, CONNECTION_TYPE_TCPSERVER, 0, funs, srvp, NULL);
	if (conp==NULL)
	{
		free(srvp);
		closesocket(fd);
	}
	else
	{
		/* accepting a connection request is similar to receiving */
		conp->recv_packet = tcp_connection_accept_client;
dbgprintf("--tcp_connection_startup_server:");
	}
	return conp;
}

/*  make a client connection to a server and add this connection 
    into the golbal connection pool
	Inputs:
		<param> the parameters of the host
		<funs> user-defined functions
	Returns:
		a handle to the connection
*/
MHANDLE
tcp_connection_make_client (CLNTPRM *param, USERFUN *funs)
{
    CONNPRT *clnt=NULL;
    unsigned int ip, trials;
    int fd = 0;
	int port;

dbgprintf("++tcp_connection_make_client: timeout (%d) retries (%d)",
		param->connection_timeout, param->connection_retrials);

	/* check at least some amount of values */
	if (param->connection_timeout < 10) param->connection_timeout = 10;
    trials = param->connection_retrials;
    if (param->host==NULL)
        ip = param->server_ip;

    /* try to make a connection */
    do
    {
        port = param->listen_port;
        /* make the connection request */
        fd = tcp_make_client(param->host, &port, &ip, 1);
        if (fd > 0)
        {
            /* wait for the connection for writing */
            if (tcp_nonblocking_wait(fd, param->connection_timeout, 1))
                break;
            closesocket(fd);
            fd = 0;
        }
    } while(trials--);

#if 0 /* fd often OK becuase nonblocking is used for connect() */
dbgprintf("==tcp_connection_make_client: fd (%d)", fd);
#endif
    if (fd > 0)
    {
        param->server_ip = ip;
        param->local_port = port;
		param = (CLNTPRM*) my_malloc(param, sizeof(CLNTPRM));
		if (!param)
		{
            closesocket(fd);
			return NULL;
		}
        /* add this connection into the global connection pool */
        clnt = tcp_connection_add(fd, port, CONNECTION_TYPE_TCPCLIENT, 1, funs, param, NULL);
        if (clnt==NULL)
		{
			free(param);
            closesocket(fd);
		}
    }
dbgprintf("--tcp_connection_make_client: fd=%d", fd);
    return clnt;
}




